/*   Copyright 2016 Rohm Semiconductor

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
#ifndef i2c_common_pp_h
#define i2c_common_pp_h

#include "rohm_hal2.h"       //types, DEBUG_print*, USE_*_HARDWARE_I2C

/**
* RegisterWriter class for writing sensor registers via I2C object
*/
class RegisterWriter
{
public:
    /**
    * Use pre-instantiated I2C instance for HAL.
    *
    * @param i2c_obj pre-instantiated i2c object.
    */
    RegisterWriter(I2C &i2c_obj);

    /**
    * Create a i2c instance which is connected to specified I2C pins.
    *
    * @param sda SDA pin
    * @param sdl SCL pin
    */
    RegisterWriter(PinName sda = I2C_SDA, PinName scl = I2C_SCL);

    ~RegisterWriter();

    /**
    * General read @buf_len value(s) to @*buf from sensor @reg in address @sad.
    * @param sad Slave address of sensor
    * @param reg Register of sensor
    * @param *buf uint8_t[@buf_len] for read data
    * @param buf_len amount of data to read from @reg
    */
    uint8_t read_register(uint8_t sad, uint8_t reg, uint8_t* buf, uint8_t buf_len);

    /**
    * FIFO Read @buf_len value(s) to @*buf from sensor @reg in address @sad.
    * Difference is the usage of stop-bit between commands.
    * @param sad Slave address of sensor
    * @param reg Register of sensor
    * @param *buf uint8_t[@buf_len] for read data
    * @param buf_len amount of data to read from @reg
    */
    uint8_t read_fifo_register(uint8_t sad, uint8_t reg, uint8_t* buf, uint8_t buf_len);

    /**
    * Read @buf_len value(s) in high speed to @*buf from sensor @reg in address @sad.
    * Reference to kx123 specification page 24, hs 3.4mhZ mode.
    * @param sad Slave address of sensor
    * @param reg Register of sensor
    * @param *buf uint8_t[@buf_len] for read data
    * @param buf_len amount of data to read from @reg
    */
    uint8_t hs_read_register(uint8_t sad, uint8_t reg, uint8_t* buf, uint8_t buf_len);
    
    /**
    * Write @data_len value(s) from @*data to sensor @reg in address @sad.
    * @param sad Slave address of sensor
    * @param reg Register of sensor
    * @param *data uint8_t[@data_len] for written data
    * @param data_len amount of data to written to @reg
    */
    bool write_register(uint8_t sad, uint8_t reg, uint8_t* data, uint8_t data_len);

    /**
    * Write 1 value from @data to sensor @reg in address @sad.
    * @param sad Slave address of sensor
    * @param reg Register of sensor
    * @param data to be written
    */
    bool write_register(uint8_t sad, uint8_t reg, uint8_t data);

    /**
    * Write @data_len value(s) in high speed from @*data to sensor @reg in address @sad.
    * @param sad Slave address of sensor
    * @param reg Register of sensor
    * @param *data uint8_t[@data_len] for written data
    * @param data_len amount of data to written to @reg
    */
    bool hs_write_register(uint8_t sad, uint8_t reg, uint8_t* data, uint8_t data_len);

    /**
    * Read-change-write register (@sad/@reg)
    * @param sad Slave address of sensor
    * @param reg Register of sensor
    * @param mask bits to clear before applying new @bits
    * @param bits value to write
    * @return true on error, false on ok
    */
    bool change_bits(uint8_t sad, uint8_t reg, uint8_t mask, uint8_t bits);

private:
    I2C i2c_bus;
    bool self_created_i2c;

    void set_hs_mode_for_one_command();

};




#endif


