/* LWIPBP3595Interface.cpp */
/* Copyright (C) 2016 Grape Systems, Inc. */
/* The base file is LWIPInterface.cpp. */

/* LWIPInterface.cpp */
/* LWIP implementation of NetworkInterfaceAPI
 * Copyright (c) 2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "LWIPBP3595Interface.h"
#include "LWIPBP3595Interface_BssType.h"
#include "WlanBP3595.h"

#include "lwip/inet.h"
#include "lwip/netif.h"
#include "lwip/dhcp.h"
#include "lwip/tcpip.h"
#include "lwip/sockets.h"
#include "lwip/netdb.h"
#include "netif/etharp.h"
#include "wifi_arch.h"
#include "lwip/netif.h"
#include "lwip/udp.h"
#include "lwip/tcp.h"
#include "lwip/tcp_impl.h"
#include "lwip/timers.h"
#include "lwip/dns.h"
#include "lwip/def.h"
#include "lwip/ip_addr.h"

#define LWIP_TIMEOUT 180000

/* TCP/IP and Network Interface Initialisation */
static struct netif netif;

static char ip_addr[NSAPI_IP_SIZE] = "\0";
static char mac_addr[NSAPI_MAC_SIZE] = "\0";

static Semaphore tcpip_inited(0);
static Semaphore netif_linked(0);
static Semaphore netif_up(0);

static void tcpip_init_irq(void *)
{
    tcpip_inited.release();
}

static void netif_link_irq(struct netif *netif)
{
    if (netif_is_link_up(netif)) {
        netif_linked.release();
    }
}

static void netif_status_irq(struct netif *netif)
{
    if (netif_is_up(netif)) {
        strcpy(ip_addr, inet_ntoa(netif->ip_addr));
        netif_up.release();
    }
}

static void init_netif(ip_addr_t *ipaddr, ip_addr_t *netmask, ip_addr_t *gw)
{
    tcpip_init(tcpip_init_irq, NULL);
    tcpip_inited.wait();

    memset((void*) &netif, 0, sizeof(netif));
    netif_add(&netif, ipaddr, netmask, gw, NULL, wifi_arch_enetif_init, tcpip_input);
    netif_set_default(&netif);

    netif_set_link_callback  (&netif, netif_link_irq);
    netif_set_status_callback(&netif, netif_status_irq);
}

static void set_mac_address(void)
{
    snprintf(mac_addr, 19, "%02x:%02x:%02x:%02x:%02x:%02x", netif.hwaddr[0], netif.hwaddr[1],
             netif.hwaddr[2], netif.hwaddr[3], netif.hwaddr[4], netif.hwaddr[5]);
}

static void _wlan_inf_callback(uint8_t ucType, uint16_t usWid, uint16_t usSize, uint8_t *pucData) {
    if ((ucType == 'I') && (usWid == 0x0005)) {
        if (pucData[0] == 0x01) {     // CONNECTED
            /* Notify the EthernetInterface driver that WLAN was connected */
            WlanBP3595_Connected();
        } else {
            /* Notify the EthernetInterface driver that WLAN was disconnected */
            WlanBP3595_Disconnected();
        }
    }
}

static int _wlan_init() {
    uint32_t status;

    /* Initialize WlanBP3595 */
    if (WlanBP3595_Init(&_wlan_inf_callback) != 0) {
        return -1;
    }

    /* Wait until WLAN_BP3595_START  timeout 60s */
    while (1) {
        Thread::wait(200);
        status = WlanBP3595_GetWlanSts();
        if (status == WLAN_BP3595_START) {
            break;
        }
    }

    return 0;
}

static int _wlan_setting(const char *ssid, const char *pass, nsapi_security_t security)
{
    int     ret;
    grp_u8  ucWidData8;     // 8bit wid data
    grp_wld_byte_array  tBAWidData;     // byte array wid data

    // Set BSS type
    ucWidData8 = BSS_TYPE;
    ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_BSS_TYPE, &ucWidData8);
    if (ret != 0) {
        return -1;
    }

    // Set SSID
    tBAWidData.pucData = (grp_u8 *)ssid;
    tBAWidData.ulSize  = strlen((char *)tBAWidData.pucData);
    ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_SSID, &tBAWidData);
    if (ret != 0) {
        return -1;
    }

    if ((security == NSAPI_SECURITY_WPA) || (security == NSAPI_SECURITY_WPA2)) {
        // Set PSK
        tBAWidData.pucData = (grp_u8 *)pass;
        tBAWidData.ulSize  = strlen((char *)tBAWidData.pucData);
        ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_11I_PSK, &tBAWidData);
        if (ret != 0) {
            return -1;
        }
    }

    // Set 11i mode
    switch (security) {
        case NSAPI_SECURITY_WEP:
            ret = strlen(pass);
            if (ret == 5) {
                ucWidData8 = 0x03;  // WEP64
            } else if (ret == 13) {
                ucWidData8 = 0x07;  // WEP128
            } else {
                return -1;
            }
            break;
        case NSAPI_SECURITY_WPA:
        case NSAPI_SECURITY_WPA2:
            ucWidData8 = 0x79;  // WPA/WPA2 Mixed
            break;
        case NSAPI_SECURITY_NONE:
        default:
            ucWidData8 = 0x00;
            break;
    }
    ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_11I_MODE, &ucWidData8);
    if (ret != 0) {
        return -1;
    }

    if (security == NSAPI_SECURITY_WEP) {
        // Set WEP KEY
        tBAWidData.pucData = (grp_u8 *)pass;
        tBAWidData.ulSize  = strlen((char *)tBAWidData.pucData);
        ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_WEP_KEY, &tBAWidData);
        if (ret != 0) {
            return -1;
        }
    }

    return 0;
}

// LWIPBP3595Interface implementation
int LWIPBP3595Interface::connect(
    const char *ssid, 
    const char *pass, 
    nsapi_security_t security)
{
    _wlan_init();

    // Set up network
    init_netif(0, 0, 0);
    set_mac_address();

    _wlan_setting(ssid, pass, security);

    // Connect to network
    wifi_arch_enable_interrupts();

    dhcp_start(&netif);

    // Wait for an IP Address
    // -1: error, 0: timeout
    if (netif_up.wait(LWIP_TIMEOUT) < 0) {
        return NSAPI_ERROR_DHCP_FAILURE;
    }

    return 0;
}

int LWIPBP3595Interface::disconnect()
{
    dhcp_release(&netif);
    dhcp_stop(&netif);

    wifi_arch_disable_interrupts();

    return 0;
}

const char *LWIPBP3595Interface::get_ip_address()
{
    return ip_addr;
}

const char *LWIPBP3595Interface::get_mac_address()
{
    return mac_addr;
}
