/*****************************************************************************
  KX224_I2C.cpp

 Copyright (c) 2017 ROHM Co.,Ltd.

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
*
*  KX224-1053 3 axis accelerometer library
*
*  @modified by Ren Boting
*  @version 1.0
*  @date    18-February-2019
*
*  Library for "KX224-1053 3 axis accelerometer library"
*    https://www.rohm.co.jp/sensor-shield-support/accelerometer
*
*/
#include "KX224.h"

KX224::KX224(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr), _g_sens(4096)
{
}

KX224::~KX224()
{
}

uint8_t KX224::initialize(void)
{
  char reg ;
  uint16_t gsel;

  read(KX224_WHO_AM_I, &reg, sizeof(reg));
  DEBUG_PRINT("\r\nKX224_WHO_AMI Register Value = 0x%x\r\n", reg);
  
  if (reg != KX224_WAI_VAL) {
	  DEBUG_PRINT("\nCan't find KX224\r\n");
	  return 1;
  }

  reg = KX224_CNTL1_VAL;
  write(KX224_CNTL1, &reg);

  reg = KX224_ODCNTL_VAL;
  write(KX224_ODCNTL, &reg);

  read(KX224_CNTL1, &reg, sizeof(reg));
  gsel = reg & KX224_CNTL1_GSELMASK;

  reg |= KX224_CNTL1_PC1;
  write(KX224_CNTL1, &reg);
  
  switch(gsel) {
    case KX224_CNTL1_GSEL_8G :
      // (Equivalent counts) / (Range) = (32768 / 8)
      _g_sens = 4096;
    break;

    case KX224_CNTL1_GSEL_16G :
      // (Equivalent counts) / (Range) = (32768 / 16)
      _g_sens = 2048;
    break;

    case KX224_CNTL1_GSEL_32G :
      // (Equivalent counts) / (Range) = (32768 / 32)
      _g_sens = 1024;
    break;

    default:
    	DEBUG_PRINT("\r!!! rgsel value (0x%x) is invalid\r", gsel);
    	return 2;
  }
  return 0;
}

void KX224::get_val(float *data)
{
  char val[6];
  uint16_t acc[3];
  // get raw value
  read(KX224_XOUT_L, &val[0], 6);
  acc[0] = ((int16_t)val[1] << 8) | (int16_t)(val[0]);
  acc[1] = ((int16_t)val[3] << 8) | (int16_t)(val[2]);
  acc[2] = ((int16_t)val[5] << 8) | (int16_t)(val[4]);

  // Convert LSB to g
  data[0] = (float)acc[0] / _g_sens;
  data[1] = (float)acc[1] / _g_sens;
  data[2] = (float)acc[2] / _g_sens;
}

void KX224::write(uint8_t memory_address, char *data)
{
  char tmp[3];
  tmp[0]=memory_address;
  tmp[1]=*data;
  m_i2c.write(m_addr, &tmp[0], 2);
}

void KX224::read(uint8_t memory_address, char *data, int size)
{
  char t[1] = {memory_address};
  m_i2c.write(m_addr, t, 1, true);
  m_i2c.read(m_addr, data, size);
}
