/* Copyright (c) 2015 ARM Ltd., MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*
*
*  KX022 Accelerometer library
*
*  @author  Toyomasa Watarai
*  @version 1.0
*  @date    30-December-2015
*
*  Library for "KX022 Accelerometer library" from Kionix a Rohm group
*    http://www.kionix.com/product/KX022-1020
*
*/

#include "mbed.h"
#include "KX022.h"

KX022::KX022(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr)
{
    initialize();
}

KX022::KX022(I2C &i2c_obj, int addr) : m_i2c(i2c_obj), m_addr(addr)
{
    initialize();
}

KX022::~KX022()
{
}

void KX022::initialize()
{
    unsigned char buf;
    unsigned char reg[2];

    DEBUG_PRINT("KX022 init started\n\r");
    readRegs(KX022_WHO_AM_I, &buf, sizeof(buf));
    if (buf != KX022_WAI_VAL) {
        DEBUG_PRINT("KX022 initialization error. (WAI %d, not %d)\n\r", buf, KX022_WAI_VAL);
        DEBUG_PRINT("Trying to config anyway, in case there is some compatible sensor connected.\n\r");
    }

    reg[0] = KX022_CNTL1;
    reg[1] = 0x41;
    writeRegs(reg, 2);

    reg[0] = KX022_ODCNTL;
    reg[1] = 0x02;
    writeRegs(reg, 2);

    reg[0] = KX022_CNTL3;
    reg[1] = 0xD8;
    writeRegs(reg, 2);

    reg[0] = KX022_TILT_TIMER;
    reg[1] = 0x01;
    writeRegs(reg, 2);

    reg[0] = KX022_CNTL1;
    reg[1] = 0xC1;
    writeRegs(reg, 2);
}
    
float KX022::getAccX()
{
    return (float(getAccAxis(KX022_XOUT_L))/16384);
}

float KX022::getAccY()
{
    return (float(getAccAxis(KX022_YOUT_L))/16384);
}

float KX022::getAccZ()
{
    return (float(getAccAxis(KX022_ZOUT_L))/16384);
}

void KX022::getAccAllAxis(float * res)
{
    res[0] = getAccX();
    res[1] = getAccY();
    res[2] = getAccZ();
}

int16_t KX022::getAccAxis(uint8_t addr)
{
    int16_t acc;
    uint8_t res[2];

    readRegs(addr, res, 2);
    acc = ((res[1] << 8) | res[0]);
    return acc;
}

void KX022::readRegs(int addr, uint8_t * data, int len)
{
    int read_nok;
    char t[1] = {addr};
    
    m_i2c.write(m_addr, t, 1, true);
    read_nok = m_i2c.read(m_addr, (char *)data, len);
    if (read_nok){
        DEBUG_PRINT("Read fail\n\r");        
        }
}

void KX022::writeRegs(uint8_t * data, int len)
{
    m_i2c.write(m_addr, (char *)data, len);
}
