/* Copyright (c) 2015 ARM Ltd., MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*
*
*  KX022 Accelerometer library
*
*  @author  Toyomasa Watarai
*  @version 1.0
*  @date    30-December-2015
*
*  Library for "KX022 Accelerometer library" from Kionix a Rohm group
*    http://www.kionix.com/product/KX022-1020
*
*/

#ifndef KX022_H
#define KX022_H

#include "mbed.h"

#define KX022_DEFAULT_SLAVE_ADDRESS (0x1E << 1)
#define KX022_WAI_VAL         (0x14)

#define KX022_XOUT_L          (0x06)
#define KX022_XOUT_H          (0x07)
#define KX022_YOUT_L          (0x08)
#define KX022_YOUT_H          (0x09)
#define KX022_ZOUT_L          (0x0A)
#define KX022_ZOUT_H          (0x0B)
#define KX022_WHO_AM_I        (0x0F)
#define KX022_CNTL1           (0x18)
#define KX022_CNTL3           (0x1A)
#define KX022_ODCNTL          (0x1B)
#define KX022_TILT_TIMER      (0x22)

#define _DEBUG
#ifdef _DEBUG
extern Serial pc;
#define DEBUG_PRINT(...) pc.printf(__VA_ARGS__)
#else
#define DEBUG_PRINT(...)
#endif

/**
* KX022 accelerometer example
*
* @code
* #include "mbed.h"
* #include "KX022.h"
*
* int main(void) {
*
* KX022 acc(I2C_SDA, I2C_SCL);
* PwmOut rled(LED_RED);
* PwmOut gled(LED_GREEN);
* PwmOut bled(LED_BLUE);
*
*     while (true) {
*         rled = 1.0 - abs(acc.getAccX());
*         gled = 1.0 - abs(acc.getAccY());
*         bled = 1.0 - abs(acc.getAccZ());
*         wait(0.1);
*     }
* }
* @endcode
*/
class KX022
{
public:
    /**
    * KX022 constructor
    *
    * @param sda SDA pin
    * @param sdl SCL pin
    * @param addr slave address of the I2C peripheral (default: 0x3C)
    */
    KX022(PinName sda, PinName scl, int addr = KX022_DEFAULT_SLAVE_ADDRESS);

    /**
     * Create a KX022 instance which is connected to specified I2C pins
     * with specified address
     *
     * @param i2c_obj I2C object (instance)
     * @param addr slave address of the I2C-bus peripheral (default: 0x3C)
     */
    KX022(I2C &i2c_obj, int addr = KX022_DEFAULT_SLAVE_ADDRESS);

    /**
    * KX022 destructor
    */
    ~KX022();

    /** Initializa KX022 sensor
     *
     *  Configure sensor setting
     *
     */
    void initialize(void);

    /**
     * Get X axis acceleration
     *
     * @returns X axis acceleration
     */
    float getAccX();

    /**
     * Get Y axis acceleration
     *
     * @returns Y axis acceleration
     */
    float getAccY();

    /**
     * Get Z axis acceleration
     *
     * @returns Z axis acceleration
     */
    float getAccZ();

    /**
     * Get XYZ axis acceleration
     *
     * @param res array where acceleration data will be stored
     */
    void getAccAllAxis(float * res);

private:
    I2C m_i2c;
    int m_addr;
    void readRegs(int addr, uint8_t * data, int len);
    void writeRegs(uint8_t * data, int len);
    int16_t getAccAxis(uint8_t addr);

};

#endif
