/* Copyright (c) 2017 Renesas Electronics., MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef _BM1422AGMV_H_
#define _BM1422AGMV_H_

#include "mbed.h"

#define BM1422AGMV_DEVICE_ADDRESS_0E    (0x0E << 1)
#define BM1422AGMV_DEVICE_ADDRESS_0F    (0x0F << 1)
#define BM1422AGMV_WIA_VAL             (0x41)

#define BM1422AGMV_WIA                 (0x0F)
#define BM1422AGMV_DATAX               (0x10)
#define BM1422AGMV_STA1                (0x18)
#define BM1422AGMV_CNTL1               (0x1B)
#define BM1422AGMV_CNTL2               (0x1C)
#define BM1422AGMV_CNTL3               (0x1D)
#define BM1422AGMV_AVE_A               (0x40)
#define BM1422AGMV_CNTL4               (0x5C)

#define BM1422AGMV_STA1_RD_DRDY        (1 << 6)

#define BM1422AGMV_CNTL1_FS1           (1 << 1)
#define BM1422AGMV_CNTL1_ODR_10Hz      (0 << 3)
#define BM1422AGMV_CNTL1_RST_LV        (1 << 5)
#define BM1422AGMV_CNTL1_OUT_BIT       (1 << 6)
#define BM1422AGMV_CNTL1_PC1           (1 << 7)

#define BM1422AGMV_CNTL2_DRP           (1 << 2)
#define BM1422AGMV_CNTL2_DREN          (1 << 3)

#define BM1422AGMV_CNTL3_FORCE         (1 << 6)

#define BM1422AGMV_AVE_A_AVE4          (0 << 2)

#define BM1422AGMV_CNTL1_VAL           (BM1422AGMV_CNTL1_FS1 | BM1422AGMV_CNTL1_OUT_BIT | BM1422AGMV_CNTL1_PC1)
#define BM1422AGMV_CNTL2_VAL           (BM1422AGMV_CNTL2_DREN)
#define BM1422AGMV_CNTL3_VAL           (BM1422AGMV_CNTL3_FORCE)
#define BM1422AGMV_CNTL4_VAL           (0x0000)
#define BM1422AGMV_AVE_A_VAL           (BM1422AGMV_AVE_A_AVE4)

#define BM1422AGMV_14BIT_SENS          (24)
#define BM1422AGMV_12BIT_SENS          (6)

//#define _DEBUG
#ifdef _DEBUG
#undef DEBUG_PRINT
extern Serial pc;
#define DEBUG_PRINT(...) pc.printf(__VA_ARGS__)
#else
#define DEBUG_PRINT(...)
#endif

class BM1422AGMV
{
public:
    /**
    * BM1422 constructor
    *
    * @param sda SDA pin
    * @param sdl SCL pin
    * @param addr slave address of the I2C peripheral (default: 0x1C)
    */
    BM1422AGMV(PinName sda, PinName scl, int addr = BM1422AGMV_DEVICE_ADDRESS_0E);
    /**
     * Create a BM1422 instance which is connected to specified I2C pins
     * with specified address
     *
     * @param i2c_obj I2C object (instance)
     * @param addr slave address of the I2C-bus peripheral (default: 0x1C)
     */
    BM1422AGMV(I2C &i2c_obj, int addr = BM1422AGMV_DEVICE_ADDRESS_0E);
    /**
    * BM1422 destructor
    */
    ~BM1422AGMV();
    /** Initializa BM1422 sensor
     *
     *  Configure sensor setting
     *
     */
    void initialize(void);
    /**
     * Get XYZ axis acceleration
     *
     * @returns XYZ (filled in float[3]) axis acceleration
     */
    void get_val(float *data);

private:
    I2C m_i2c;
    int m_addr;
    void readRegs(int addr, uint8_t * data, int len);
    void writeRegs(uint8_t * data, int len);
    uint8_t _sens;
    /**
     * Get XYZ axis acceleration
     *
     * @returns XYZ (rawdata) axis acceleration
     */
    void get_rawval(unsigned char *data);
    /**
     * Convert XYZ axis acceleration
     *
     * @returns XYZ (float array [3]) axis acceleration
     */
    void convert_uT(int16_t *rawdata, float *data);
};

#endif
