/* Copyright (c) 2015 ARM Ltd., MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*
*
*  BM1383GLV Pressure sensor library
*
*  @author  Toyomasa Watarai
*  @version 1.0
*  @date    30-December-2015
*
*  Library for "BM1383GLV Pressure sensor library"
*    http://www.rohm.com/web/eu/news-detail?news-title=2014-12-11_ad_pressure&defaultGroupId=false
*
*/

#ifndef BM1383GLV_H
#define BM1383GLV_H

#include "mbed.h"

#define BM1383GLV_DEFAULT_SLAVE_ADDRESS         (0x5D << 1)
#define BM1383GLV_ID_VAL              (0x31)
#define BM1383GLV_ID                  (0x10)
#define BM1383GLV_POWER_DOWN          (0x12)
#define BM1383GLV_SLEEP               (0x13)
#define BM1383GLV_MODE_CONTROL        (0x14)
#define BM1383GLV_TEMPERATURE_MSB     (0x1A)
#define BM1383GLV_PRESSURE_MSB        (0x1C)

#ifdef _DEBUG
extern Serial pc;
#define DEBUG_PRINT(...) pc.printf(__VA_ARGS__)
#else
#define DEBUG_PRINT(...)
#endif

/**
* BM1383GLV pressure sensor example
*
* @code
* BM1383GLV sensor(I2C_SDA, I2C_SCL);
* Serial pc(USBTX, USBRX);
* 
* int main() {
*     pc.printf("\nBM1383GLV Pressure sensor library test program.\n");
*
*     while(1) {
*         pc.printf("pressure=%7.2f, temperature=%5.3f\n", sensor.getPressure(), sensor.getTemperature());
*         wait(0.5);
*     }
* }
* @endcode
*/
class BM1383GLV
{
public:
    /**
    * BM1383GLV constructor
    *
    * @param sda SDA pin
    * @param sdl SCL pin
    * @param addr slave address of the I2C peripheral (default: 0xBA)
    */
    BM1383GLV(PinName sda, PinName scl, int addr = BM1383GLV_DEFAULT_SLAVE_ADDRESS);

    /**
     * Create a BM1383GLV instance which is connected to specified I2C pins
     * with specified address
     *
     * @param i2c_obj I2C object (instance)
     * @param addr slave address of the I2C-bus peripheral (default: 0xBA)
     */
    BM1383GLV(I2C &i2c_obj, int addr = BM1383GLV_DEFAULT_SLAVE_ADDRESS);

    /**
    * BM1383GLV destructor
    */
    ~BM1383GLV();

    /** Initializa BM1383GLV sensor
     *
     *  Configure sensor setting
     *
     */
    void initialize(void);

    /**
     * Get pressure
     *
     * @returns pressure
     */
    float getPressure();

    /**
     * Get temerature
     *
     * @returns temperature
     */
    float getTemperature();

private:
    I2C m_i2c;
    int m_addr;
    uint8_t m_buf[3];
    void readRegs(int addr, uint8_t * data, int len);
    void writeRegs(uint8_t * data, int len);

};

#endif
