/* Copyright (c) 2015 ARM Ltd., MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*
*
*  BM1383GLV Pressure sensor library
*
*  @author  Toyomasa Watarai
*  @version 1.0
*  @date    30-December-2015
*
*  Library for "BM1383GLV Pressure sensor library"
*    http://www.rohm.com/web/eu/news-detail?news-title=2014-12-11_ad_pressure&defaultGroupId=false
*
*/

#include "mbed.h"
#include "BM1383GLV.h"

BM1383GLV::BM1383GLV(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr)
{
    initialize();
}

BM1383GLV::BM1383GLV(I2C &i2c_obj, int addr) : m_i2c(i2c_obj), m_addr(addr)
{
    initialize();
}

BM1383GLV::~BM1383GLV()
{
}

void BM1383GLV::initialize()
{
    unsigned char buf;
    unsigned char reg[2];

    readRegs(BM1383GLV_ID, &buf, sizeof(buf));
    if (buf != BM1383GLV_ID_VAL) {
        DEBUG_PRINT("BM1383GLV initialization error. (%d)\n", buf);
        return;
    }

    reg[0] = BM1383GLV_POWER_DOWN;
    reg[1] = 0x01;
    writeRegs(reg, 2);
    
    wait(1);

    reg[0] = BM1383GLV_SLEEP;
    reg[1] = 0x01;
    writeRegs(reg, 2);

    reg[0] = BM1383GLV_MODE_CONTROL;
    reg[1] = 0xC4;
    writeRegs(reg, 2);
}

float BM1383GLV::getPressure()
{
    uint32_t rawPressure;

    readRegs(BM1383GLV_PRESSURE_MSB, m_buf, 3);
    rawPressure = (((uint32_t)m_buf[0] << 16) | ((uint32_t)m_buf[1] << 8) | m_buf[2]&0xFC) >> 2;
    return (float)rawPressure / (2048);
}

float BM1383GLV::getTemperature()
{
    int32_t rawTemerature;
    
    readRegs(BM1383GLV_TEMPERATURE_MSB, m_buf, 2);
    rawTemerature = ((int32_t)m_buf[0] << 8) | (m_buf[1]);
    return (float)rawTemerature / 32;
}

void BM1383GLV::readRegs(int addr, uint8_t * data, int len)
{
    char t[1] = {addr};
    m_i2c.write(m_addr, t, 1, true);
    m_i2c.read(m_addr, (char *)data, len);
}

void BM1383GLV::writeRegs(uint8_t * data, int len)
{
    m_i2c.write(m_addr, (char *)data, len);
}
