/* Copyright (c) 2015 ARM Ltd., MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*
*
*  BM1383AGLV Pressure sensor library
*
*  @author  Toyomasa Watarai
*  @version 1.0
*  @date    30-December-2015
*
*  Library for "BM1383AGLV Pressure sensor library"
*    http://www.rohm.com/web/eu/news-detail?news-title=2014-12-11_ad_pressure&defaultGroupId=false
*
*/

#ifndef BM1383AGLV_H
#define BM1383AGLV_H

#include "mbed.h"

#define BM1383AGLV_DEFAULT_SLAVE_ADDRESS         (0x5D << 1)
#define BM1383AGLV_ID1_VAL             (0xE0)
#define BM1383AGLV_ID2_VAL             (0x32)
#define BM1383AGLV_ID1                 (0x0F)
#define BM1383AGLV_ID2                 (0x10)
#define BM1383AGLV_POWER_DOWN          (0x12)
#define BM1383AGLV_RESET               (0x13)
#define BM1383AGLV_MODE_CONTROL        (0x14)
#define BM1383AGLV_STATUS              (0x19)
#define BM1383AGLV_PRESSURE_MSB        (0x1A)
#define BM1383AGLV_PRESSURE_LSB        (0x1B)
#define BM1383AGLV_PRESSURE_XLSB       (0x1C)
#define BM1383AGLV_TEMPERATURE_MSB     (0x1D)
#define BM1383AGLV_TEMPERATURE_LSB     (0x1E)

#define BM1383AGLV_WAIT_READY_STATUS    0

#ifdef _DEBUG
extern Serial pc;
#define DEBUG_PRINT(...) pc.printf(__VA_ARGS__)
#else
#define DEBUG_PRINT(...)
#endif

/**
* BM1383AGLV pressure sensor example
*
* @code
* BM1383AGLV sensor(I2C_SDA, I2C_SCL);
* Serial pc(USBTX, USBRX);
* 
* int main() {
*     pc.printf("\nBM1383AGLV Pressure sensor library test program.\n");
*
*     while(1) {
*         pc.printf("pressure=%7.2f, temperature=%5.3f\n", sensor.getPressure(), sensor.getTemperature());
*         wait(0.5);
*     }
* }
* @endcode
*/
class BM1383AGLV
{
public:
    /**
    * BM1383AGLV constructor
    *
    * @param sda SDA pin
    * @param sdl SCL pin
    * @param addr slave address of the I2C peripheral (default: 0xBA)
    */
    BM1383AGLV(PinName sda, PinName scl, int addr = BM1383AGLV_DEFAULT_SLAVE_ADDRESS);

    /**
     * Create a BM1383AGLV instance which is connected to specified I2C pins
     * with specified address
     *
     * @param i2c_obj I2C object (instance)
     * @param addr slave address of the I2C-bus peripheral (default: 0xBA)
     */
    BM1383AGLV(I2C &i2c_obj, int addr = BM1383AGLV_DEFAULT_SLAVE_ADDRESS);

    /**
    * BM1383AGLV destructor
    */
    ~BM1383AGLV();

    /** Initializa BM1383AGLV sensor
     *
     *  Configure sensor setting
     *
     */
    void initialize(void);

    /**
     * Get pressure
     *
     * @returns pressure
     */
    float getPressure();

    /**
     * Get temerature
     *
     * @returns temperature
     */
    float getTemperature();

private:
    I2C *i2c_p;
    I2C &i2c;
    int m_addr;
    uint8_t m_buf[3];
    void readRegs(int addr, uint8_t * data, int len);
    void writeRegs(uint8_t * data, int len);

};

#endif
