/*****************************************************************************
  BH1749NUC.cpp

 Copyright (c) 2018-2019 ROHM Co.,Ltd.

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
******************************************************************************/
#include "mbed.h"
#include "BH1749NUC.h"

BH1749NUC::BH1749NUC(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr)
{
}

BH1749NUC::~BH1749NUC()
{
    m_addr = 0;
}

int BH1749NUC::init(void)
{
    int rc;
    char reg;

    rc = read(BH1749NUC_SYSTEM_CONTROL, &reg, sizeof(reg));
    if (rc != 0) {
        DEBUG_PRINT("Can't access BH1749NUC\n");
        return (rc);
    }
    reg = reg & 0x3F;
    DEBUG_PRINT("BH1749NUC Part ID Value = %02x\n", reg);

    if (reg != BH1749NUC_PART_ID_VAL) {
        DEBUG_PRINT("Can't find BH1749NUC\n");
        return (rc);
    }

    rc = read(BH1749NUC_MANUFACTURER_ID, &reg, sizeof(reg));
    if (rc != 0) {
        DEBUG_PRINT("Can't access BH1749NUC\n");
        return (rc);
    }
    DEBUG_PRINT("BH1749NUC MANUFACTURER ID Register Value = %02x\n", reg);

    if (reg != BH1749NUC_MANUFACT_ID_VAL) {
        DEBUG_PRINT("Can't find BH1749NUC\n");
        return (rc);
    }

    reg = BH1749NUC_MODE_CONTROL1_VAL;
    rc = write(BH1749NUC_MODE_CONTROL1, &reg, sizeof(reg));
    if (rc != 0) {
        DEBUG_PRINT("Can't write BH1749NUC MODE_CONTROL1 register\n");
        return (rc);
    }

    reg = BH1749NUC_MODE_CONTROL2_VAL;
    rc = write(BH1749NUC_MODE_CONTROL2, &reg, sizeof(reg));
    if (rc != 0) {
        DEBUG_PRINT("Can't write BH1749NUC MODE_CONTROL2 register\n");
        return (rc);
    }

    wait_ms(WAIT_TMT2_MAX);
    return (rc);
}

int BH1749NUC::get_rawval(char *data)
{
    int rc;

    rc = read(BH1749NUC_RED_DATA_LSB, data, GET_BYTE_RED_TO_GREEN2);
    if (rc != 0) {
        DEBUG_PRINT("Can't get BH1749NUC RGB, IR and GREEN2 value\n");
    }

    return (rc);
}

int BH1749NUC::get_val(unsigned short *data)
{
    int rc;
    char val[GET_BYTE_RED_TO_GREEN2];

    rc = get_rawval(val);
    if (rc != 0) {
        return (rc);
    }

    //val[6] and val[7] are RESERVED Register Value
    data[0] = ((unsigned short)val[1] << 8) | val[0];
    data[1] = ((unsigned short)val[3] << 8) | val[2];
    data[2] = ((unsigned short)val[5] << 8) | val[4];
    data[3] = ((unsigned short)val[9] << 8) | val[8];
    data[4] = ((unsigned short)val[11] << 8) | val[10];

    return (rc);
}

int BH1749NUC::write(char memory_address, char *data, int size)
{
    int rc = 0;

    char t = memory_address;
    m_i2c.write(m_addr, &t, 1, true);
    rc = m_i2c.write(m_addr, data, size);
    return (rc);
}

int BH1749NUC::read(char memory_address, char *data, int size)
{
    char t = memory_address;
    m_i2c.write(m_addr, &t, 1, true);
    return m_i2c.read(m_addr, data, size);
}
