//LCD表示系を一新した
//1行目に走行モード、機体状態、記憶マーカ数を表示
//2行目にDefault_Speedの値を表示する。


////ライントレースサンプル
#include "mbed.h"
#include "CRotaryEncoder.h"
#include "TB6612.h"
#include "AQM0802.h"


//☆★☆★各種パラメータ調整箇所☆★☆★☆★
#define     DEFAULT_SPEED       300     //1走目の基本速度
#define     LOW_SPEED           300     //2走目の低速速度[mm/sec]
#define     MEDIUM_SPEED        500     //2走目の低速速度[mm/sec]
#define     HIGH_SPEED          750     //2走目の低速速度[mm/sec]
#define     DEFAULT_HIGH_SPEED  700     //3走目の低速速度[mm/sec]
#define     STOP_DISTANCE   100000      //停止距離200000[um]⇒20[cm]
#define     TURN_POWER      0.6         //コースアウト時の旋回力
#define     PULSE_TO_UM     28          //エンコーダ1パルス当たりのタイヤ移動距離[um]
#define     INTERRUPT_TIME  1000        //割りこみ周期[us]
#define     DEFAULT_GRAY    0.2f        //フォトリフレクタデジタル入力の閾値
                                        //シリアル通信でSensor_Digital値を確認し調整する。
#define     MARKER_WIDTH    10000       //マーカ幅[um]（ビニルテープ幅19000[um]以内）
                                        //コースの傷によってマーカ誤検知する場合は値を大きくする。
#define     CROSS_JUDGE     4           //ラインセンサいくつ以上白線検知で交差点認識するか設定。
//モータ速度のゲイン関連(むやみに調整しない)
#define     M_KP            0.002f      //P(比例)制御成分
#define     M_KD            0.001f      //D(微分)制御成分

//フォトリフレクタのゲイン（外側に行くにつれ値を何倍させたいか調整する。）
#define     S_K1            1.0f        //float演算させる値には必ずfを付ける
#define     S_K2            2.0f        //2倍
#define     S_K3            3.0f        //4倍


//ラインセンサ各種制御成分
#define     S_KP_DEFAULT    1.0f        //ラインセンサ比例成分。大きいほど曲がりやすい
#define     S_KP_HIGH       1.0f
#define     S_KP_MEDIUM     1.0f
#define     S_KP_LOW        1.0f
#define     S_KP_DEFAULT_HIGH   1.0f

#define     S_KD_DEFAULT    1.0f        //ラインセンサ微分成分。大きいほど急なラインずれに強くなる。
#define     S_KD_HIGH       1.0f
#define     S_KD_MEDIUM     1.0f
#define     S_KD_LOW        1.0f
#define     S_KD_DEFAULT_HIGH   0.7f
//////////☆★☆★☆★☆★☆★//////////////



//スイッチ状態の定義
#define PUSH    0                   //スイッチ押したときの状態
#define PULL    1                   //スイッチ離したときの状態
//機体状態の定義
#define STOP                0x80    //機体停止状態
#define RUN_START           0x40    //スタートマーカ通過
#define RUN_COURSE_LOUT     0x20    //左コースアウト状態
#define RUN_COURSE_CENTER   0x18    //ライン中央走行状態
#define RUN_COURSE_ROUT     0x04    //右コースアウト状態
#define SECOND_RUN          0x02    //機体停止状態
#define TUARD_RUN           0x01    //機体設定モード


//デジタル入力オブジェクト定義
DigitalIn   push_sw(D13);
/////アナログ入力オブジェクト定義//////////
  
AnalogIn    s1(D3);
AnalogIn    s2(A6);
AnalogIn    s3(A5);
AnalogIn    s4(A4);
AnalogIn    s5(A3);
AnalogIn    s6(A2);
AnalogIn    s7(A1);
AnalogIn    s8(A0);

///////////////////////////////////////  
CRotaryEncoder encoder_a(D1,D0);    //モータAのエンコーダ
CRotaryEncoder encoder_b(D11,D12);  //モータBのエンコーダ
Ticker      timer;                  //タイマ割込み用 
TB6612      motor_a(D2,D7,D6);      //モータA制御用(pwma,ain1,ain2)
TB6612      motor_b(D10,D8,D9);     //モータB制御用(pwmb,bin1,bin2)
AQM0802     lcd(I2C_SDA,I2C_SCL);   //液晶制御用

//使用変数の定義
int         Sw_Ptn      = PULL;     //スイッチの現在情報
int         Old_Sw_Ptn  = PULL;     //スイッチの過去の情報
int         Sw_Cnt = 0;             //スイッチの押された回数
char        Coner_str[3];           //LCD用配列。コーナマーカ通過数表示
            ///各ラインセンサ情報格納用
double      S1_Data,S2_Data,S3_Data,S4_Data,S5_Data,S6_Data,S7_Data,S8_Data; 
double      All_Sensor_Data;        //ラインセンサ総データ量
double      Sensor_Diff[2]={0,0};   //ラインセンサ偏差
double      Sensor_P =0.0f;         //ラインセンサP(比例成分)制御量
double      Sensor_D =0.0f;         //ラインセンサD(微分成分)制御量
double      Sensor_PD=0.0f;         //ラインセンサP,D成分の合計
char        Gray_Str[5];            //LCD閾値表示用文字列
float       Gray=DEFAULT_GRAY;      //白黒のしきい値設定
long int    Enc_Count_A=0,Enc_Count_B=0;    //エンコーダパルス数。現在速度推定に使用する。
long int    Enc_A_Rotate=0,Enc_B_Rotate=0;  //エンコーダパルス数。停止時の機体速度調整に使用。
long int    Stop_Distance=STOP_DISTANCE;    //ストップ時の徐行→停止に使う
        /////↓エンコーダパルス数。コース記憶時のエンコーダパルスの蓄積に使用する。
long long int    Memory_Enc_Count_A=0,Memory_Enc_Count_B=0;   
char        MemoryA_Str[5];                 //LCD表示用。左モータの走行距離格納
char        MemoryB_Str[5];                 //LCD表示用。右モータの走行距離格納
long int    Distance_A=0,Distance_B=0;      //タイヤ移動距離を格納[mm]。
                                            //機体の現在速度の推定に使用する。タイマ割込みごとにリセットされる。
            ////↓左右タイヤの走行距離格納。1走目のコース記憶処理に使用する。マーカ通過ごとにリセット。                                
long int    Distance_memory_A=0, Distance_memory_B=0;

long int    S_Kp=S_KP_DEFAULT;////センサP成分
long int    S_Kd=S_KD_DEFAULT;////センサD成分

long int Marker_Run_Distance=0; 
long int Speed_A=0,  Speed_B=0;             //機体の現在速度を格納
long int Default_Speed  = DEFAULT_SPEED;    //1走目の標準速度
long int Low_Speed      = LOW_SPEED;        //2走目以降の低速
long int Medium_Speed   = MEDIUM_SPEED;     //2走目以降の中速
long int High_Speed     = HIGH_SPEED;       //2走目以降の高速
long int Default_High_Speed   =  DEFAULT_HIGH_SPEED;//3走目の高速
char     Speed_Str[5];                      //LCD速度表示用文字列
long int Target_Speed_A=0,Target_Speed_B=0; //目標速度
long int Motor_A_Diff[2]={0,0};             //過去の速度偏差と現在の速度偏差を格納
long int Motor_B_Diff[2]={0,0};             //
float Motor_A_P,Motor_B_P;                  //モータ速度制御P成分
float Motor_A_D,Motor_B_D;                  //モータ速度制御D成分
float Motor_A_PD,Motor_B_PD;                //モータ速度制御PD合成
float Motor_A_Pwm,Motor_B_Pwm;              //モータへの出力
unsigned char Sensor_Digital    =0x00;
unsigned char Old_Sensor_Digital=0x00;
int   Sensor_Cnt=0;
unsigned char Machine_Status    =STOP;      //機体状態
unsigned char Old_Machine_Status=0x00;      //過去の機体状態
int Marker_Pass_Flag = 0;                   //マーカ通過中であることを示すフラグ
int Old_Marker_Pass_Flag=0;                 //過去のマーカ通過情報
int Corner_Flag=0;                          //コーナセンサがマーカを検知したことを示すフラグ
int SG_Flag=0;                              //スタート・ゴールセンサがマーカを検知したことを示すフラグ
int SG_Cnt=0;                               //スタート・ゴールセンサのマーカ通過数
int Cross_Flag=0;                           //交差点を通過したことを示すフラグ
long int Imaginary_Speed=0;                 //曲率演算にて求めた２走目以降の仮想走行速度
long long int course_data[100][3];                  //コース情報の記憶用配列マーカ数は行数分まで対応し、
                                            //列情報は左タイヤ走行距離、右タイヤ走行距離、演算で求めた2走目走行速度を格納する。
int Row=0;                                  //行情報を格納。マーカ通過ごとに加算し、走行終了後リセットされる。
int Marker_Cnt=0;


void sensor_analog_read(){//ラインセンサの情報取得処理。
    S1_Data=s1.read();
    S2_Data=s2.read();
    S3_Data=s3.read();
    S4_Data=s4.read();
    S5_Data=s5.read();
    S6_Data=s6.read();
    S7_Data=s7.read();
    S8_Data=s8.read();    
}

//８つのフォトリフレクタの入力を8ビットのデジタルパターンに変換。センサの状態をデジタルパターンで
//扱うために使用する。（if文などの条件式でセンサ情報を使いやすいようにするため）
void sensor_digital_read(){
    Sensor_Cnt=0;
    Old_Sensor_Digital=Sensor_Digital;
    if(S1_Data>Gray){
            Sensor_Digital |= 0x80;   //7ビット目のみセット (1にする。)
    }else   Sensor_Digital &= 0x7F;   //7ビット目のみマスク（0にする。）
    if(S2_Data>Gray){
            Sensor_Digital |= 0x40;   //6ビット目のみセット (1にする。)
            Sensor_Cnt++;
    }else   Sensor_Digital &= 0xBF;   //6ビット目のみマスク（0にする。）
    if(S3_Data>Gray){
            Sensor_Digital |= 0x20;   //5ビット目のみセット (1にする。)
            Sensor_Cnt++;
    }else    Sensor_Digital &= 0xDF;   //5ビット目のみマスク（0にする。）
    if(S4_Data>Gray){
            Sensor_Digital |= 0x10;   //4ビット目のみセット (1にする。)
            Sensor_Cnt++;
    }else    Sensor_Digital &= 0xEF;   //4ビット目のみマスク（0にする。）
    if(S5_Data>Gray){
            Sensor_Digital |= 0x08;   //3ビット目のみセット (1にする。)
            Sensor_Cnt++;
    }else   Sensor_Digital &= 0xF7;   //3ビット目のみマスク（0にする。）
    if(S6_Data>Gray){
            Sensor_Digital |= 0x04;   //2ビット目のみセット (1にする。)
            Sensor_Cnt++;
    }else   Sensor_Digital &= 0xFB;   //2ビット目のみマスク（0にする。）
    if(S7_Data>Gray){
            Sensor_Digital |= 0x02;   //1ビット目のみセット (1にする。)
            Sensor_Cnt++;
    }else   Sensor_Digital &= 0xFD;   //1ビット目のみマスク（0にする。）
    if(S8_Data>Gray){
            Sensor_Digital |= 0x01;   //0ビット目のみセット (1にする。)
    }else   Sensor_Digital &= 0xFE;   //0ビット目のみマスク（0にする。）    
}

//機体の状態をデジタルパターンでセットする関数。モニタリング、条件分岐で
//機体の状態が交差点通過中なのか停止状態なのか判定するために使用する。
void Machine_Status_Set(){
    Old_Machine_Status=Machine_Status; 
       
    //機体がライン中央に位置するとき
    if(Sensor_Digital&RUN_COURSE_CENTER )Machine_Status|=RUN_COURSE_CENTER;
    else Machine_Status &= 0xE7;//ライン中央情報のマスク
    if((Sensor_Digital==0x00)&&(Old_Sensor_Digital==0x40)){//左センサコースアウト時
        Machine_Status|=RUN_COURSE_LOUT;//左コースアウト状態のビットをセット
    }else if((Machine_Status&RUN_COURSE_LOUT)&&(Sensor_Digital&RUN_COURSE_CENTER)){
        //左コースアウト状態かつ機体がライン中央に復帰したとき
        Machine_Status &= 0xDF;//左コースアウト情報のみマスク    
    }
    if((Sensor_Digital==0x00)&&(Old_Sensor_Digital==0x02)){//右センサコースアウト時
        Machine_Status|=RUN_COURSE_ROUT;//右コースアウト状態のビットをセット
    }else if((Machine_Status&RUN_COURSE_ROUT)&&(Sensor_Digital&RUN_COURSE_CENTER)){
        //右コースアウト状態かつ機体がライン中央に復帰したとき
        Machine_Status &= 0xFB;//右コースアウト情報のみマスク    
    }     
}

//コーナマーカ検知時に動作。
//エンコーダから取得されたパルス数を距離換算し、事前用意した記憶用配列に格納する。
void corner_curvature(){
    course_data[Row][0]=0;//予め配列の情報は初期化しておく。
    course_data[Row][1]=0;//
    Imaginary_Speed=0;
    ////まずはマーカ間の走行距離を取得する。            
    course_data[Row][0]=(Memory_Enc_Count_A*PULSE_TO_UM);//左タイヤのマーカ間の走行距離を格納
    course_data[Row][1]=(Memory_Enc_Count_B*PULSE_TO_UM);//右タイヤのマーカ間の走行距離を格納                
                                
    //左右タイヤの走行距離の差分を用いた曲率演算
    if(((course_data[Row][0])<(course_data[Row][1]*1.2f)) || ((course_data[Row][1])<(course_data[Row][0]*1.2f))){//左右の回転差が20%以下の時
        Imaginary_Speed=High_Speed;//高速とする。    
    }else if((course_data[Row][0]<(course_data[Row][1]*1.5)) || (course_data[Row][1]<(course_data[Row][0]*1.5))){//左右の回転差が５0%以下の時
        Imaginary_Speed=Medium_Speed;//中カーブは中速とする。
    }else if((course_data[Row][0]<(course_data[Row][1]*2)) || (course_data[Row][1]<(course_data[Row][0]*2))){//左右の回転差が100%以下の時
        Imaginary_Speed=Low_Speed;//低速
    }
    else{
        Imaginary_Speed=Low_Speed;//低速
    }                              
    course_data[Row][2]=Imaginary_Speed; //仮想の演算速度を格納
                                             
}

//2走目の加減速走行に使用。1走目で演算したマーカ間の仮想速度を機体目標速度に設定する。
void second_speed_control(){
    if(course_data[Row][2]==Default_Speed){//標準速度区間のとき
        S_Kp=S_KP_DEFAULT;
        S_Kd=S_KD_DEFAULT;        
    }else if(course_data[Row][2]==Low_Speed){//低速走行区間のとき
        S_Kp=S_KP_LOW;
        S_Kd=S_KD_LOW;
    }else if(course_data[Row][2]==Medium_Speed){//中速走行区間のとき
        S_Kp=S_KP_MEDIUM;
        S_Kd=S_KD_MEDIUM;        
    }else if(course_data[Row][2]==High_Speed){//高速走行区間のとき
        S_Kp=S_KP_HIGH;
        S_Kd=S_KD_HIGH;        
    }else{//
        S_Kp=S_KP_DEFAULT_HIGH;
        S_Kd=S_KD_DEFAULT_HIGH;         
    }    
    Target_Speed_A=course_data[Row][2];//記憶走行で演算した仮想速度を使う
    Target_Speed_B=course_data[Row][2];                                   
}


/////LCD表示用関数。機体の現在状態を表示する。
void display_print(void){
    char lcd_top_str[8],lcd_low_str[8];   //lcd表示用配列を用意   各行分
    char stop_or_go;                //機体停止状態'-'か走行状態'G'を格納
    //////表示クリア
    lcd.locate(0,0);
    lcd.print("        ");
    lcd.locate(0,1);
    lcd.print("        ");
        
    ////////0行目の表示//////////////////////////////
    if(Machine_Status&STOP){            //機体停止状態のとき
        stop_or_go='-';
    }else if(!(Machine_Status&STOP)){   //機体走行状態のとき
        stop_or_go='G';
    }
    sprintf(lcd_top_str,"%d:%c %03d",Sw_Cnt,stop_or_go,Marker_Cnt);//0行の表示文字の整理
    lcd.locate(0,0);                    //表示位置を0行目にセット
    lcd.print(lcd_top_str);             //表示
    
    ////////1行目の表示/////////////////////////////
    if(Sw_Cnt<=2){                          //1走目か2走目のとき
            sprintf(lcd_low_str,"SPD:%04d",Default_Speed);        //1行の表示文字の整理
    }else   sprintf(lcd_low_str,"SPD:%04d",Default_High_Speed);   //3走目のとき
    lcd.locate(0,1);                    //表示位置を1行目にセット
    lcd.print(lcd_low_str);             //表示
}



//タイマ割り込み1[ms]周期
void timer_interrupt(){
    
    //ラインセンサ情報取得
    sensor_analog_read();
    sensor_digital_read();
    
    //機体状態の取得
    Machine_Status_Set();
    
    //交差点の認識
    if(Sensor_Cnt>=CROSS_JUDGE )Cross_Flag=1;//ラインセンサ4つ以上検知状態の時は交差点を示す。
    
    //エンコーダパルス数の取得
    Enc_Count_A=encoder_a.Get();            
    Enc_Count_B=-encoder_b.Get();
    
    if(!(Machine_Status&STOP)){//機体走行中のとき
        //////コース記憶に使用するエンコーダパルス数の蓄積処理
        Memory_Enc_Count_A+=Enc_Count_A;
        Memory_Enc_Count_B+=Enc_Count_B;    
    }   
    //各種マーカの検知
    Old_Marker_Pass_Flag=Marker_Pass_Flag;//過去のフラグを退避
    if(Sensor_Digital&0x81){                   //マーカセンサ検知時
        Marker_Pass_Flag=1;                     //マーカ通過中フラグをON
        if(Sensor_Digital&0x80)Corner_Flag=1;  //コーナセンサの検知
        if(Sensor_Digital&0x01)SG_Flag=1;      //スタートゴールセンサの検知
        if((Corner_Flag==1)&&(SG_Flag==1));//交差点通過中。何もしない    
    }else Marker_Pass_Flag=0;//マーカ通過終了       

 
    //マーカ通過後、マーカ種類判別    
    if((Old_Marker_Pass_Flag==1)&&(Marker_Pass_Flag==0)){//マーカ通過後
        if(Marker_Run_Distance>MARKER_WIDTH){   //マーカ幅がもっともらしいとき
            if(Cross_Flag==1);                  //交差点の時は何もしない
            else if((SG_Flag==1)&&(SG_Cnt==0)){ //ゴールスタートマーカの時⇒1回目
                SG_Cnt=1;
                if(Sw_Cnt==1){                  //1走目のとき                  
                    corner_curvature();     //曲率に応じた速度推定処理
                    Memory_Enc_Count_A=0;   //マーカ間の左タイヤエンコーダパルス数のクリア
                    Memory_Enc_Count_B=0;   //マーカ間の右タイヤエンコーダパルス数のクリア                
                    Row++;
                }else if(Sw_Cnt==2){            //2走目のとき                            
                    second_speed_control(); //記憶情報から目標速度を設定する処理
                    Row++;                  //次のコース情報へ                                            
                }else{//3走目は定速で高速走行
                    Target_Speed_A=Default_High_Speed;
                    Target_Speed_B=Default_High_Speed;
                }
            }else if((SG_Flag==1)&&(SG_Cnt==1)){//ゴールスタートマーカの時⇒2回目
                if(Sw_Cnt==1){
                    corner_curvature();         //曲率に応じた速度推定処理
                    Memory_Enc_Count_A=0;       //マーカ間の左タイヤエンコーダパルス数のクリア
                    Memory_Enc_Count_B=0;       //マーカ間の右タイヤエンコーダパルス数のクリア                                     
                }else if(Sw_Cnt==2){                //2走目のとき　　　何もしない
                }else;                          //3走目のとき　　　何もしない
                Marker_Cnt=Row;                 //マーカカウント数をLCD表示用に退避させる                                    
                Machine_Status|=STOP;           //機体停止状態へ                
                Row=0;                          //コース記憶用配列の行情報[通過マーカ情報]の初期化 
                SG_Cnt=0;                       //スタート・ゴールマーカ情報のリセット   
            }else if(Corner_Flag==1){           //コーナマーカの時  
                if(Sw_Cnt==1){                      //1走目のとき
                    corner_curvature();         //曲率に応じた速度推定処理
                    Memory_Enc_Count_A=0;       //マーカ間の左タイヤエンコーダパルス数のクリア
                    Memory_Enc_Count_B=0;       //マーカ間の右タイヤエンコーダパルス数のクリア                                     
                    Row++;
                }else if(Sw_Cnt==2){                //2走目のとき             
                    second_speed_control();     //記憶情報から目標速度を設定する処理  
                    Row++;                  //次のコース情報へ                                          
                }else{//3走目は定速で高速走行
                    Target_Speed_A=Default_High_Speed;
                    Target_Speed_B=Default_High_Speed;
                }                                                              
            }
        }else;//マーカではなく、誤検知だった場合。何もしない
        
        
        
        //////マーカ通過後は各種マーカ判定用フラグ、パラメータのリセット処理を行う。
        Corner_Flag=0;
        SG_Flag=0;
        Cross_Flag=0;
        Marker_Run_Distance=0;                  //マーカ通過距離情報リセット
        
    }/////////////////各種マーカ判定処理の終了
    
    //センサ取得値の重ね合わせ（端のセンサほどモータ制御量を大きくする）
    All_Sensor_Data=-(S2_Data*S_K3+S3_Data*S_K2+S4_Data*S_K1)+(S5_Data*S_K1+S6_Data*S_K2+S7_Data*S_K3);
    Sensor_Diff[1]=Sensor_Diff[0];//過去のラインセンサ偏差を退避
    Sensor_Diff[0]=All_Sensor_Data;
    Sensor_P=All_Sensor_Data*S_Kp;                  //ラインセンサ比例成分の演算       
    Sensor_D=(Sensor_Diff[0]-Sensor_Diff[1])*S_Kd;  //ラインセンサ微分成分の演算
    Sensor_PD=Sensor_P+Sensor_D; 
          
    ////モータ現在速度の取得            
    Distance_A=(Enc_Count_A*PULSE_TO_UM);   //移動距離をmm単位で格納
    Distance_B=(Enc_Count_B*PULSE_TO_UM);
    Speed_A=(Distance_A*1000)/INTERRUPT_TIME;//走行速度演算[mm/s]
    Speed_B=(Distance_B*1000)/INTERRUPT_TIME;
    /////モータの速度制御
    //過去の速度偏差を退避
    Motor_A_Diff[1]=Motor_A_Diff[0];
    Motor_B_Diff[1]=Motor_B_Diff[0];
    //現在の速度偏差を取得。    
    Motor_A_Diff[0]=(Target_Speed_A-Speed_A);
    Motor_B_Diff[0]=(Target_Speed_B-Speed_B);   

    //P成分演算
    Motor_A_P=Motor_A_Diff[0]*M_KP;
    Motor_B_P=Motor_B_Diff[0]*M_KP;
    //D成分演算
    Motor_A_D=(Motor_A_Diff[0]-Motor_A_Diff[1])*M_KD;
    Motor_B_D=(Motor_B_Diff[0]-Motor_B_Diff[1])*M_KD;
    //モータ速度制御のPD合成
    Motor_A_PD=Motor_A_P+Motor_A_D;
    Motor_B_PD=Motor_B_P+Motor_B_D;
    //最終的なモータ制御量の合成    
    Motor_A_Pwm=Motor_A_PD+Sensor_PD;
    Motor_B_Pwm=Motor_B_PD-Sensor_PD;

   //モータ制御量の上限下限設定
    if(Motor_A_Pwm>0.95f)Motor_A_Pwm=0.95f;
    else if(Motor_A_Pwm<-0.95)Motor_A_Pwm=-0.95f;
    if(Motor_B_Pwm>0.95f)Motor_B_Pwm=0.95f;
    else if(Motor_B_Pwm<-0.95f)Motor_B_Pwm=-0.95f;       
    //モータへの出力    
    if(!(Machine_Status&STOP)){//マシンが停止状態でなければ
        if(Machine_Status&RUN_COURSE_LOUT){         //左端センサ振り切れた時
            motor_a=-(-TURN_POWER); //左旋回
            motor_b=-(TURN_POWER);    
        }else if(Machine_Status&RUN_COURSE_ROUT){   //右端センサ振り切れた時
            motor_a=-(TURN_POWER);  //右旋回    
            motor_b=-(-TURN_POWER);
        }else if(Cross_Flag==1){//交差点通過中
            motor_a=-0.3;//交差点なので控えめの速度で直進
            motor_b=-0.3;
        }
        else{
            motor_a=-Motor_A_Pwm;
            motor_b=-Motor_B_Pwm;
        }
    }else{//停止状態の時はモータへの出力は無効
        if(Stop_Distance<STOP_DISTANCE){
            motor_a=-Motor_A_Pwm/4;
            motor_b=-Motor_B_Pwm/4;

        }else{
            motor_a=0;
            motor_b=0;
        }
    }
    //////機体停止時の徐行用の走行距離演算
    if(Machine_Status&STOP){//機体停止状態の時
        if(Stop_Distance>=STOP_DISTANCE){
         Enc_A_Rotate+=Enc_Count_A;//閾値用に左エンコーダ値の蓄積 
         Enc_B_Rotate+=Enc_Count_B;//速度用に右エンコーダ値の蓄積                   
        }    
        if(Enc_A_Rotate<-6400)Enc_A_Rotate=-6400;
        if(Enc_A_Rotate>6400)Enc_A_Rotate=6400;        
        if(Enc_B_Rotate<-6400)Enc_B_Rotate=-6400;
        if(Enc_B_Rotate>6400)Enc_B_Rotate=6400;
        if((Distance_A>=0)&&(Distance_B>=0))Stop_Distance+=(Distance_A+Distance_B)/2;        
        if(Stop_Distance<0)Stop_Distance=0;
        if(Stop_Distance>STOP_DISTANCE)Stop_Distance=STOP_DISTANCE;
    }  
    if(Marker_Pass_Flag==1){//マーカ通過中は通過距離情報を蓄積する。
        Marker_Run_Distance+=(Distance_A+Distance_B)/2;
    }     
               
    //割込み終了時の各種パラメータリセット処理 
    //エンコーダ関連情報のリセット
    encoder_a.Set(0);//エンコーダクラスのパルス数情報のリセット
    encoder_b.Set(0);
    
    Enc_Count_A = 0;//速度制御用エンコーダパルス数情報のリセット                    
    Enc_Count_B = 0;//
    Distance_A=0;   //速度制御用距離情報のリセット
    Distance_B=0;    
}

int main() { 
    timer.attach_us(&timer_interrupt,INTERRUPT_TIME);//タイマ割り込みスタート 
    lcd.cls();//表示クリア
    
    while(1){
        Old_Sw_Ptn=Sw_Ptn;  //過去のスイッチ入力情報を退避
        Sw_Ptn=push_sw;     //現在のスイッチ入力情報の取得
        display_print();       //LCDに現在の機体状態を表示
        
        if(Machine_Status&STOP){//機体停止状態の時                                 
            if(Sw_Cnt<=2)Default_Speed=DEFAULT_SPEED+(Enc_B_Rotate/16);                 //1,2走目標準速度調整                
            else if(Sw_Cnt>2)Default_High_Speed=DEFAULT_HIGH_SPEED+(Enc_B_Rotate/16);  //3走目
        }    
                 
        if((Old_Sw_Ptn==PULL)&&(Sw_Ptn==PUSH)){//スイッチが押された瞬間
            if(Machine_Status&STOP){//機体停止状態の時
                Sw_Cnt++;  
                wait(0.5);
                display_print();       //LCDに現在の機体状態を表示                
                wait(1.5);
                Corner_Flag=0;
                SG_Flag=0;
                Cross_Flag=0;
                Row=0;                          //コース記憶用配列の行情報[通過マーカ情報]の初期化 
                SG_Cnt=0;                       //スタート・ゴールマーカ情報のリセット                   
                Marker_Run_Distance=0;//マーカ通過距離情報リセット                
                if(Sw_Cnt<=2){          //1,2走目のとき
                    S_Kp=S_KP_DEFAULT;    
                    S_Kd=S_KD_DEFAULT;
                    Target_Speed_A=Default_Speed;
                    Target_Speed_B=Default_Speed;                      
                }else{              //3走目以降のとき
                    S_Kp=S_KP_DEFAULT_HIGH;     
                    S_Kd=S_KD_DEFAULT_HIGH;
                    Target_Speed_A=Default_High_Speed;
                    Target_Speed_B=Default_High_Speed;                                        
                }
                Stop_Distance=0;
                Machine_Status&=0x7F;//ストップ状態解除 
     
            }else{//機体走行中であったとき
                //各種フラグのクリア
                Corner_Flag=0;
                SG_Flag=0;
                Cross_Flag=0;
                Row=0;                          //コース記憶用配列の行情報[通過マーカ情報]の初期化 
                SG_Cnt=0;                       //スタート・ゴールマーカ情報のリセット                   
                Marker_Run_Distance=0;//マーカ通過距離情報リセット
                Machine_Status |= STOP;//機体停止状態にする。                
            }
        }               
    }
}