/*******************************************************************************
* RenBED Car used to drive RenBuggy with servo, motor and encoder(optional)    *
* Copyright (c) 2014 Mark Jones                                                *
*                                                                              *
* Permission is hereby granted, free of charge, to any person obtaining a copy *
* of this software and associated documentation files (the "Software"), to deal*
* in the Software without restriction, including without limitation the rights *
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell    *
* copies of the Software, and to permit persons to whom the Software is        *
* furnished to do so, subject to the following conditions:                     *
*                                                                              *
* The above copyright notice and this permission notice shall be included in   *
* all copies or substantial portions of the Software.                          *
*                                                                              *
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR   *
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,     *
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE  *
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER       *
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,*
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN    *
* THE SOFTWARE.                                                                *
*                                                                              *
* Car.cpp                                                                      *
*                                                                              *
* V1.1 31/03/2014                                          Mark Jones          *
*******************************************************************************/

#ifndef CAR_C
#define CAR_C

#include "Car.h"
#include "mbed.h"

const int SERVO_PWM = 1500;             // 1500 = centre.
const int SERVO_PWM_PERIOD = 2000;  
const int SERVO_PWM_RANGE = 500;        // + or - 500 microseconds.
const float SERVO_DEGREES_RANGE = 45.0; // + or - from centre is full right/left.
    
const int MOTOR_PWM = 20000;
const int MOTOR_PERIOD = 20000;

Car::Car(PinName servoPin, PinName motorPin) 
 : m_servo(servoPin), m_motor(motorPin) {
    
    m_speed = 15000;
    m_countsPerRevolution = 0;
    m_wheelCircumference = 0;
    
    configureServo_us(SERVO_PWM, SERVO_PWM_PERIOD, 
    SERVO_PWM_RANGE, SERVO_DEGREES_RANGE);
    
    configureMotor_us(MOTOR_PWM, MOTOR_PERIOD);
}

Car::Car(PinName servoPin, PinName motorPin, int countsPerRevolution, float wheelCircumference, PinName sensorPin)
 : m_servo(servoPin), m_motor(motorPin), m_sensor(sensorPin) {
    configureEncoder(countsPerRevolution, wheelCircumference);
    
    m_speed = 15000;
    setDirection(0);
    
    configureServo_us(SERVO_PWM, SERVO_PWM_PERIOD, 
    SERVO_PWM_RANGE, SERVO_DEGREES_RANGE);
    
    configureMotor_us(MOTOR_PWM, MOTOR_PERIOD);
    
    m_encoderCount = 0;
    m_sensor.setSampleFrequency(1000);
    m_sensor.setSamplesTillHeld(5);

    Car* basePointer = dynamic_cast<Car*>(this);    
    m_sensor.attach_deasserted_held(basePointer, &Car::updateEncoderCount);
}

Car::~Car() {
}

void Car::setSpeed(int speed_us) {
    m_speed = speed_us;
}

void Car::updateEncoderCount() {
    m_encoderCount++;
}

void Car::forwards_measured(float distance) {  
    
    int countsForward = (int)(distance * (m_countsPerRevolution / m_wheelCircumference));
    
    m_encoderCount = 0;
    bool isMoving = true;
    m_motor.pulsewidth_us(m_speed);
    
    while(isMoving) {
        wait(0.2); // <<-- for some unknown reason, this requires a delay to work :-S
        if(countsForward < m_encoderCount)
        {
            isMoving = false;
        }
    }
    // When it's finished, stop the buggy.
    stop();
    
    return;
}   

void Car::forwards_timed() {
    m_motor.pulsewidth_us(m_speed);
}

void Car::stop() {
    m_motor.pulsewidth_us(0);
}

void Car::setDirection(int degrees) {
    float angleOffset = m_servoRange * (m_servoDegrees / degrees);
    m_servo.pulsewidth_us(1500 + angleOffset);
}

void Car::configureServo_us(int pulsewidth_us, int period_us, int range, float degrees) {
    m_servo.pulsewidth_us(pulsewidth_us);
    m_servo.period_us(period_us);
    m_servoRange = range;
    m_servoDegrees = degrees;
}

void Car::configureServo_ms(int pulsewidth_ms, int period_ms, int range, float degrees) {
    m_servo.pulsewidth_ms(pulsewidth_ms);
    m_servo.period_ms(period_ms);
    m_servoRange = range;
    m_servoDegrees = degrees;
}

void Car::configureMotor_us(int pulsewidth_us, int period_us) {
    m_motor.pulsewidth_us(pulsewidth_us);
    m_motor.period_us(period_us);
}

void Car::configureMotor_ms(int pulsewidth_ms, int period_ms) {
    m_motor.pulsewidth_ms(pulsewidth_ms);
    m_motor.period_ms(period_ms);
}

void Car::configureEncoder(int countsPerRevolution, float wheelCircumference) {
    m_countsPerRevolution = countsPerRevolution;
    m_wheelCircumference = wheelCircumference; 
}

#endif // CAR_C