/*******************************************************************************
* RenBED Car used to drive RenBuggy with servo and 1 motor                     *
* Copyright (c) 2014 Mark Jones                                                *
*                                                                              *
* Permission is hereby granted, free of charge, to any person obtaining a copy *
* of this software and associated documentation files (the "Software"), to deal*
* in the Software without restriction, including without limitation the rights *
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell    *
* copies of the Software, and to permit persons to whom the Software is        *
* furnished to do so, subject to the following conditions:                     *
*                                                                              *
* The above copyright notice and this permission notice shall be included in   *
* all copies or substantial portions of the Software.                          *
*                                                                              *
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR   *
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,     *
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE  *
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER       *
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,*
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN    *
* THE SOFTWARE.                                                                *
*                                                                              *
* Car.h                                                                        *
*                                                                              *
* V1.1 31/03/2014                                           Mark Jones         *
*******************************************************************************/

#ifndef CAR_H
#define CAR_H

#include "mbed.h"
#include "PinDetect.h"

/**
* RenBuggyServo Example:
* @code
* 
* 
#include "mbed.h"
#include "Car.h"

Car myCar(P1_25, P1_24);

// Main entry point of application.
int main() {
    
    // Drive the RenBuggy!
    myCar.setSpeed(20000);
    myCar.setDirection(0);
    
    myCar.forwards();   
    wait(1);
    myCar.setDirection(45);
    wait(1);
    myCar.setDirection(-45);
    wait(1);
    myCar.setDirection(0);
    myCar.stop(); 
    
    myCar.forwards(10); // Move the car a final 10cm.
    myCar.stop();
}
// End programme.
@endcode
*/

/**
** The car class for controlling the RenBuggy_Servo. 
*/
class Car {  
    private:

    PwmOut m_servo;
    PwmOut m_motor;
    
    int m_speed;
    
    int m_encoderCount;
    
    int m_servoRange;       // Pulsewidth range to full left/right from centre (1.5ms)
    float m_servoDegrees;   // Angle to full right/left turn from centre (0).
    
    float m_wheelCircumference; // The circumference of the wheel with stripes.
    int m_countsPerRevolution;  // The number of stripes on the wheel.
    
    PinDetect m_sensor; // For debouncing.
    
    void updateEncoderCount();
    
    public:    
    /** Constructs the car with PwmOut objects for servo and motor.
    * 
    * @param servoPin is the pin used for pwm output for driving the servo.
    * @param motorPin is the pin used for pwm output for driving the motor.
    */
    Car(PinName servoPin, PinName motorPin);
    
    /** Constructs the car with PwmOut objects for servo and motor, and configures
    * the encoder.
    *
    * @param servoPin is the pin used for pwm output for driving the servo.
    * @param motorPin is the pin used for pwm output for driving the motor.
    * @param countsPerRevolution is the number of counts the encoder
    * makes in one full cycle of the wheel.
    * @param wheelCircumference is the circumference of the wheel being
    * read by the encoder.
    * @param sensorPin is the pin required for the encoder for debouncing.
    */
    Car(PinName servoPin, PinName motorPin, int countsPerRevolution, float wheelCircumference, PinName sensorPin);
    
    /**
    * Deconstructs the car.
    */
    ~Car();
    
    /**
    * Sets the speed the buggy will move at.
    * @param speed_us is the speed the car will move at, in microseconds.
    */
    void setSpeed(int speed_us);
    
    /**
    * Moves the car in the direction it is facing, for a specified
    * distance.
    * @param distance is how far the car will travel, in cm.
    */
    void forwards_measured(float distance);
    
    /**
    * Moves the car in the direction it is facing, until a user
    * tells it to stop.
    */
    void forwards_timed();
    
    /**
    * Stops the car from moving.
    */
    void stop();
    
    /**
    * Sets the direction the car will face. This is used to navigate the car.
    * @param degrees is the angle of the servo, where -45 is full 
    * left, 0 is centre and +45 is full right.
    */
    void setDirection(int degrees);
    
    /**
    * Configures the servo with a pulsewidth, period, range and degrees. Pulsewidth and
    * period is accepted in microsecond format.
    * @param pulsewidth_us is pwm pulsewidth for the servo, in mircoseconds.
    * @param period_us is the pwm period for the servo, in mircoseconds.
    * @param range is the pulsewidth range to full left/right turn of the servo from centre (1500us).
    * @param degrees is the angle to full right/left turn of the servo from centre (0).
    */
    void configureServo_us(int pulsewidth_us, int period_us, int range, float degrees);
    
    /**
    * Configures the servo with a pulsewidth, period, range and degrees. Pulsewidth and
    * period is accepted in millisecond format.
    * @param pulsewidth_ms is pwm pulsewidth for the servo, in millisecond.
    * @param period_ms is the pwm period for the servo, in millisecond.
    * @param range is the pulsewidth range to full left/right turn of the servo from centre (1.5ms).
    * @param degrees is the angle to full right/left turn of the servo from centre (0).
    */
    void configureServo_ms(int pulsewidth_ms, int period_ms, int range, float degrees);
    
    /**
    * Configures the pulsewidth and period for the motor, in microseconds.
    * @param pulsewidth_us is the pwm pulsewidth for the motor, in mircoseconds.
    * @param period_us is the pwm period for the motor, in microseconds.
    */
    void configureMotor_us(int pulsewidth_us, int period_us);
    
    /**
    * Configures the pulsewidth and period for the motor, in milliseconds.
    * @param pulsewidth_ms is the pwm pulsewidth for the motor, in milliseconds.
    * @param period_ms is the pwm period for the motor, in milliseconds.
    */
    void configureMotor_ms(int pulsewidth_ms, int period_ms);
    
    /**
    * Provides information required to make use of an encoder for 
    * specifying distance.
    * @param countsPerRevolution is the number of counts the encoder
    * makes in one full cycle of the wheel.
    * @param wheelCircumference is the circumference of the wheel being
    * read by the encoder.
    */
    void configureEncoder(int countsPerRevolution, float wheelCircumference);
};

#endif // CAR_H