/******************************************************************************
* DISCLAIMER
* This software is supplied by Renesas Electronics Corporation and is only
* intended for use with Renesas products. No other uses are authorized. This
* software is owned by Renesas Electronics Corporation and is protected under
* all applicable laws, including copyright laws.
* THIS SOFTWARE IS PROVIDED "AS IS" AND RENESAS MAKES NO WARRANTIES REGARDING
* THIS SOFTWARE, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING BUT NOT
* LIMITED TO WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
* AND NON-INFRINGEMENT. ALL SUCH WARRANTIES ARE EXPRESSLY DISCLAIMED.
* TO THE MAXIMUM EXTENT PERMITTED NOT PROHIBITED BY LAW, NEITHER RENESAS
* ELECTRONICS CORPORATION NOR ANY OF ITS AFFILIATED COMPANIES SHALL BE LIABLE
* FOR ANY DIRECT, INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES FOR
* ANY REASON RELATED TO THIS SOFTWARE, EVEN IF RENESAS OR ITS AFFILIATES HAVE
* BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
* Renesas reserves the right, without notice, to make changes to this software
* and to discontinue the availability of this software. By using this software,
* you agree to the additional terms and conditions found by accessing the
* following link:
* http://www.renesas.com/disclaimer
* Copyright (C) 2012 - 2015 Renesas Electronics Corporation. All
* rights reserved.
******************************************************************************/

/******************************************************************************
Includes   <System Includes> , "Project Includes"
******************************************************************************/
#include <stdio.h>
#include <ctype.h>
#include <stdint.h>
extern "C" {
#include "iodefine.h"
#include "dev_drv.h"
#include "devdrv_sdg.h"
}
#include "mbed.h"
#include "PinNames.h"

#include "piano4int.h"

/******************************************************************************
Macro definitions
******************************************************************************/
#define ASSERT( eval )                          \
    if (!(eval)) {                              \
        printf(                                 \
            "%s: failed! at %d in %s\n",        \
            #eval,                              \
            __LINE__,                           \
            __FILE__);                          \
    }

/* message to caller */
#define     UNIT_MSG    (32)


/******************************************************************************
Typedef definitions
******************************************************************************/
/* a sound generator unit */
typedef struct {
    uint16_t        speed;          /* count of a tempo     */
    uint8_t         loudness;       /* current loudness     */
    uint16_t        count;          /* remain of time       */
    NOTE            note;           /* a next note          */
    const int8_t    *score;         /* a next score         */
    const int8_t    *top;           /* top of a score       */
    uint8_t         channel;        /* channel number       */
#ifdef UNIT_MSG
    char            msg[UNIT_MSG];  /* message              */
#endif
} UNIT;


/******************************************************************************
Imported global variables and functions (from other files)
******************************************************************************/


/******************************************************************************
Exported global variables and functions (to be accessed by other files)
******************************************************************************/


/******************************************************************************
Private global variables and functions
******************************************************************************/
/* interrupt routine */
static void play_score(void);


/* all work area for piano */
struct {
    UNIT            unit[SDG_CH_TOTAL];
    uint16_t        fin;
    Ticker          metronome;          /* base tempo */
    PIANO4INT_CB    callback;
} pianoT;


/* init params for sound generator */
static const struct {
    uint16_t    channel;
    R_SDG_CLOCK clock;
} unitHW[SDG_CH_TOTAL] = {
    {   DEVDRV_CH_0, R_SDG_CLOCK_4 },
    {   DEVDRV_CH_1, R_SDG_CLOCK_4 },
    {   DEVDRV_CH_2, R_SDG_CLOCK_4 },
    {   DEVDRV_CH_3, R_SDG_CLOCK_4 },
};


/* dummy entry of score (no sounds)*/
static const int8_t noscore[] = "";

/* parameters of a rest */
static const NOTE   rest4 = {
    0,  /* tone         */
    0,  /* sfs          */
    0,  /* loud         */
    0,  /* attenuation  */
};


/* table of notes */
static const struct {
    uint8_t     tone;
    uint8_t     sfs;
} note_timing[MAXOCTAVE][MAXNOTE] = {
    {   /* octave 0 */
        {   0,   0 },  /* -A  */
        {   0,   0 },  /* -A# */
        {   0,   0 },  /* -B  */
        { 124, 255 },  /* -C  */
        { 117, 255 },  /* -C# */
        { 111, 255 },  /* -D  */
        { 105, 255 },  /* -D# */
        {  99, 255 },  /* -E  */
        {  93, 255 },  /* -F  */
        {  88, 255 },  /* -F# */
        {  83, 255 },  /* -G  */
        {  78, 255 },  /* -G# */
        {   0,   0 },  /* -R  */
    },
    {   /* octave 1 */
        {  86, 220 },  /*  A  */
        {  81, 220 },  /*  A# */
        {  76, 220 },  /*  B  */
        {  72, 220 },  /*  C  */
        {  68, 220 },  /*  C# */
        {  64, 220 },  /*  D  */
        {  60, 220 },  /*  D# */
        {  57, 220 },  /*  E  */
        {  54, 220 },  /*  F  */
        {  51, 220 },  /*  F# */
        {  48, 220 },  /*  G  */
        {  45, 220 },  /*  G# */
        {   0,   0 },  /*  R  */
    },
    {   /* octave 2 */
        {  49, 190 },  /* +A  */
        {  47, 190 },  /* +A# */
        {  44, 190 },  /* +B  */
        {  41, 190 },  /* +C  */
        {  39, 190 },  /* +C# */
        {  37, 190 },  /* +D  */
        {  35, 190 },  /* +D# */
        {  33, 190 },  /* +E  */
        {  31, 190 },  /* +F  */
        {  29, 190 },  /* +F# */
        {  27, 190 },  /* +G  */
        {  26, 190 },  /* +G# */
        {   0,   0 },  /* +R  */
    },
    {   /* octave 3 */
        {  31, 150 },  /* *A  */
        {  29, 150 },  /* *A# */
        {  28, 150 },  /* *B  */
        {  26, 150 },  /* *C  */
        {  25, 150 },  /* *C# */
        {  23, 150 },  /* *D  */
        {  22, 150 },  /* *D# */
        {  21, 150 },  /* *E  */
        {  19, 150 },  /* *F  */
        {  18, 150 },  /* *F# */
        {  17, 150 },  /* *G  */
        {  16, 150 },  /* *G# */
        {   0,   0 },  /* *R  */
    },
};


/* name table of notes */
#ifdef UNIT_MSG
static const char *note_name[MAXNOTE] = {
    "A",
    "A#",
    "B",
    "C",
    "C#",
    "D",
    "D#",
    "E",
    "F",
    "F#",
    "G",
    "G#",
    "R",
};
#endif /*UNIT_MSG*/


/******************************************************************************
* Function Name: piano4int_start
* Description  : Play the score thru sound generator.
*              : This function receives the sound language,
*              : and plays specified sound language.
* Arguments    : const int8_t *scores  : score
*              : PIANO4INT_CB callback : callback function
* Return Value : none
******************************************************************************/
void piano4int_start(const int8_t *scores[SDG_CH_TOTAL], PIANO4INT_CB callback)
{
    const int8_t    *score;     /* pointer to score */
    uint16_t        speed;      /* sound speed */
    int32_t         ch;         /* channel no */
    int32_t         ret;        /* function result */

    /* prepare for playing the score */
    for (ch = 0; ch < SDG_CH_TOTAL; ch++) {
        if (scores[ch] == NULL) {
            /* score is not specified */
            pianoT.unit[ch].score = noscore;
            pianoT.unit[ch].top   = noscore;
        } else {
            /* prepare for playing a score */
            score = scores[ch];

            /* get the speed */
            speed = 0;
            while (isdigit(*score)) {
                speed = (speed * 10) + ((*score++) - '0');
            }
            if (0 == speed) {
                /* can not play the score */
                printf("speed[%d]:%d, can not play, there is no speed information.\n", (int)ch, speed);
                return;
            } else {
                printf("speed[%d]:%d\n", (int)ch, speed);
                /* calculate a base speed for ticker */
                pianoT.unit[ch].speed = BASESPEED / speed;

                /* default loudness (unsupported to change) */
                pianoT.unit[ch].loudness = DEFLOUDNESS;

                /* set top of a score */
                pianoT.unit[ch].top   = score;
                pianoT.unit[ch].score = score;

                /* play no sounds */
                pianoT.unit[ch].count = 0;

                /* rest4 */
                pianoT.unit[ch].note = rest4;

                /* channel number */
                pianoT.unit[ch].channel = unitHW[ch].channel;

#ifdef  UNIT_MSG
                /* message */
                pianoT.unit[ch].msg[0] = '\0';
#endif

                /* Initialize Sound Generator */
                ret = R_SDG_Open(unitHW[ch].channel, unitHW[ch].clock);
                ASSERT(0 == ret);
            }
        }
    }

    /* set callback function */
    pianoT.callback = callback;

    /* clear flags to finish playing */
    pianoT.fin = 0;

    /* start the metronome  (unit: 1ms) */
    pianoT.metronome.attach_us(&play_score, 1000.0);
}

/******************************************************************************
* Function Name: piano4int_stop
* Description  : Stop the sounds through sound generator.
* Arguments    : bool force : flag to stop forcedly (now, ignored)
* Return Value : none
******************************************************************************/
void piano4int_stop(bool force)
{
    int32_t         ch;
    int32_t         ret;

    /* wait until sounds stop */
    do {
        /* wait 10.0 sec */
        wait_us(10.0 * 1000.0);
#ifdef UNIT_MSG
        for (ch = 0; ch < SDG_CH_TOTAL; ch++) {
            if ('\0' != pianoT.unit[ch].msg[0]) {
                printf(&pianoT.unit[ch].msg[0]);
                pianoT.unit[ch].msg[0] = '\0';
            }
        }
#endif /*UNIT_MSG*/
    } while (pianoT.fin != ((1 << SDG_CH_TOTAL) - 1));
    /* all scores finished */

    /* stops the metronome */
    pianoT.metronome.detach();

    /* stop all sound generators */
    for (ch = 0; ch < SDG_CH_TOTAL; ch++) {
        if (pianoT.unit[ch].top != noscore) {
            /* stop sound generator */
            ret = R_SDG_Close(unitHW[ch].channel);
            ASSERT(0 == ret);
        }
    }
}

/******************************************************************************
* Function Name: piano4int_status
* Description  : Get the flag to finish playing the score
* Arguments    : none
* Return Value : The flag
* Note         : Output the comment if the macro "UNIT_MSG" is defined
******************************************************************************/
bool piano4int_status(void)
{
#ifdef UNIT_MSG
    int32_t ch;

    /* output the comments from interrupts */
    for (ch = 0; ch < SDG_CH_TOTAL; ch++) {
        if ('\0' != pianoT.unit[ch].msg[0]) {
            printf(&pianoT.unit[ch].msg[0]);
            pianoT.unit[ch].msg[0] = '\0';
        }
    }
#endif

    return (pianoT.fin != ((1 << SDG_CH_TOTAL) - 1));
}


/******************************************************************************
* Function Name: play_score
* Description  : Play the score on interrupt
* Arguments    : none
* Return Value : none
******************************************************************************/
void play_score(void)
{
    UNIT            *unit;
    const int8_t    *score;
    uint16_t        octave;
    uint16_t        key;
    uint16_t        pitch;
    uint16_t        dot;
    uint16_t        tie;
    uint8_t         loud;
    uint16_t        duration;
    uint16_t        value;
    uint16_t        ch;
    bool            err;

    for (ch = 0; ch < SDG_CH_TOTAL; ch++) {
        /* Check finish flag */
        if (0 != (pianoT.fin & (1 << ch))) {
            continue;   /* finish */
        }

        if (0 == pianoT.unit[ch].count) {
            unit    = &pianoT.unit[ch];
            score   = unit->score;

            if ('\0' == *score) {
                /* set finish flag */
                pianoT.fin |= (1 << ch);
                continue;
            }

            /* Initialize parameters */
            octave  = 0;
            key     = 0;
            pitch   = 0;
            dot     = 0;
            tie     = 0;
            err     = false;

            /* skip some spaces */
            while (' ' == *score) {
                score++;
            }

            /* octave */
            switch (*score) {
                case OCTAVE0_CHAR:  { octave = OCTAVE_LEVEL_0; score++; break; }
                case OCTAVE1_CHAR:  { octave = OCTAVE_LEVEL_1; score++; break; }
                case OCTAVE2_CHAR:  { octave = OCTAVE_LEVEL_2; score++; break; }
                case OCTAVE3_CHAR:  { octave = OCTAVE_LEVEL_3; score++; break; }
                default:            { octave = OCTAVE_LEVEL_1;          break; }
            }

            /* key */
            switch (toupper(*score)) {
                case NOTE_DO_CHAR:  { key = KEY_C; score++; break; }
                case NOTE_RE_CHAR:  { key = KEY_D; score++; break; }
                case NOTE_MI_CHAR:  { key = KEY_E; score++; break; }
                case NOTE_FA_CHAR:  { key = KEY_F; score++; break; }
                case NOTE_SO_CHAR:  { key = KEY_G; score++; break; }
                case NOTE_RA_CHAR:  { key = KEY_A; score++; break; }
                case NOTE_SI_CHAR:  { key = KEY_B; score++; break; }
                case NOTE_REST_CHAR:{ key = KEY_R; score++; break; }
                default:
                    err = true;
            }

            if (((KEY_A == key) && (OCTAVE_LEVEL_0 == octave)) ||
                ((KEY_B == key) && (OCTAVE_LEVEL_0 == octave))) {
                err = true;
            }

            /* sharp */
            if (NOTE_SHARP == (*score)) {
                if (KEY_R == key) {
                    err = true;
                } else {
                    key++;
                    score++;
                }
            }

            /* pitch */
            if (isdigit(*score)) {
                pitch = ((*score++) - '0');
                if (isdigit(*score)) {
                    pitch = (pitch * 10) + ((*score++) - '0');
                }
            } else {
                err = true;
            }

            /* dotted */
            if (NOTE_DOT == (*score)) {
                dot = DOTTED_NOTE;
                score++;
            } else {
                dot = NOT_DOTTED_NONE;
            }

            /* tie */
            if (NOTE_TIE == (*score)) {
                tie = TIED_NOTE;
                score++;
            } else {
                tie = NOT_TIED_NONE;
            }

            /* set score */
            unit->score = score;

#ifdef  UNIT_MSG
            snprintf(&unit->msg[0], UNIT_MSG - 1, "%d)%d%s%d -> %d/%d/%d\n",
                unit->channel,
                octave,
                note_name[key],
                pitch,
                note_timing[octave][key].tone,
                note_timing[octave][key].sfs,
                unit->speed / pitch);
#endif /* UNIT_MSG */

            if (!err) {
                /* set the sound volume */
                loud     = (0 == note_timing[octave][key].sfs) ? (0): (unit->loudness);
                /* set */
                duration = unit->speed / pitch;
                if (dot) {
                    duration += (duration >> 1);
                }

                /* set next note or rest */
                unit->note.tone         = note_timing[octave][key].tone;
                unit->note.sfs          = note_timing[octave][key].sfs;
                unit->note.loud         = loud;
                unit->note.attenuation  = tie;

                R_SDG_Tone(pianoT.unit[ch].channel, &pianoT.unit[ch].note);
                pianoT.unit[ch].count = duration;
            } else {
                pianoT.fin |= (1 << ch);
            }

            /* marker */
            if (NOTE_MARKER == (*score)) {
                score++;
                value = 0;
                if (isdigit(*score)) {
                    value = ((*score++) - '0');
                    if (isdigit(*score)) {
                        value = (value * 10) + ((*score++) - '0');
                    }
                }
                unit->score = score;

                if (NULL != pianoT.callback) {
                    pianoT.callback(pianoT.unit[ch].channel, value);
                }
            }
        } else {
            pianoT.unit[ch].count--;
        }
    }
}

/* End of File */
