/*******************************************************************************
* DISCLAIMER
* This software is supplied by Renesas Electronics Corporation and is only
* intended for use with Renesas products. No other uses are authorized. This
* software is owned by Renesas Electronics Corporation and is protected under
* all applicable laws, including copyright laws.
* THIS SOFTWARE IS PROVIDED "AS IS" AND RENESAS MAKES NO WARRANTIES REGARDING
* THIS SOFTWARE, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING BUT NOT
* LIMITED TO WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
* AND NON-INFRINGEMENT. ALL SUCH WARRANTIES ARE EXPRESSLY DISCLAIMED.
* TO THE MAXIMUM EXTENT PERMITTED NOT PROHIBITED BY LAW, NEITHER RENESAS
* ELECTRONICS CORPORATION NOR ANY OF ITS AFFILIATED COMPANIES SHALL BE LIABLE
* FOR ANY DIRECT, INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES FOR
* ANY REASON RELATED TO THIS SOFTWARE, EVEN IF RENESAS OR ITS AFFILIATES HAVE
* BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
* Renesas reserves the right, without notice, to make changes to this software
* and to discontinue the availability of this software. By using this software,
* you agree to the additional terms and conditions found by accessing the
* following link:
* http://www.renesas.com/disclaimer
* Copyright (C) 2012 - 2015 Renesas Electronics Corporation. All rights reserved.
*******************************************************************************/

/******************************************************************************
Includes   <System Includes> , "Project Includes"
******************************************************************************/
#include <stdint.h>
#include "dev_drv.h"                    /* Device Driver common header */
#include "devdrv_sdg.h"                 /* SDG Driver header */
#include "iodefine.h"
#include "gpio_iobitmask.h"
#include "sdg_iobitmask.h"
#include "rza_io_regrw.h"

/******************************************************************************
Typedef definitions
******************************************************************************/
typedef int32_t (*Userdef_Open)(void);
typedef int32_t (*Userdef_Close)(void);

/******************************************************************************
Macro definitions
******************************************************************************/
#define REGW8(CH, REG, BIT, VAL) RZA_IO_RegWrite_8(&CH->REG, VAL, SDGn_##REG##_##BIT##_SHIFT, SDGn_##REG##_##BIT)

/******************************************************************************
Imported global variables and functions (from other files)
******************************************************************************/


/******************************************************************************
Exported global variables and functions (to be accessed by other files)
******************************************************************************/


/******************************************************************************
Private global variables and functions
******************************************************************************/
static struct st_sdg * const stSDG[SDG_CH_TOTAL] = {
    &SDG0,
    &SDG1,
    &SDG2,
    &SDG3,
};


/******************************************************************************
* Function Name: R_SDG_Open
* Description  : Start the SDG specified by the argument channel.
* Arguments    : uint32_t channel  : SDG channel (0, 1, 2, or 3)
*              : R_SDG_CLOCK clock : SGCLK
* Return Value : DEVDRV_SUCCESS    : Success to start counting SDG
*              : DEVDRV_ERROR      : Failure
******************************************************************************/
int32_t R_SDG_Open(uint32_t channel, R_SDG_CLOCK clock)
{
    static const Userdef_Open   userdef_open[SDG_CH_TOTAL] = {
        &Userdef_SDG0_Open,
        &Userdef_SDG1_Open,
        &Userdef_SDG2_Open,
        &Userdef_SDG3_Open,
    };
    int32_t ret = DEVDRV_SUCCESS;


    /* check argument */
    if (channel >= SDG_CH_TOTAL) {
        return DEVDRV_ERROR;        /* argument error */
    }

    /* ==== SDG initialization ==== */
    ret = userdef_open[channel]();
    if (DEVDRV_SUCCESS == ret) {
        /* ==== Start SDG ==== */
        REGW8(stSDG[channel], SGCR1, STPM, 1);                /* set SDG deactivate method: none */
        REGW8(stSDG[channel], SGCR1, SGCK, (uint8_t)clock);   /* clock */
        REGW8(stSDG[channel], SGCSR, SGIE, 0);                /* interrupt: disabled */
    }

    return ret;
}

/******************************************************************************
* Function Name: R_SDG_Close
* Description  : Stop the SDG specified by the argument channel.
* Arguments    : uint32_t channel : SDG channel (0, 1, 2 or 3)
* Return Value : DEVDRV_SUCCESS   : Success to stop counting SDG
*              : DEVDRV_ERROR     : Failure
******************************************************************************/
int32_t R_SDG_Close(uint32_t channel)
{
    static const Userdef_Close    userdef_close[SDG_CH_TOTAL] = {
        &Userdef_SDG0_Close,
        &Userdef_SDG1_Close,
        &Userdef_SDG2_Close,
        &Userdef_SDG3_Close,
    };
    int32_t ret = DEVDRV_SUCCESS;

    /* check argument */
    if (channel >= SDG_CH_TOTAL) {
        return DEVDRV_ERROR;        /* Argument error */
    }

    /* ==== SDG finalization ==== */
    ret = userdef_close[channel]();

    return ret;
}


/******************************************************************************
* Function Name: R_SDG_Tone
* Description  : Set freqency registers
* Arguments    : uint32_t channel : SDG channel (0, 1, 2 or 3)
*              : NOTE *note       : a note parameters
* Return Value : DEVDRV_SUCCESS   : Success to play a note
*              : DEVDRV_ERROR     : Failure
******************************************************************************/
int32_t R_SDG_Tone(uint32_t channel, NOTE *note)
{
    uint16_t    tone;
    uint16_t    sfs;
    uint16_t    loud;
    uint16_t    att;
    int32_t     ret = DEVDRV_SUCCESS;


    /* check parameter */
    if ((channel >= SDG_CH_TOTAL) || (note->tone > SDG_TONE_MAX)) {
        return DEVDRV_ERROR;        /* Argument error */
    }

    tone = (0 == note->tone) ? (1): (note->tone);
    sfs  = (0 == note->sfs ) ? (1): (note->sfs);
    loud = note->loud;
    att  = note->attenuation;

    /* ==== Set frequency registers ==== */
    /* start sgd */
    REGW8(stSDG[channel], SGCR1, SGST,  1);         /* start sgd */
    REGW8(stSDG[channel], SGCR2, SGEND, 0);         /* stop bit : continue */
    /* enable SGTFR / SGSFR register to write */
    REGW8(stSDG[channel], SGCR2, TCHG,  1);         /* change : enable */
    /* set each parameter */
    REGW8(stSDG[channel], SGTFR, TONE,  tone);      /* tone */
    REGW8(stSDG[channel], SGSFR, SFS,   sfs);       /* sfs */
    REGW8(stSDG[channel], SGLR,  LD,    loud);      /* loud */
    REGW8(stSDG[channel], SGCR1, DPF,   att);       /* attenuation */
    if (note->attenuation != 0) {
        REGW8(stSDG[channel], SGCR1, SGST, 0);      /* stop sgd */
    }

    return ret;
}

/* End of File */
