/*
 * SecureDweet.h
 *
 *  Created on: Aug 15, 2016
 *      Author: Faheem Inayat
 * Created for: Renesas Electronics America HQ, Santa Clara, CA, USA
 *
 * Copyright (c) 2016 Renesas Electronics America (REA) and Faheem Inayat
 */
/*
 * MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
 /**
    Example
    
    @code

    #include "mbed.h"
    #include "rtos.h"
    #include "EthernetInterface.h"
    #include "string.h"
    #include "GRPeachBoard.h"
    #include "SecureDweet.h"

    #define THING_NAME  "_YOUR_THING_NAME_HERE_"
    #define WRITE_KEY   "_OPTIONAL_WRITE_KEY_IF_THING_IS_LOCKED_"
    #define READ_KEY    "_OPTIONAL_READ_KEY_IF_THING_IS_LOCKED_"

    void printAndFlush(void const * msg)
    {
        fprintf ( stdout, "%s", (char *) msg);
        fflush ( stdout);
    }
    
    bool setupEthernet(EthernetInterface * & ethernet)
    {
        printAndFlush ("\n\rConnecting Ethernet ...");
    
        RtosTimer * connectProgressTimer = new RtosTimer (printAndFlush, osTimerPeriodic, (void *) ".");
        connectProgressTimer->start (500);
    
        int err = ethernet->init ();
        char errMsg[256];
        if (err == 0)
        {
            err = ethernet->connect ();
            if (err == 0)
            {
                sprintf (
                        errMsg,
                        "\n\rEthernet connected with specs:\n\r\tIP Address:  %s\n\r\tSubnet Mask: %s\n\r\tGateway:     %s\n\r\tMAC Address: %s\n\r",
                        ethernet->getIPAddress (), ethernet->getNetworkMask (), ethernet->getGateway (),
                        ethernet->getMACAddress ());
            }
            else
            {
                sprintf (errMsg, " Ethernet connect failed with error: %d\n\r", err);
            }
        }
        else
        {
            sprintf (errMsg, " Ethernet init failed with error: %d\n\r", err);
        }
    
        printAndFlush (errMsg);
        connectProgressTimer->stop ();
        delete connectProgressTimer;
    
        return (err == 0);
    }
    
    int main(void)
    {
        EthernetInterface * ethernet = new EthernetInterface ();
        IDweetParser * device = new GRPeachBoard (); // Implementation of IDweetParser
    
        SecureDweet * dweetListener = new SecureDweet (device, THING_NAME, READ_KEY);
    
        while (true)
        {
            setupEthernet (ethernet);
    
            dweetListener -> fetchLatestDweet ();
            dweetListener -> listenToDweet(); // this is an infinite blocking function call
    
            // In another thread, use the following function call to stop listening
            // to dweets (exit from the above function call)
            // dweetListener -> stopListentingToDweet();
        }
    }

    @endcode
 */

#ifndef __SECUREDWEET_H_
    #define __SECUREDWEET_H_

#include "mbed.h"
#include "stdio.h"
#include "HttpText.h"
#include "IDweetParser.h"
#include "SecureHttpClient.h"

class SecureDweet : public ISecureHttpClientChunkDataListener
{
    private:
        IDweetParser * device;

        const char * thingName;
        const char * readKey;
        const char * writeKey;

        char * latestUrl;
        char * listenUrl;

        SecureHttpClient * httpClient;
        char * httpRxBuffer;
        HttpText * httpText;

        bool listeningFlag;

        SecureDweet(const SecureDweet & other);

    public:
        SecureDweet ( IDweetParser * _device, const char * _thingName, const char * _readKey = NULL, const char * _writeKey = NULL );
        virtual ~SecureDweet ();

        virtual void chunkRead ( IHttpDataIn * pDataIn );

        void fetchLatestDweet();
        void listenToDweet();
        void stopListentingToDweet ();
};

#endif /* __SECUREDWEET_H_ */

