/*
 * SecureHttpClient.h
 *
 *  Created on: Aug 15, 2016
 *      Author: Faheem Inayat
 * Created for: Renesas Electronics America HQ, Santa Clara, CA, USA
 * 
 * Copyright (c) 2016 Renesas Electronics America (REA) and Faheem Inayat
 */
/*
 * MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef SECUREDWEET_SECUREHTTPCLIENT_H_
    #define SECUREDWEET_SECUREHTTPCLIENT_H_

    #define HTTP_CLIENT_DEFAULT_TIMEOUT 300000

#include "IHttpDataIn.h"
#include "IHttpDataOut.h"
#include "ISecureHttpClientChunkDataListener.h"

#include "mbed.h"

///HTTP client results
enum HttpResult
{
    HTTP_PROCESSING, ///<Processing
    HTTP_PARSE, ///<url Parse error
    HTTP_DNS, ///<Could not resolve name
    HTTP_PRTCL, ///<Protocol error
    HTTP_NOTFOUND, ///<HTTP 404 Error
    HTTP_REFUSED, ///<HTTP 403 Error
    HTTP_ERROR, ///<HTTP xxx error
    HTTP_TIMEOUT, ///<Connection timeout
    HTTP_CONN, ///<Connection error
    HTTP_CLOSED, ///<Connection was closed by remote host
    HTTP_REDIRECT, ///<HTTP 300 - 303
    HTTP_OK = 0, ///<Success
};

/**
 * HTTPS Client based on mbed and wolfSSL HTTP Client
 */
class SecureHttpClient
{
    private:
        enum HTTP_METH
        {
            HTTP_GET, HTTP_POST, HTTP_PUT, HTTP_DELETE, HTTP_HEAD
        };

        HttpResult connect ( const char* url, HTTP_METH method, IHttpDataOut* pDataOut, IHttpDataIn* pDataIn,
                             int timeout ); //Execute request
        HttpResult recv ( char* buf, size_t minLen, size_t maxLen, size_t* pReadLen ); //0 on success, err code on failure
        HttpResult send ( const char* buf, size_t len = 0 ); //0 on success, err code on failure
        HttpResult flush ( void ); //0 on success, err code on failure
        HttpResult parseURL ( const char* url, const char* scheme, char* host, size_t maxHostLen,
                              uint16_t* port, char* path, size_t maxPathLen ); //Parse URL

        void wolfssl_free ( void );
        HttpResult bAuth ( void );

        //Parameters

        int m_timeout;

        const char * m_basicAuthUser;
        const char * m_basicAuthPassword;
        int m_httpResponseCode;

        const char * header;
        char * redirect_url;
        int redirect_url_size;
        int redirect;

        /* for wolfSSL */
        int SSLver;
        uint16_t port;
        struct WOLFSSL_CTX* ctx;
        struct WOLFSSL * ssl;

        ISecureHttpClientChunkDataListener * m_chunkDataListener;

    public:
        SecureHttpClient ();
        virtual ~SecureHttpClient ();

        /**
         Provides a basic authentification feature (Base64 encoded username and password)
         Pass two NULL pointers to switch back to no authentication
         @param user username to use for authentication, must remain valid durlng the whole HTTP session
         @param user password to use for authentication, must remain valid durlng the whole HTTP session
         */
        HttpResult basicAuth ( const char* user, const char* password ); //Basic Authentification

        //High Level setup functions
        /** Execute a GET request on the URL
         Blocks until completion
         @param url : url on which to execute the request
         @param pDataIn : pointer to an IHttpDataIn instance that will collect the data returned by the request, can be NULL
         @param timeout waiting timeout in ms (osWaitForever for blocking function, not recommended)
         @return 0 on success, HTTP error (<0) on failure
         */
        HttpResult get ( const char* url, IHttpDataIn* pDataIn, int timeout = HTTP_CLIENT_DEFAULT_TIMEOUT ); //Blocking

        /** Execute a POST request on the URL
         Blocks until completion
         @param url : url on which to execute the request
         @param dataOut : a IHttpDataOut instance that contains the data that will be posted
         @param pDataIn : pointer to an IHttpDataIn instance that will collect the data returned by the request, can be NULL
         @param timeout waiting timeout in ms (osWaitForever for blocking function, not recommended)
         @return 0 on success, HTTP error (<0) on failure
         */
        HttpResult post ( const char* url, const IHttpDataOut& dataOut, IHttpDataIn* pDataIn, int timeout =
        HTTP_CLIENT_DEFAULT_TIMEOUT ); //Blocking

        /** Execute a PUT request on the URL
         Blocks until completion
         @param url : url on which to execute the request
         @param dataOut : a IHttpDataOut instance that contains the data that will be put
         @param pDataIn : pointer to an IHttpDataIn instance that will collect the data returned by the request, can be NULL
         @param timeout waiting timeout in ms (osWaitForever for blocking function, not recommended)
         @return 0 on success, HTTP error (<0) on failure
         */
        HttpResult put ( const char* url, const IHttpDataOut& dataOut, IHttpDataIn* pDataIn, int timeout =
        HTTP_CLIENT_DEFAULT_TIMEOUT ); //Blocking

        /** Execute a DELETE request on the URL
         Blocks until completion
         @param url : url on which to execute the request
         @param pDataIn : pointer to an IHttpDataIn instance that will collect the data returned by the request, can be NULL
         @param timeout waiting timeout in ms (osWaitForever for blocking function, not recommended)
         @return 0 on success, HTTP error (<0) on failure
         */
        HttpResult del ( const char* url, IHttpDataIn* pDataIn, int timeout = HTTP_CLIENT_DEFAULT_TIMEOUT ); //Blocking

        /** Get last request's HTTP response code
         @return The HTTP response code of the last request
         */
        int getHTTPResponseCode ();

        void setHeader ( const char *header ); /* set http headers */
        HttpResult setSslVersion ( int minorV ); /* set SSL/TLS version. 0: SSL3, 1: TLS1.0, 2: TLS1.1, 3: TLS1.2 */
        void setRedirectUrlBuffer ( char *url, int size ); /* set URL buffer for redirection */
        void setChunkDataListener ( ISecureHttpClientChunkDataListener * listener );
};

#endif /* SECUREDWEET_SECUREHTTPCLIENT_H_ */

