/*
 * SecureDweet.h
 *
 *  Created on: Aug 15, 2016
 *      Author: Faheem Inayat
 * Created for: Renesas Electronics America HQ, Santa Clara, CA, USA
 *
 * Copyright (c) 2016 Renesas Electronics America (REA) and Faheem Inayat
 */
/*
 * MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "SecureDweet.h"
#include "Json.h"

#if 0
//Enable debug
    #include <cstdio>
    #define logInfo(x, ...) do{std::fprintf(stdout,"%s:%s:%s| [SecureDweet:INFO] "x"\r\n", __FILE__, __func__, __LINE__,##__VA_ARGS__);std::fflush(stdout);}while(0)
    #define logWarning(x, ...) do{std::fprintf(stdout,"%s:%s:%s| [SecureDweet:WARN] "x"\r\n", __FILE__, __func__, __LINE__,##__VA_ARGS__);std::fflush(stdout);}while(0)
    #define logError(x, ...) do{std::fprintf(stdout,"%s:%s:%s| [SecureDweet:ERROR] "x"\r\n", __FILE__, __func__, __LINE__,##__VA_ARGS__);std::fflush(stdout);}while(0)
#else
//Disable debug
    #define logInfo(x, ...)
#endif

#define logWarning(x, ...)
#define logError(x, ...)

#define HTTP_BUFFER_SIZE    1024
#define HTTP_CLIENT_DEFAULT_TIMEOUT 300000

#define DWEET_URL_GET_LATEST "https://dweet.io/get/latest/dweet/for/"
#define DWEET_URL_GET_LISTEN_TO "https://dweet.io/listen/for/dweets/from/"
#define DWEET_KEY_PARAM_NAME "?key="

const char KEEP_ALIVE_HEADER [] = "Connection: Keep-Alive\r\nKeep-Alive: timeout=300, max=50\r\n";

SecureDweet::SecureDweet ( IDweetParser * _device, const char * _thingName, const char * _readKey,
                           const char * _writeKey )
        : device ( _device ), thingName ( _thingName ), readKey ( _readKey ), writeKey ( _writeKey )
{
    size_t thingNameLength = strlen ( thingName );
    size_t keyLength = 0;

    if ( writeKey != NULL )
    {
        keyLength = ( strlen ( writeKey ) + strlen ( DWEET_KEY_PARAM_NAME ) );
    }
    else if ( readKey != NULL )
    {
        keyLength = ( strlen ( readKey ) + strlen ( DWEET_KEY_PARAM_NAME ) );
    }

    {
        size_t lastestUrlLength = strlen ( DWEET_URL_GET_LATEST ) + thingNameLength + keyLength;
        latestUrl = new char [ lastestUrlLength + 1 ];
        int written = sprintf ( latestUrl, "%s%s", DWEET_URL_GET_LATEST, thingName );
        if ( writeKey != NULL )
        {
            sprintf ( latestUrl + written, "%s%s", DWEET_KEY_PARAM_NAME, writeKey );
        }
        else if ( readKey != NULL )
        {
            sprintf ( latestUrl + written, "%s%s", DWEET_KEY_PARAM_NAME, readKey );
        }
    }

    {
        size_t listenUrlLength = strlen ( DWEET_URL_GET_LISTEN_TO ) + thingNameLength + keyLength;
        listenUrl = new char [ listenUrlLength + 1 ];
        int written = sprintf ( listenUrl, "%s%s", DWEET_URL_GET_LISTEN_TO, thingName );
        if ( writeKey != NULL )
        {
            sprintf ( listenUrl + written, "%s%s", DWEET_KEY_PARAM_NAME, writeKey );
        }
        else if ( readKey != NULL )
        {
            sprintf ( listenUrl + written, "%s%s", DWEET_KEY_PARAM_NAME, readKey );
        }
    }

    httpClient = new SecureHttpClient ();
    httpClient->setChunkDataListener ( this );
    httpRxBuffer = new char[ HTTP_BUFFER_SIZE + 1];
    httpText = new HttpText ( httpRxBuffer, HTTP_BUFFER_SIZE );

//    httpClient -> setPeerVerification ( VERIFY_PEER );
//    // Explicitly set the ROOT certificates for dweet.io
//    if ( httpClient -> addRootCACertificate ( sslCertificates ) != HTTP_OK ){
//        logError ( "loading SSL certificates failed" );
//    }

    listeningFlag = false;

}

SecureDweet::SecureDweet ( const SecureDweet & )
        : device ( NULL ), thingName ( NULL ), readKey ( NULL ), writeKey ( NULL )
{
    latestUrl = NULL;
    listenUrl = NULL;
    httpClient = NULL;
    httpRxBuffer = NULL;
    httpText = NULL;
    listeningFlag = false;
}

SecureDweet::~SecureDweet ()
{
    delete [] latestUrl;
    delete [] listenUrl;
    delete httpClient;
    delete[] httpRxBuffer;
    delete httpText;
}

void SecureDweet::fetchLatestDweet ()
{
    httpClient->setHeader ( NULL );
    HttpResult res = httpClient->get ( latestUrl, httpText, HTTP_CLIENT_DEFAULT_TIMEOUT );

    if ( res != HTTP_OK )
    {
        logError ( "HTTP GET failed [%d][%s]", res, httpRxBuffer );
    }
    else
    {
        logInfo ( "Got response:\r\n%s", httpRxBuffer );
        device->parseJsonData ( httpRxBuffer );
    }
}

void SecureDweet::chunkRead ( IHttpDataIn * pDataIn )
{
    if ( pDataIn != NULL ) {
        int chunkDataSize = 0;
        int valuesExtracted = sscanf ( httpRxBuffer, "%x", &chunkDataSize );
        if ( valuesExtracted == 1 ) // Reading exactly one number
        {
            // Max Chunk Size is 32-bit (4 Hex Digits).  Parse it!
            char hexNumber [ 7 ];
            int hexNumberLength = sprintf ( hexNumber, "%X\n\r", chunkDataSize );
            char * lastCr = strrchr ( httpRxBuffer + hexNumberLength, '\n' );
            if ( lastCr != NULL )
            {
                int bufferLength = (int) ( lastCr - ( httpRxBuffer + hexNumberLength ) ) - 1;

                if ( bufferLength == chunkDataSize ) // There's a possibility that the chunk is complete now.  Try to pass it
                {
                    Json::unescape (httpRxBuffer + hexNumberLength);
                    logInfo("Got chunked response:\r\n%s", httpRxBuffer + hexNumberLength );
                    device->parseJsonData (httpRxBuffer + hexNumberLength );

                    pDataIn -> writeReset();
                }
                else if ( bufferLength > chunkDataSize ) // Something is messed up, just clean up the buffer
                {
                    pDataIn -> writeReset();
                }
            }
        }
    }
}

void SecureDweet::listenToDweet()
{
    listeningFlag = true;
    while ( listeningFlag == true )
    {
        httpClient -> setHeader(KEEP_ALIVE_HEADER);
        httpClient -> get (listenUrl, httpText, HTTP_CLIENT_DEFAULT_TIMEOUT);
        httpClient -> setHeader(NULL);
    }
}

void SecureDweet::stopListentingToDweet ()
{
    listeningFlag = false;
}


