#include "mbed.h"
#include "USBHostMouse.h"
extern "C" {
#include "devdrv_sdg.h"
}
/* defines */
#define SDG_OUTPUT_CH         (1)
#define DEFAULT_LOUDNESS      (4)
#define DIV_MAX               (100)
#define DIV_DEFAULT           (75)
#define DIV_MIN               (50)
#define MODE_TONE             (2)

#define MOUSE_BUTTON_LEFGT    (0x01)
#define MOUSE_BUTTON_RIGHT    (0x02)
#define MOUSE_BUTTON_CENTER   (0x04)

#define OCTAVE_MAX            (4)
#define OCTAVE_LEV_MAX        (OCTAVE_MAX - 1)
#define OCTAVE_LEV_MIN        (0)
static const struct {
    uint8_t     tone_max;
    uint8_t     tone_min;
    uint8_t     sfs;
} octave_table[OCTAVE_MAX] = {
    { 31, 16, 150},  /* octave 3(*) */
    { 50, 24, 190},  /* octave 2(+) */
    { 86, 43, 220},  /* octave 1() */
    {124, 73, 255},  /* octave 0(-) */
};

/*  RAMS   */
DigitalOut led(LED1);
static NOTE output_note;
static int32_t work_tone;
static int32_t work_octave_lev;
static int32_t work_div;


void onMouseEvent(uint8_t buttons, int8_t x, int8_t y, int8_t z) {
    
    //update div
    work_div += z;
    if (work_div < DIV_MIN) {
        work_div = DIV_MIN;
    } else if (work_div > DIV_MAX) {
        work_div = DIV_MAX;
    }
    if ((buttons & MOUSE_BUTTON_CENTER) == MOUSE_BUTTON_CENTER) {
        work_div = DIV_DEFAULT;
    }
    
    //update tone
    x = x * (-1);
    work_tone += x;
    if (work_tone < (octave_table[work_octave_lev].tone_min * work_div * MODE_TONE)) {
        if (work_octave_lev != OCTAVE_LEV_MIN) {
            work_octave_lev--;
            work_tone = octave_table[work_octave_lev].tone_max * work_div * MODE_TONE;
        } else {
            work_tone = octave_table[work_octave_lev].tone_min * work_div * MODE_TONE;
        }
    } else if (work_tone > (octave_table[work_octave_lev].tone_max * work_div * MODE_TONE)) {
        if (work_octave_lev != OCTAVE_LEV_MAX) {
            work_octave_lev++;
            work_tone = octave_table[work_octave_lev].tone_min * work_div * MODE_TONE;
        } else {
            work_tone = octave_table[work_octave_lev].tone_max * work_div * MODE_TONE;
        }
    }

    
    if ((buttons & MOUSE_BUTTON_RIGHT) == MOUSE_BUTTON_RIGHT) {
        work_tone = 72 * work_div * MODE_TONE;
        work_octave_lev = 2;
    }
    output_note.tone = work_tone / (work_div * MODE_TONE);
    output_note.sfs  = octave_table[work_octave_lev].sfs;
    
    printf("sfs: %d, tone: %d, div: %d\r\n", output_note.sfs, output_note.tone, work_div);
    if ((buttons & MOUSE_BUTTON_LEFGT) == MOUSE_BUTTON_LEFGT) {
        // mouse left button clicked
        output_note.loud = DEFAULT_LOUDNESS;
        R_SDG_Tone(SDG_OUTPUT_CH, &output_note);
    } else {
        // mouse left button released
        output_note.loud = 0;
        R_SDG_Tone(SDG_OUTPUT_CH, &output_note);
    }
}

void mouse_task(void const *) {
    int32_t        ret;        /* function result */
    USBHostMouse mouse;
    
    while(1) {
        // try to connect a USB mouse
        while(!mouse.connect())
            Thread::wait(500);
        
        // when connected, attach handler called on mouse event
        mouse.attachEvent(onMouseEvent);
        
        ret = R_SDG_Open(SDG_OUTPUT_CH, R_SDG_CLOCK_4);
        if (ret < 0) {
            printf("SDG Open Error\r\n");
        }
        
        
        // wait until the mouse is disconnected
        while(mouse.connected())
            Thread::wait(500);
            
        ret = R_SDG_Open(SDG_OUTPUT_CH, R_SDG_CLOCK_4);
        if (ret < 0) {
            printf("SDG Close Error\r\n");
        }
    }
}

int main() {
    // init SDG Note
    work_tone = 72;       /* C */
    work_octave_lev = 2;
    work_div  = DIV_DEFAULT;
    output_note.tone              = work_tone;
    output_note.sfs               = octave_table[work_octave_lev].sfs;
    output_note.loud              = 0;
    output_note.attenuation       = 0;

    Thread mouseTask(mouse_task, NULL, osPriorityNormal, 256 * 4);
    while(1) {
        led=!led;
        Thread::wait(500);
    }
}