/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include "USBHostSerial.h"

#if USBHOST_SERIAL

#include "dbg.h"

USBHostSerial::USBHostSerial(): circ_buf() {
    host = USBHost::getHostInst();
    size_bulk_in = 0;
    size_bulk_out = 0;
    init();
}

void USBHostSerial::init() {
    dev = NULL;
    bulk_in = NULL;
    bulk_out = NULL;
    dev_connected = false;
    serial_intf = -1;
    serial_device_found = false;
}

bool USBHostSerial::connected()
{
    return dev_connected;
}

bool USBHostSerial::connect() {

    if (dev_connected) {
        return true;
    }
    for (int i = 0; i < MAX_DEVICE_CONNECTED; i++) {
        if ((dev = host->getDevice(i)) != NULL) {

            if(host->enumerate(dev, this))
                break;
            
            if (serial_device_found) {
                bulk_in = dev->getEndpoint(serial_intf, BULK_ENDPOINT, IN);
                bulk_out = dev->getEndpoint(serial_intf, BULK_ENDPOINT, OUT);
                
                if (!bulk_in || !bulk_out)
                    break;
                
                USB_INFO("New Serial device: VID:%04x PID:%04x [dev: %p]", dev->getVid(), dev->getPid(), dev);
                dev->setName("Serial");
                host->registerDriver(dev, serial_intf, this, &USBHostSerial::init);
                
                size_bulk_in = bulk_in->getSize();
                size_bulk_out = bulk_out->getSize();
                
                bulk_in->attach(this, &USBHostSerial::rxHandler);
                bulk_out->attach(this, &USBHostSerial::txHandler);
                
                host->bulkRead(dev, bulk_in, buf, size_bulk_in, false);
                dev_connected = true;
                return true;
            }
        }
    }
    init();
    return false;
}

void USBHostSerial::rxHandler() {
    if (bulk_in) {
        int len = bulk_in->getLengthTransferred();
        if (bulk_in->getState() == USB_TYPE_IDLE) {
            for (int i = 0; i < len; i++) {
                circ_buf.queue(buf[i]);
            }
            rx.call();
        }
        host->bulkRead(dev, bulk_in, buf, size_bulk_in, false);
    }
}

void USBHostSerial::txHandler() {
    if (bulk_out) {
        if (bulk_out->getState() == USB_TYPE_IDLE) {
            tx.call();
        }
    }
}

int USBHostSerial::_putc(int c) {
    if (bulk_out) {
        if (host->bulkWrite(dev, bulk_out, (uint8_t *)&c, 1) == USB_TYPE_OK) {
            return 1;
        }
    }
    return -1;
}


int USBHostSerial::_getc() {
    uint8_t c = 0;
    while ((bulk_in != NULL) && (circ_buf.isEmpty()));
    if (bulk_in == NULL) {
        return -1;
    }
    circ_buf.dequeue(&c);
    return c;
}


uint8_t USBHostSerial::available() {
    return circ_buf.available();
}

/*virtual*/ void USBHostSerial::setVidPid(uint16_t vid, uint16_t pid)
{
    // we don't check VID/PID for MSD driver
}

/*virtual*/ bool USBHostSerial::parseInterface(uint8_t intf_nb, uint8_t intf_class, uint8_t intf_subclass, uint8_t intf_protocol) //Must return true if the interface should be parsed
{
    if ((serial_intf == -1) &&
        (intf_class == SERIAL_CLASS) &&
        (intf_subclass == 0x00) &&
        (intf_protocol == 0x00)) {
        serial_intf = intf_nb;
        return true;
    }
    return false;
}

/*virtual*/ bool USBHostSerial::useEndpoint(uint8_t intf_nb, ENDPOINT_TYPE type, ENDPOINT_DIRECTION dir) //Must return true if the endpoint will be used
{
    if (intf_nb == serial_intf) {
        if (type == BULK_ENDPOINT) {
            serial_device_found = true;
            return true;
        }
    }
    return false;
}

#endif


