/* Copyright (c) 2010-2012 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef USBHOSTMSD_H
#define USBHOSTMSD_H

#include "USBHostConf.h"

#if USBHOST_MSD

#include "USBHost.h"
#include "FATFileSystem.h"

/** 
 * A class to communicate a USB flash disk
 */
class USBHostMSD : public IUSBEnumerator, public FATFileSystem {
public:
    /**
    * Constructor
    *
    * @param rootdir mount name
    */
    USBHostMSD(const char * rootdir);

    /**
    * Check if a MSD device is connected
    *
    * @return true if a MSD device is connected
    */
    bool connected();

    /**
     * Try to connect to a MSD device
     *
     * @return true if connection was successful
     */
    bool connect();

protected:
    //From IUSBEnumerator
    virtual void setVidPid(uint16_t vid, uint16_t pid);
    virtual bool parseInterface(uint8_t intf_nb, uint8_t intf_class, uint8_t intf_subclass, uint8_t intf_protocol); //Must return true if the interface should be parsed
    virtual bool useEndpoint(uint8_t intf_nb, ENDPOINT_TYPE type, ENDPOINT_DIRECTION dir); //Must return true if the endpoint will be used

    // From FATFileSystem
    virtual int disk_initialize();
    virtual int disk_status() {return 0;};
    virtual int disk_read(uint8_t * buffer, uint64_t sector);
    virtual int disk_write(const uint8_t * buffer, uint64_t sector);
    virtual int disk_sync() {return 0;};
    virtual uint64_t disk_sectors();

private:
    USBHost * host;
    USBDeviceConnected * dev;
    bool dev_connected;
    USBEndpoint * bulk_in;
    USBEndpoint * bulk_out;

    // Bulk-only CBW
    typedef __packed struct {
        uint32_t Signature;
        uint32_t Tag;
        uint32_t DataLength;
        uint8_t  Flags;
        uint8_t  LUN;
        uint8_t  CBLength;
        uint8_t  CB[16];
    } CBW;

    // Bulk-only CSW
    typedef __packed struct {
        uint32_t Signature;
        uint32_t Tag;
        uint32_t DataResidue;
        uint8_t  Status;
    } CSW;

    CBW cbw;
    CSW csw;

    int SCSITransfer(uint8_t * cmd, uint8_t cmd_len, int flags, uint8_t * data, uint32_t transfer_len);
    int testUnitReady();
    int readCapacity();
    int inquiry(uint8_t lun, uint8_t page_code);
    int SCSIRequestSense();
    int dataTransfer(uint8_t * buf, uint32_t block, uint8_t nbBlock, int direction);
    int checkResult(uint8_t res, USBEndpoint * ep);

    int blockSize;
    uint64_t blockCount;

    int msd_intf;
    bool msd_device_found;
    bool disk_init;
    
    void init();

};

#endif

#endif

