/* Copyright (c) 2010-2012 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef USBDEVICECONNECTED_H
#define USBDEVICECONNECTED_H

#include "stdint.h"
#include "USBEndpoint.h"
#include "USBHostConf.h"

class USBHostHub;

typedef struct {
    bool in_use;
    uint8_t nb_endpoint;
    uint8_t intf_class;
    uint8_t intf_subclass;
    uint8_t intf_protocol;
    USBEndpoint * ep[MAX_ENDPOINT_PER_INTERFACE];
    FunctionPointer detach;
} INTERFACE; 

class USBDeviceConnected
{
public:

    /**
    * Constructor
    */
    USBDeviceConnected();

    /**
    * Attach an USBEndpoint to this device
    *
    * @param ep pointeur on the USBEndpoint which will be attached
    * @returns true if successful, false otherwise
    */
    bool addEndpoint(uint8_t intf_nb, USBEndpoint * ep);

    /**
    * Retrieve an USBEndpoint by its TYPE and DIRECTION
    *
    * @param intf_nb the interface on which to lookup the USBEndpoint
    * @param type type of the USBEndpoint looked for
    * @param dir direction of the USBEndpoint looked for
    * @param index the index of the USBEndpoint whitin the interface
    * @returns pointer on the USBEndpoint if found, NULL otherwise
    */
    USBEndpoint * getEndpoint(uint8_t intf_nb, ENDPOINT_TYPE type, ENDPOINT_DIRECTION dir, uint8_t index = 0);

    /**
    * Retrieve an USBEndpoint by its index
    *
    * @param index index of the USBEndpoint
    * @returns pointer on the USBEndpoint if found, NULL otherwise
    */
    USBEndpoint * getEndpoint(uint8_t intf_nb, uint8_t index);

    /**
    * Add a new interface to this device
    *
    * @param intf_nb interface number
    * @param intf_class interface class
    * @param intf_subclass interface subclass
    * @param intf_protocol interface protocol
    * @returns true if successful, false otherwise
    */
    bool addInterface(uint8_t intf_nb, uint8_t intf_class, uint8_t intf_subclass, uint8_t intf_protocol);

    /**
    * Get the number of interfaces attached to this USB device
    *
    * @returns number of interfaces
    */
    uint8_t getNbInterface() {
        return nb_interf;
    };

    /**
    * Get a specific interface
    *
    * @param index index of the interface to be fetched
    * @returns interface
    */
    INTERFACE * getInterface(uint8_t index);

    /**
     *  Attach a member function to call when a the device has been disconnected
     *
     *  @param intf_nb interface number
     *  @param tptr pointer to the object to call the member function on
     *  @param mptr pointer to the member function to be called
     */
    template<typename T>
    inline void onDisconnect(uint8_t intf_nb, T* tptr, void (T::*mptr)(void)) {
        if ((mptr != NULL) && (tptr != NULL)) {
            intf[intf_nb].detach.attach(tptr, mptr);
        }
    }

    /**
     * Attach a callback called when the device has been disconnected
     *
     *  @param intf_nb interface number
     *  @param fn function pointer
     */
    inline void onDisconnect(uint8_t intf_nb, void (*fn)(void)) {
        if (fn != NULL) {
            intf[intf_nb].detach.attach(fn);
        }
    }

    /**
    * Disconnect the device by calling a callback function registered by a driver
    */
    void disconnect();

    // setters
    void init(uint8_t hub, uint8_t port, bool lowSpeed);
    inline void setAddress(uint8_t addr_) { addr = addr_; };
    inline void setVid(uint16_t vid_) { vid = vid_; };
    inline void setPid(uint16_t pid_) { pid = pid_; };
    inline void setClass(uint8_t device_class_) { device_class = device_class_; };
    inline void setSubClass(uint8_t device_subclass_) { device_subclass = device_subclass_; };
    inline void setProtocol(uint8_t pr) { proto = pr; };
    inline void setSizeControlEndpoint(uint32_t size) { sizeControlEndpoint = size; };
    inline void activeAddress(bool active) { activeAddr = active; };
    inline void setEnumerated() { enumerated = true; };
    inline void setHubParent(USBHostHub * hub) { hub_parent = hub; };
    inline void setName(const char * name_) { strcpy(name, name_); };

    //getters
    inline uint8_t     getPort() { return port; };
    inline uint8_t     getHub() { return hub_nb; };
    inline uint8_t     getAddress() { return addr; };
    inline uint16_t    getVid() { return vid; };
    inline uint16_t    getPid() { return pid; };
    inline uint8_t     getClass() { return device_class; };
    inline uint8_t     getSubClass() { return device_subclass; };
    inline uint8_t     getProtocol() { return proto; };
    inline bool        getSpeed() { return speed; };
    inline uint32_t    getSizeControlEndpoint() { return sizeControlEndpoint; };
    inline bool        isActiveAddress() { return activeAddr; };
    inline bool        isEnumerated() { return enumerated; };
    inline USBHostHub * getHubParent() { return hub_parent; };
    inline const char * getName() { return name; };
    
    // in case this device is a hub
    USBHostHub * hub;

private:
    USBHostHub * hub_parent;

    INTERFACE intf[MAX_INTF];
    uint32_t sizeControlEndpoint;
    uint8_t hub_nb;
    uint8_t port;
    uint16_t vid;
    uint16_t pid;
    uint8_t addr;
    uint8_t device_class;
    uint8_t device_subclass;
    uint8_t proto;
    bool speed;
    volatile bool activeAddr;
    volatile bool enumerated;
    uint8_t nb_interf;

    char name[10];

    void init();
};

#endif

