//#define _DEBUG_ALL

#ifndef HTTP_SERVER_H
#define HTTP_SERVER_H

#ifdef _DEBUG_ALL
#include <stdio.h>
#include <stdarg.h>
static inline void http_server_debug_print(const char *format, ...) {
    va_list args;
    va_start(args, format);
    vfprintf(stderr, format, args);
    va_end(args);
}
#else
static inline void http_server_debug_print(const char *format, ...) {}
#endif

#include <string>
using std::string;

#include <map>
using std::map;

#include "HTTPRequestHandler.h"
#include "rtos.h"
#include "mbed.h"

struct handlersComp { //Used to order handlers in the right way
    bool operator() (const string& handler1, const string& handler2) const {
        //The first handler is longer than the second one
        if (handler1.length() > handler2.length())
            return true; //Returns true if handler1 is to appear before handler2
        else if (handler1.length() < handler2.length())
            return false;
        else //To avoid the == case, sort now by address
            return ((&handler1)>(&handler2));
    }
};

map< string, HTTPRequestHandler*(*)(const char*, const char* , TCPSocket* ), handlersComp > m_lpHandlers;
template<typename T>
void HTTPServerAddHandler(const char* path)  //Template decl in header
{
    m_lpHandlers[path] = &T::inst;
}

void ListenThread(void const *args);
enum HTTP_METH {
    HTTP_GET,
    HTTP_POST,
    HTTP_HEAD
};

bool getRequest(TCPSocket* client, string* path, string* meth)
{
    char req[128];
    char c_path[128];
    char c_meth[128];
    const int maxLen = 128;
    char* p = req;
    //Read Line
    int ret;
    int len = 0;
    for(int i = 0; i < maxLen - 1; i++) {
        ret = client->recv(p, 1);
        if(ret <= 0) {
            return false;
        }
        if( (len > 1) && *(p-1)=='\r' && *p=='\n' ) {
            p--;
            len-=2;
            break;
        } else if( *p=='\n' ) {
            len--;
            break;
        }
        p++;
        len++;
    }
    *p = 0;
    http_server_debug_print("Parsing request : %s\r\n", req);
    ret = sscanf(req, "%s %s HTTP/%*d.%*d", c_meth, c_path);
    if(ret !=2)        return false;
    *meth = string(c_meth);
    *path = string(c_path);
    return true;
}

void dispatchRequest(TCPSocket* client)
{
    string path;
    string meth;
    HTTP_METH methCode;
    http_server_debug_print("Dispatching req\r\n");
    if( !getRequest(client,&path, &meth ) ) {
        http_server_debug_print("dispatchRequest Invalid request\r\n");
        return; //Invalid request
    }
    if( !meth.compare("GET") ) {
        http_server_debug_print("dispatchRequest HTTP_GET\r\n");
        methCode = HTTP_GET;
    } else if( !meth.compare("POST") ) {
        http_server_debug_print("dispatchRequest HTTP_POST\r\n");
        methCode = HTTP_POST;
    } else if( !meth.compare("HEAD") ) {
        http_server_debug_print("dispatchRequest HTTP_HEAD\r\n");
        methCode = HTTP_HEAD;
    } else {
        http_server_debug_print("dispatchRequest() Parse error\r\n");
        return;
    }
    http_server_debug_print("Looking for a handler\r\n");
    map< string, HTTPRequestHandler*(*)(const char*, const char*, TCPSocket*), handlersComp >::iterator it;
    int root_len = 0;
    for (it = m_lpHandlers.begin(); it != m_lpHandlers.end(); it++) {
        http_server_debug_print("Checking %s...\r\n", (*it).first.c_str());
        root_len = (*it).first.length();
        if ( root_len &&
                !path.compare( 0, root_len, (*it).first ) &&
                (path[root_len] == '/' || path[root_len] == '\0')) {
            http_server_debug_print("Found (%s)\r\n", (*it).first.c_str());
            // Found!
            break;  // for
        }
    }
    if((it == m_lpHandlers.end()) && !(m_lpHandlers.empty())) {
        http_server_debug_print("Using default handler\r\n");
        it = m_lpHandlers.end();
        it--; //Get the last element
        if( ! (((*it).first.length() == 0) || !(*it).first.compare("/")) ) //This is not the default handler
            it = m_lpHandlers.end();
        root_len = 0;
    }
    if(it == m_lpHandlers.end()) {
        http_server_debug_print("No handler found\r\n");
        return;
    }
    http_server_debug_print("Handler found.\r\n");
    HTTPRequestHandler* pHdlr = (*it).second((*it).first.c_str(), path.c_str() + root_len, client);
    //****  client = NULL; //We don't own it anymore
    switch(methCode) {
        case HTTP_GET:
            pHdlr->doGet();
            break;
        case HTTP_POST:
            pHdlr->doPost();
            break;
        case HTTP_HEAD:
            pHdlr->doHead();
            break;
    }
    delete pHdlr;
    http_server_debug_print("(dispatcherRequest)return\r\n");
    return ;
}

#ifndef HTTP_SERVER_THREAD_MAX
#define HTTP_SERVER_THREAD_MAX 1
#endif

#if (HTTP_SERVER_THREAD_MAX > 1) 
static Thread *threads[HTTP_SERVER_THREAD_MAX];
static bool soket_rady[HTTP_SERVER_THREAD_MAX];
static TCPSocket * clients[HTTP_SERVER_THREAD_MAX];

static void HTTPServerChild (void* param)
{
    bool* p_rady = &soket_rady[(int)param];

    while (1) {
        ThisThread::flags_wait_all(1);
        TCPSocket* client = clients[(int)param];
        dispatchRequest(client);
        client->close();
        *p_rady = true;
    }
}

void HTTPServerStart(NetworkInterface *net, int port = 80, osPriority priority = osPriorityNormal)
{
    int i;
    TCPSocket server;

    for (i = 0; i < HTTP_SERVER_THREAD_MAX; i++) {
        soket_rady[i] = true;
        threads[i] = new Thread(priority, 1024 * 3);
        threads[i]->start(callback(HTTPServerChild, (void *)i));
    }

    server.open(net);
    server.bind(port);
    server.listen();
    http_server_debug_print("Wait for new connection...\r\n");

    while (1) {
        while (1) {
            for (i = 0; i < HTTP_SERVER_THREAD_MAX; i++) {
                if (soket_rady[i] != false) {
                    break;
                }
            }
            if (i < HTTP_SERVER_THREAD_MAX) {
                break;
            }
            ThisThread::sleep_for(5);
        }

        clients[i] = server.accept();
        if (clients[i] != NULL) {
            // fork child process
            soket_rady[i] = false;
            threads[i]->flags_set(1);
        }
    }
}

#else  // HTTP_SERVER_THREAD_MAX == 1

void HTTPServerStart(NetworkInterface *net, int port = 80, osPriority priority = osPriorityNormal)
{
    TCPSocket server;
    TCPSocket * p_client;

    (void)priority;

    server.open(net);
    server.bind(port);
    server.listen();
    http_server_debug_print("Wait for new connection...\r\n");

    while (1) {
        p_client = server.accept();
        if (p_client != NULL) {
            dispatchRequest(p_client);
            p_client->close();
        }
    }
}

#endif

#include "Handler/RPCHandler.h"
#include "Handler/FSHandler.h"
#include "Handler/SimpleHandler.h"
#include "SnapshotHandler.h"

#endif
