#include "mbed.h"
#include "picojson.h"
#include "iot_platform.h"
#include <string>
#include <iostream>
#include <vector>
#include "easy-connect.h"
#include "http_request.h"
#include "NTPClient.h"
#include <time.h>

/* Detect result */
result_hvcp2_fd_t result_hvcp2_fd[DETECT_MAX];
result_hvcp2_bd_t result_hvcp2_bd[DETECT_MAX];
uint32_t result_hvcp2_bd_cnt;
uint32_t result_hvcp2_fd_cnt;

Timer http_resp_time;   // response time
uint16_t data[4];       //for color data

#define JST_OFFSET 9

#define ACCESS_CODE "Bearer <ACCESS CODE>"
#error "You have to replace the above <ACCESS CODE> with yours"

std::string put_uri_base("<Base URI>/v1/<Tenant ID>/<Path-to-Resource>.json");
#error "You have to replace <Base URI>, <Tenant ID> and <Path-to-Resource> with yours"

std::string put_uri;

// json-object for camera
picojson::object o_bd[DETECT_MAX], o_fd[DETECT_MAX], o_fr[DETECT_MAX], o_scr[DETECT_MAX];
// json-object for sensor
picojson::object o_acc, o_atmo, o_col, o_temp;

picojson::array data_array_acc(6);
picojson::array data_array_atmo(6);
picojson::array data_array_col(6);
picojson::array data_array_temp(6);

// URI for GET request
std::string get_uri("<Base URI>/v1/<Tenant ID>/<Path-to-Resource>/_past.json");
#error "You have to replace <Base URI>, <Tenant ID> and <Path-to-Resource> with yours"

void dump_response(HttpResponse* res)
{
    DEBUG_PRINT("Status: %d - %s\n", res->get_status_code(),
                res->get_status_message().c_str());

    DEBUG_PRINT("Headers:\n");
    for (size_t ix = 0; ix < res->get_headers_length(); ix++) {
        DEBUG_PRINT("\t%s: %s\n", res->get_headers_fields()[ix]->c_str(),
                    res->get_headers_values()[ix]->c_str());
    }
    DEBUG_PRINT("\nBody (%d bytes):\n\n%s\n", res->get_body_length(),
                res->get_body_as_string().c_str());
}

std::string create_put_uri(std::string uri_base)
{
    time_t ctTime;
    struct tm *pnow;
    char date_and_hour[50];
    std::string uri;
    
    ctTime = time(NULL);
    pnow = localtime(&ctTime);
    sprintf(date_and_hour, "?$date=%04d%02d%02dT%02d%02d%02d.000%%2B%02d00",
        (pnow->tm_year + 1900), (pnow->tm_mon + 1), pnow->tm_mday,
        (pnow->tm_hour + JST_OFFSET - pnow->tm_isdst), pnow->tm_min,
        pnow->tm_sec, (JST_OFFSET - pnow->tm_isdst));

    uri = uri_base + date_and_hour;
    
    return(uri);
}

int iot_put(NetworkInterface *network, picojson::object o4)
{

#ifdef ENABLED_NTP
    put_uri = create_put_uri(put_uri_base);
#else
    put_uri = put_uri_base;
#endif  // ENABLED_NTP

    // PUT request to IoT Platform
    HttpRequest* put_req = new HttpRequest(network, HTTP_PUT, put_uri.c_str());
    put_req->set_header("Authorization", ACCESS_CODE);

    picojson::value v_all(o4);

    std::string body = v_all.serialize();

    HttpResponse* put_res = put_req->send(body.c_str(), body.length());
    
    if (!put_res) {
        DEBUG_PRINT("HttpRequest failed (error code %d)\n", put_req->get_error());
        return 1;
    }

    delete put_req;
    return 0;
}

int iot_get(NetworkInterface *network)
{
    // Do GET request to IoT Platform
    // By default the body is automatically parsed and stored in a buffer, this is memory heavy.
    // To receive chunked response, pass in a callback as last parameter to the constructor.
    HttpRequest* get_req = new HttpRequest(network, HTTP_GET, get_uri.c_str());
    get_req->set_header("Authorization", ACCESS_CODE);

    HttpResponse* get_res = get_req->send();

    if (!get_res) {
        DEBUG_PRINT("HttpRequest failed (error code %d)\n", get_req->get_error());
        return 1;
    }

    DEBUG_PRINT("\n----- HTTP GET response -----\n");

    delete get_req;

    return 0;
}

int send_hvc_info(NetworkInterface *network)
{
    /* No face detect */
    if (result_hvcp2_fd_cnt == 0) {
        /* Do nothing */
    } else {
        for (uint32_t i = 0; i < result_hvcp2_fd_cnt; i++) {
            /* picojson-object clear */
            o_fd[i].clear();
            o_fr[i].clear();
            o_scr[i].clear();
            /* Type */
            o_fd[i]["RecordType"] = picojson::value((string)"HVC-P2(face)");
            o_fd[i]["id"] = picojson::value((string) "0001-0005");
            /* Age */
            o_fd[i]["Age"] = picojson::value((double)result_hvcp2_fd[i].age.age);
            /* Gender */
            o_fd[i]["Gender"] = picojson::value((double)result_hvcp2_fd[i].gender.gender);
            /* FaceRectangle */
            o_fr[i]["Top"]  = picojson::value((double)result_hvcp2_fd[i].face_rectangle.MinY);
            o_fr[i]["Left"] = picojson::value((double)result_hvcp2_fd[i].face_rectangle.MinX);
            o_fr[i]["Width"] = picojson::value((double)result_hvcp2_fd[i].face_rectangle.Width);
            o_fr[i]["Height"] = picojson::value((double)result_hvcp2_fd[i].face_rectangle.Height);
            /* Scores */
            o_scr[i]["Neutral"] = picojson::value((double)result_hvcp2_fd[i].scores.score_neutral);
            o_scr[i]["Anger"] = picojson::value((double)result_hvcp2_fd[i].scores.score_anger);
            o_scr[i]["Happiness"] = picojson::value((double)result_hvcp2_fd[i].scores.score_happiness);
            o_scr[i]["Surprise"] = picojson::value((double)result_hvcp2_fd[i].scores.score_surprise);
            o_scr[i]["Sadness"] = picojson::value((double)result_hvcp2_fd[i].scores.score_sadness);
            /* insert 2 structures */
            o_fd[i]["FaceRectangle"] = picojson::value(o_fr[i]);
            o_fd[i]["Scores"] = picojson::value(o_scr[i]);
        }
    }

    /* No body detect */
    if (result_hvcp2_bd_cnt == 0) {
        /* Do nothing */
    } else {
        for (uint32_t i = 0; i < result_hvcp2_bd_cnt; i++) {
            /* picojson-object clear */
            o_bd[i].clear();
            /* Type */
            o_bd[i]["RecordType"] = picojson::value((string)"HVC-P2(body)");
            o_bd[i]["id"] = picojson::value((string)"0001-0006");
            /* BodyRectangle */
            o_bd[i]["Top"] = picojson::value((double)result_hvcp2_bd[i].body_rectangle.MinY);
            o_bd[i]["Left"] = picojson::value((double)result_hvcp2_bd[i].body_rectangle.MinX);
            o_bd[i]["Width"] = picojson::value((double)result_hvcp2_bd[i].body_rectangle.Width);
            o_bd[i]["Height"] = picojson::value((double)result_hvcp2_bd[i].body_rectangle.Height);
        }
    }

    DEBUG_PRINT("Face detect count : %d\n", result_hvcp2_fd_cnt);
    DEBUG_PRINT("Body detect count : %d\n", result_hvcp2_bd_cnt);

    http_resp_time.reset();
    http_resp_time.start();

    /* send data */
    if (result_hvcp2_fd_cnt == 0) {
        /* No need to send data */
    } else {
        for (uint32_t i = 0; i < result_hvcp2_fd_cnt; i++) {
            iot_put(network, o_fd[i]);
        }
    }
    if (result_hvcp2_bd_cnt == 0) {
        /* No need to send data */
    } else {
        for (uint32_t i = 0; i < result_hvcp2_bd_cnt; i++) {
            iot_put(network, o_bd[i]);
        }
    }
    DEBUG_PRINT("iot_put() Response time:%dms\n", http_resp_time.read_ms());
    return 0;
}

void iot_ready_task(void)
{

    /* Initialize http */
    NetworkInterface *network = easy_connect(true);
    MBED_ASSERT(network);

#ifdef ENABLED_NTP
    // Generate the string indicating the date and hour specified for PUT request
    NTPClient ntp;
    time_t ctTime;
    struct tm *pnow;
    NTPResult ret;

    ret = ntp.setTime("ntp.nict.jp");
    MBED_ASSERT( ret==0 );

    ctTime = time(NULL);
#endif  // Enabled_NTP

    while (1) {

        semaphore_wait_ret = iot_ready_semaphore.wait();
        MBED_ASSERT(semaphore_wait_ret != -1);

        /* send hvc-p2 data */
        http_resp_time.reset();
        http_resp_time.start();
        send_hvc_info(network);
        DEBUG_PRINT("send_hvc_info() Response time:%dms\n", http_resp_time.read_ms());

        iot_ready_semaphore.release();

        Thread::wait(WAIT_TIME);
    };
}
