#include "mbed.h"
#include "DisplayBace.h"
#include "rtos.h"
#include "LCD_shield_config_4_3inch.h"
#include "recognition_proc.h"
#include "touch_proc.h"
#include "iot_platform.h"

static DisplayBase Display;
static DigitalOut  lcd_pwon(P7_15);
static DigitalOut  lcd_blon(P8_1);
static PwmOut      lcd_cntrst(P8_15);
static Thread      recognitionTask;
static Thread      touchTask;

extern void iot_ready_task(void);

Serial pc(USBTX, USBRX);
DigitalOut D_D0(P2_14);
DigitalOut D_D1(P2_15);

/****** LCD ******/
static void IntCallbackFunc_LoVsync(DisplayBase::int_type_t int_type)
{
    /* Interrupt callback function for Vsync interruption */
    touch_lcd_int(int_type);
}

static void Init_LCD_Display(void)
{
    DisplayBase::graphics_error_t error;
    DisplayBase::lcd_config_t lcd_config;
    PinName lvds_pin[8] = {
        /* data pin */
        P5_7, P5_6, P5_5, P5_4, P5_3, P5_2, P5_1, P5_0
    };

    lcd_pwon = 0;
    lcd_blon = 0;
    Thread::wait(100);
    lcd_pwon = 1;
    lcd_blon = 1;

    Display.Graphics_Lvds_Port_Init(lvds_pin, 8);

    /* Graphics initialization process */
    lcd_config = LcdCfgTbl_LCD_shield;
    error = Display.Graphics_init(&lcd_config);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        mbed_die();
    }

    /* Interrupt callback function setting (Vsync signal output from scaler 0) */
    error = Display.Graphics_Irq_Handler_Set(DisplayBase::INT_TYPE_S0_LO_VSYNC, 0, IntCallbackFunc_LoVsync);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        mbed_die();
    }
}

/****** main ******/
int main(void)
{

    Thread *iotReadyTask = new Thread(osPriorityNormal, 4 * 1024);
    pc.baud(115200);
    printf("Start.\n");

    /* Initialization of LCD */
    D_D0 = 1;
    D_D1 = 1;

    Init_LCD_Display();

    /* Start recognition processing */
    recognitionTask.start(callback(recognition_task, &Display));

    /* Start touch panel processing */
    touchTask.start(callback(touch_task, &Display));

    /* Start IOT ready processing */
    iotReadyTask->start(callback(iot_ready_task));

    /* Backlight on */
    Thread::wait(200);
    lcd_cntrst.write(1.0);

    /* Wait for the threads to finish */
    recognitionTask.join();
    touchTask.join();
    iotReadyTask->join();
}
