/*******************************************************************************
* DISCLAIMER
* This software is supplied by Renesas Electronics Corporation and is only
* intended for use with Renesas products. No other uses are authorized. This
* software is owned by Renesas Electronics Corporation and is protected under
* all applicable laws, including copyright laws.
* THIS SOFTWARE IS PROVIDED "AS IS" AND RENESAS MAKES NO WARRANTIES REGARDING
* THIS SOFTWARE, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING BUT NOT
* LIMITED TO WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
* AND NON-INFRINGEMENT. ALL SUCH WARRANTIES ARE EXPRESSLY DISCLAIMED.
* TO THE MAXIMUM EXTENT PERMITTED NOT PROHIBITED BY LAW, NEITHER RENESAS
* ELECTRONICS CORPORATION NOR ANY OF ITS AFFILIATED COMPANIES SHALL BE LIABLE
* FOR ANY DIRECT, INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES FOR
* ANY REASON RELATED TO THIS SOFTWARE, EVEN IF RENESAS OR ITS AFFILIATES HAVE
* BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
* Renesas reserves the right, without notice, to make changes to this software
* and to discontinue the availability of this software. By using this software,
* you agree to the additional terms and conditions found by accessing the
* following link:
* http://www.renesas.com/disclaimer*
* Copyright (C) 2015 Renesas Electronics Corporation. All rights reserved.
*******************************************************************************/

#ifndef DISP_TFT_H
#define DISP_TFT_H

#include "r_typedefs.h"
#include "display.h"
#include "disp_graphics.h"

/*--- Macro definition ---*/
/* Display size */
#define DSP_TFT_WIDTH                   (800u)
#define DSP_TFT_HEIGHT                  (480u)

/* Graphics layer */
#define DSP_TFT_LAYER_0                 (0u)
#define DSP_TFT_LAYER_1                 (1u)
#define DSP_TFT_LAYER_NUM               (2u)
#define DSP_TFT_LAYER_NON               (0xFFFFu)

#define DSP_TFT_M2_AUDIO_BUF_SIZE       (400u)
#define DSP_TFT_M3_AUDIO_BUF_SIZE       (800u)

#define DSP_TFT_M3_PROC_COUNT_PER_SEC      (10u) /* Processing number per 1 second */
                                                 /* in the case of the display mode 3. */
#define DSP_TFT_M3_AUDIO_SAMPLE_PER_100MS  (4u)  /* Sample number of the audio data per 100ms */
                                                 /* in the case of the display mode 3. */
#define DSP_TFT_M3_AUDIO_SAMPLE_PER_SEC    (DSP_TFT_M3_PROC_COUNT_PER_SEC * DSP_TFT_M3_AUDIO_SAMPLE_PER_100MS)

/*--- User defined types ---*/
typedef struct {
    uint32_t        pic_pos_x;          /* Display X position of the key picture. */
    uint32_t        pic_pos_y;          /* Display Y position of the key picture. */
    uint32_t        pic_siz_x;          /* Width size of the key picture. */
    uint32_t        pic_siz_y;          /* Height size of the key picture. */
    SYS_KeyCode     key_code;           /* Key code of the key picture. */
} dsp_cnv_key_t;

typedef struct {
    uint32_t        req_cyc_cnt;        /* Cycle counter of audio data acquisition. */
    uint32_t        m2_buf_cnt;         /* Data counter in the buffer for TFT display mode 2. */
    uint32_t        m3_buf_cnt;         /* Data counter in the buffer for TFT display mode 3. */
    int16_t         req_buf[DSP_TFT_M2_AUDIO_BUF_SIZE]; /* Buffer for audio data acquisition. */
    int16_t         m2_buf[DSP_TFT_M2_AUDIO_BUF_SIZE];  /* Buffer for TFT display mode 2. */
    int16_t         m3_buf[DSP_TFT_M3_AUDIO_BUF_SIZE];  /* Buffer for TFT display mode 3. */
    uint32_t        m3_sample_cnt;      /* Data counter in the audio sample */
    uint32_t        m3_target_cnt;      /* Data counter in the time correction */
    bool            m3_pause_flag;      /* Pause decision flag */
} dsp_audio_t;

/* Control data of display thread */
/* These data are used only in the TFT module. */
typedef struct {
    int32_t         disp_phase_no;      /* The making phase of the display image */
    dsp_audio_t     audio_data;         /* Audio data for the display */
    SYS_KeyCode     key_code;
    dsp_tftlayer_t  tft_info[DSP_TFT_LAYER_NUM];/* VRAM structure of each display layer */
} dsp_tft_ctrl_t;

/** Initialises tft control module
 *
 */
void dsp_init_tft(dsp_tft_ctrl_t * const p_tft);

/** Executes the main processing of tft control module
 *
 *  @param mail_id Mail ID.
 *  @param p_com Pointer to common data in all display module.
 *  @param p_tft Pointer to management data of TFT module.
 */
void dsp_output_tft(const DSP_MAIL_ID mail_id, 
                    const dsp_com_ctrl_t * const p_com, dsp_tft_ctrl_t * const p_tft);

/** This function converts the touch position of current screen into the key code.
 *
 *  @param disp_mode Display mode
 *  @param pos_x Touch X position
 *  @param pos_y Touch Y position
 *
 *  @returns 
 *    If touch position is on the picture of key, this function returns key code.
 *    Otherwise this function returns "SYS_KEYCODE_NON".
 */
SYS_KeyCode dsp_convert_key_tft(const uint32_t disp_mode, 
                                    const uint32_t pos_x, const uint32_t pos_y);

/** This function clears the audio data to use by TFT display.
 *
 *  @param p_tft Pointer to management data of TFT module.
 */
void dsp_clear_tft_audio_data(dsp_tft_ctrl_t * const p_tft);

#endif /* DISP_TFT_H */
