/*******************************************************************************
* DISCLAIMER
* This software is supplied by Renesas Electronics Corporation and is only
* intended for use with Renesas products. No other uses are authorized. This
* software is owned by Renesas Electronics Corporation and is protected under
* all applicable laws, including copyright laws.
* THIS SOFTWARE IS PROVIDED "AS IS" AND RENESAS MAKES NO WARRANTIES REGARDING
* THIS SOFTWARE, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING BUT NOT
* LIMITED TO WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
* AND NON-INFRINGEMENT. ALL SUCH WARRANTIES ARE EXPRESSLY DISCLAIMED.
* TO THE MAXIMUM EXTENT PERMITTED NOT PROHIBITED BY LAW, NEITHER RENESAS
* ELECTRONICS CORPORATION NOR ANY OF ITS AFFILIATED COMPANIES SHALL BE LIABLE
* FOR ANY DIRECT, INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES FOR
* ANY REASON RELATED TO THIS SOFTWARE, EVEN IF RENESAS OR ITS AFFILIATES HAVE
* BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
* Renesas reserves the right, without notice, to make changes to this software
* and to discontinue the availability of this software. By using this software,
* you agree to the additional terms and conditions found by accessing the
* following link:
* http://www.renesas.com/disclaimer*
* Copyright (C) 2015 Renesas Electronics Corporation. All rights reserved.
*******************************************************************************/

#include "mbed.h"
#include "rtos.h"
#include "misratypes.h"
#include "DisplayBace.h"
#include "display.h"
#include "disp_tft.h"
#include "disp_tft_scrn.h"

/*--- Macro definition of LCD display driver ---*/
#define LCD_INPUT_CLK           (66.67)     /* Input clock: P1 clk [MHz] */
#define LCD_OUTPUT_CLK          (33.26)     /* Output clock: LCD clk [MHz] */
#define LCD_HSYNC_PERIOD        (1056)      /* Free-running Hsync period: 800 + 92 + 128 + 36 */
#define LCD_VSYNC_PERIOD        (525)       /* Free-running Vsync period: 480 + 5 + 35 + 5 */
#define LCD_H_BACK_PORCH        (164)       /* LCD display horizontal back porch period: 128 + 36 */
#define LCD_V_BACK_PORCH        (40)        /* LCD display vertical back porch period: 35 + 5 */
#define LCD_PIN_NUM             (8)
#define LCD_INPUT_FORMAT_SIZE   (4u)

/* Power ON sequence */
#define BLON_WAIT_TIME_MS       (200)       /* Wait time of milisecond before the backlight ON. */

/* Frame buffer stride: Frame buffer stride should be set to a multiple of 32 or 128 */
/* in accordance with the frame buffer burst transfer mode. */
#define BUF_STRIDE              (((DSP_TFT_WIDTH * LCD_INPUT_FORMAT_SIZE) + 31u) & ~31u)
#define BUF_HEIGHT              (DSP_TFT_HEIGHT)
#define BUF_SIZE                (BUF_STRIDE * BUF_HEIGHT)

#if defined( __CC_ARM )
#define VRAM_SECTION1           __attribute((at(0x60100000)))
#define VRAM_SECTION2           __attribute((at(0x60480000)))
#define VRAM_SECTION3           __attribute((at(0x60600000)))
#define VRAM_SECTION4           __attribute((at(0x60780000)))
#else  /* !defined( __CC_ARM ) */
/* CAUTION! */
/* Modify the start address of "NC_BSS" section in "RZA1H.ld" file. */
#define VRAM_SECTION            __attribute((section("NC_BSS"),aligned(16)))
#define VRAM_SECTION1           VRAM_SECTION
#define VRAM_SECTION2           VRAM_SECTION
#define VRAM_SECTION3           VRAM_SECTION
#define VRAM_SECTION4           VRAM_SECTION
#endif /* defined( __CC_ARM ) */

#define SHIFT_1BIT_SIZE         (1u)

static void init_lcd_driver(const dsp_tft_ctrl_t * const p_tft);
static void update_tft_display(dsp_tft_ctrl_t * const p_tft, const uint32_t layer_id);
static void update_tft_audio_data(const dsp_com_ctrl_t * const p_com, dsp_tft_ctrl_t * const p_tft);
static void vsync_callback(DisplayBase::int_type_t int_type);
static DisplayBase::graphics_layer_t convert_layer_id(const uint32_t layer_id);
static void swap_frame_buffer(dsp_tftlayer_t * const p_info);

static DisplayBase lcd;

void dsp_init_tft(dsp_tft_ctrl_t * const p_tft)
{
    static uint32_t VRAM_SECTION1   frame_buffer_1[BUF_SIZE/sizeof(uint32_t)];
    static uint32_t VRAM_SECTION2   frame_buffer_2[BUF_SIZE/sizeof(uint32_t)];
    static uint32_t VRAM_SECTION3   frame_buffer_3[BUF_SIZE/sizeof(uint32_t)];
    static uint32_t VRAM_SECTION4   frame_buffer_4[BUF_SIZE/sizeof(uint32_t)];
    uint32_t                        i;

    if (p_tft != NULL) {
        p_tft->key_code = SYS_KEYCODE_NON;
        (void) memset(frame_buffer_1, 0, sizeof(frame_buffer_1));
        (void) memset(frame_buffer_2, 0, sizeof(frame_buffer_2));
        (void) memset(frame_buffer_3, 0, sizeof(frame_buffer_3));
        (void) memset(frame_buffer_4, 0, sizeof(frame_buffer_4));
        p_tft->tft_info[DSP_TFT_LAYER_0].p_disp_buf = &frame_buffer_1[0];
        p_tft->tft_info[DSP_TFT_LAYER_0].p_back_buf = &frame_buffer_2[0];
        p_tft->tft_info[DSP_TFT_LAYER_1].p_disp_buf = &frame_buffer_3[0];
        p_tft->tft_info[DSP_TFT_LAYER_1].p_back_buf = &frame_buffer_4[0];
        for (i = 0u; i < DSP_TFT_LAYER_NUM; i++) {
            p_tft->tft_info[i].width  = DSP_TFT_WIDTH;
            p_tft->tft_info[i].height = DSP_TFT_HEIGHT;
            p_tft->tft_info[i].stride = BUF_STRIDE;
        }
        init_lcd_driver(p_tft);
        dsp_init_tft_scrn(p_tft);
    }
}

void dsp_output_tft(const DSP_MAIL_ID mail_id, 
                    const dsp_com_ctrl_t * const p_com, dsp_tft_ctrl_t * const p_tft)
{
    uint32_t                        result = DSP_TFT_LAYER_NON;

    if ((p_com != NULL) && (p_tft != NULL)) {
        switch (mail_id) {
            case DSP_MAILID_CYCLE_IND :
                /* Executes main function of the playback screen. */
                result = dsp_make_tft_scrn(p_com, p_tft);
                break;
            case DSP_MAILID_PLAY_TIME :
            case DSP_MAILID_PLAY_INFO :
            case DSP_MAILID_PLAY_STAT :
                if (p_com->edge_track_change == true) {
                    dsp_clear_tft_audio_data(p_tft);
                }
                result = DSP_TFT_LAYER_NON;
                break;
            case DSP_MAILID_AUDIO_FIN:
                update_tft_audio_data(p_com, p_tft);
                result = DSP_TFT_LAYER_NON;
                break;
            case DSP_MAILID_CMD_STR :
            case DSP_MAILID_PRINT_STR :
            case DSP_MAILID_FILE_NAME :
            default :
                result = DSP_TFT_LAYER_NON;
                break;
        }
        if (result < DSP_TFT_LAYER_NUM) {
            /* Updates the TFT display. */
            update_tft_display(p_tft, result);
        }
    }
}

SYS_KeyCode dsp_convert_key_tft(const uint32_t disp_mode, const uint32_t pos_x, const uint32_t pos_y)
{
    SYS_KeyCode     ret;

    switch (disp_mode) {
        case DSP_DISPMODE_1:
        case DSP_DISPMODE_2:
        case DSP_DISPMODE_3:
            ret = dsp_convert_key_tft_scrn(pos_x, pos_y);
            break;
        default:
            ret = SYS_KEYCODE_NON;
            break;
    }
    return ret;
}

void dsp_clear_tft_audio_data(dsp_tft_ctrl_t * const p_tft)
{
    dsp_audio_t     *p_aud;

    if (p_tft != NULL) {
        p_aud = &p_tft->audio_data;
        p_aud->m2_buf_cnt = 0u;
        (void) memset(&p_aud->m2_buf[0], 0u, sizeof(p_aud->m2_buf));
        p_aud->m3_buf_cnt = 0u;
        (void) memset(&p_aud->m3_buf[0], 0u, sizeof(p_aud->m3_buf));
        p_aud->m3_sample_cnt = 0u;
        p_aud->m3_target_cnt = 0u;
        p_aud->m3_pause_flag = false;
    }
}

/** Initializes LCD driver using DisplayBase class.
 *
 *  @param p_tft Pointer to management data of TFT module.
 */
static void init_lcd_driver(const dsp_tft_ctrl_t * const p_tft)
{
    DisplayBase::graphics_error_t   grap_err;
    DisplayBase::rect_t             rect;
    DisplayBase::lcd_config_t       lcd_config = {
        DisplayBase::LCD_TYPE_LVDS,         /* LVDS or Pararel RGB */
        LCD_INPUT_CLK,                      /* P1  clk [MHz] */
        LCD_OUTPUT_CLK,                     /* LCD clk [MHz] */

        DisplayBase::LCD_OUTFORMAT_RGB888,  /* Output format select */
        DisplayBase::EDGE_RISING,           /* Output phase control of LCD_DATA23 to LCD_DATA0 pin */

        LCD_HSYNC_PERIOD,                   /* Free-running Hsync period */
        LCD_VSYNC_PERIOD,                   /* Free-running Vsync period */
        (uint16_t)DSP_TFT_WIDTH,            /* LCD display area size, horizontal width */
        (uint16_t)DSP_TFT_HEIGHT,           /* LCD display area size, vertical width */
        LCD_H_BACK_PORCH,                   /* LCD display horizontal back porch period */
        LCD_V_BACK_PORCH,                   /* LCD display vertical back porch period */

        DisplayBase::LCD_TCON_PIN_NON,      /* TCONn or Not use(-1) */
        DisplayBase::SIG_POL_NOT_INVERTED,  /* Polarity inversion control of signal */
        0,                                  /* Hsync width */

        DisplayBase::LCD_TCON_PIN_NON,      /* TCONn or Not use(-1) */
        DisplayBase::SIG_POL_NOT_INVERTED,  /* Polarity inversion control of signal */
        0,                                  /* Vsync width  */

        DisplayBase::LCD_TCON_PIN_3,        /* TCONn or Not use(-1) */
        DisplayBase::SIG_POL_NOT_INVERTED   /* Polarity inversion control of signal */
    };
    PinName                         lvds_pin[LCD_PIN_NUM] = {
        P5_7, P5_6, P5_5, P5_4, P5_3, P5_2, P5_1, P5_0  /* Data pin */
    };
    static DigitalOut               lcd_pwon(P7_15);
    static DigitalOut               lcd_blon(P8_1);
    static PwmOut                   lcd_cntrst(P8_15);
    uint32_t                        i;
    DisplayBase::graphics_layer_t   layer_id;

    if (p_tft != NULL) {
        lcd_pwon.write(1);                  /* LCD panel power ON. */
        /* Graphics initialization process */
        grap_err = lcd.Graphics_init(&lcd_config);
        if (grap_err == DisplayBase::GRAPHICS_OK) {
            /* Interrupt callback function setting (Vsync signal output from scaler 0) */
            grap_err = lcd.Graphics_Irq_Handler_Set(DisplayBase::INT_TYPE_S0_LO_VSYNC, 0, &vsync_callback);
            if (grap_err == DisplayBase::GRAPHICS_OK) {
                lcd.Graphics_Lvds_Port_Init(lvds_pin, LCD_PIN_NUM);
                rect.vs = 0;
                rect.vw = (uint16_t)DSP_TFT_HEIGHT;
                rect.hs = 0;
                rect.hw = (uint16_t)DSP_TFT_WIDTH;
                for (i = 0u; i < DSP_TFT_LAYER_NUM; i++) {
                    layer_id = convert_layer_id(i);
                    lcd.Graphics_Read_Setting(layer_id, (void *)p_tft->tft_info[i].p_disp_buf, 
                                            BUF_STRIDE, DisplayBase::GRAPHICS_FORMAT_ARGB8888, 
                                            DisplayBase::WR_RD_WRSWA_32BIT, &rect);
                    lcd.Graphics_Start(layer_id);
                }

                Thread::wait(BLON_WAIT_TIME_MS);
                lcd_blon.write(1);          /* LCD panel backlight ON. */
                lcd_cntrst.write(1.0);
            }
        }
    }
}

/** Interrupt callback function for Vsync interruption.
 *
 *  @param int_type The number of VDC5 interrupt types
 */
static void vsync_callback(DisplayBase::int_type_t int_type)
{
    UNUSED_ARG(int_type);
    (void) dsp_notify_cycle_time();
}

/** Updates the display of the specified layer.
 *
 *  @param p_tft Pointer to management data of TFT module.
 *  @param layer_id The number of the display layer of TFT module.
 */
static void update_tft_display(dsp_tft_ctrl_t * const p_tft, const uint32_t layer_id)
{
    DisplayBase::graphics_layer_t   result;
    DisplayBase::graphics_error_t   grap_err;

    if ((p_tft != NULL) && (layer_id < DSP_TFT_LAYER_NUM)) {
        result = convert_layer_id(layer_id);
        grap_err = lcd.Graphics_Read_Change(result, p_tft->tft_info[layer_id].p_back_buf);
        if (grap_err == DisplayBase::GRAPHICS_OK) {
            swap_frame_buffer(&p_tft->tft_info[layer_id]);
        }
    }
}

/** Updates the audio data.
 *
 *  @param p_com Pointer to common data in all display module.
 *  @param p_tft Pointer to management data of TFT module.
 */
static void update_tft_audio_data(const dsp_com_ctrl_t * const p_com, dsp_tft_ctrl_t * const p_tft)
{
    dsp_audio_t     *p_aud;
    uint32_t        i;
    uint32_t        buf_id;
    int32_t         total_val;
    bool            update_flag;
    int16_t         new_data[DSP_TFT_M3_AUDIO_SAMPLE_PER_100MS];
    uint32_t        valid_data;

    if ((p_com != NULL) && (p_tft != NULL)) {
        p_aud = &p_tft->audio_data;
        /* Stores the display data using the display mode 2. */
        (void) memcpy(&p_aud->m2_buf[0], &p_aud->req_buf[0], sizeof(p_aud->m2_buf));
        p_aud->m2_buf_cnt = sizeof(p_aud->m2_buf)/sizeof(p_aud->m2_buf[0]);
        /* Stores the display data using the display mode 3. */
        buf_id = 0u;
        valid_data = 0u;
        for (i = 0u; i < DSP_TFT_M3_AUDIO_SAMPLE_PER_100MS; i++) {
            /* Calculates the average of 2 channels. */
            total_val = p_aud->req_buf[buf_id] + p_aud->req_buf[buf_id + 1u];
            new_data[i] = (int16_t)((uint32_t)total_val >> SHIFT_1BIT_SIZE);
            buf_id += (DSP_TFT_M2_AUDIO_BUF_SIZE / DSP_TFT_M3_AUDIO_SAMPLE_PER_100MS);
            /* Checks the zero data */
            valid_data |= (uint32_t)new_data[i];
        }
        update_flag = false;
        if (p_com->play_stat == SYS_PLAYSTAT_PAUSE) {
            p_aud->m3_pause_flag = true;
            if (valid_data == 0u) {
                /* Current sampling data are not used. */
            } else {
                update_flag = true;
            }
        } else if (p_com->play_stat == SYS_PLAYSTAT_PLAY) {
            if ((p_aud->m3_pause_flag == true) && 
                (p_aud->m3_sample_cnt > (p_aud->m3_target_cnt + DSP_TFT_M3_AUDIO_SAMPLE_PER_100MS))) {
                /* Current sampling data are not used. */
            } else if ((p_aud->m3_pause_flag == true) && (valid_data == 0u)) {
                /* Current sampling data are not used. */
                p_aud->m3_pause_flag = false;
            } else {
                update_flag = true;
                p_aud->m3_pause_flag = false;
            }
        } else {
            p_aud->m3_pause_flag = false;
        }
        if (update_flag == true) {
            if (p_aud->m3_buf_cnt >= DSP_TFT_M3_AUDIO_BUF_SIZE) {
                p_aud->m3_buf_cnt = DSP_TFT_M3_AUDIO_BUF_SIZE - DSP_TFT_M3_AUDIO_SAMPLE_PER_100MS;
                for (i = 0u; i < p_aud->m3_buf_cnt; i++) {
                    p_aud->m3_buf[i] = p_aud->m3_buf[i + DSP_TFT_M3_AUDIO_SAMPLE_PER_100MS];
                }
            }
            for (i = 0u; i < DSP_TFT_M3_AUDIO_SAMPLE_PER_100MS; i++) {
                p_aud->m3_buf[p_aud->m3_buf_cnt] = new_data[i];
                p_aud->m3_buf_cnt++;
            }
            p_aud->m3_sample_cnt += DSP_TFT_M3_AUDIO_SAMPLE_PER_100MS;
            p_aud->m3_target_cnt += DSP_TFT_M3_AUDIO_SAMPLE_PER_100MS;
        }
    }
}

/** Converts the layer number of TFT module into the layer number of DisplayBase class.
 *
 *  @param layer_id The number of the display layer of TFT module.
 *
 *  @returns 
 *    Returns the layer number of DisplayBase class.
 */
static DisplayBase::graphics_layer_t convert_layer_id(const uint32_t layer_id)
{
    DisplayBase::graphics_layer_t   ret;

    switch (layer_id) {
        case DSP_TFT_LAYER_1:
            ret = DisplayBase::GRAPHICS_LAYER_1;
            break;
        case DSP_TFT_LAYER_0:
        default:
            ret = DisplayBase::GRAPHICS_LAYER_0;
            break;
    }
    return ret;
}

/** Swaps the frame buffers
 *
 *  @param p_info Pointer to VRAM structure
 */
static void swap_frame_buffer(dsp_tftlayer_t * const p_info)
{
    uint32_t                        *p_tmp;
    
    if (p_info != NULL) {
        p_tmp = p_info->p_disp_buf;
        p_info->p_disp_buf = p_info->p_back_buf;
        p_info->p_back_buf = p_tmp;
    }
}
