/*******************************************************************************
* DISCLAIMER
* This software is supplied by Renesas Electronics Corporation and is only
* intended for use with Renesas products. No other uses are authorized. This
* software is owned by Renesas Electronics Corporation and is protected under
* all applicable laws, including copyright laws.
* THIS SOFTWARE IS PROVIDED "AS IS" AND RENESAS MAKES NO WARRANTIES REGARDING
* THIS SOFTWARE, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING BUT NOT
* LIMITED TO WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
* AND NON-INFRINGEMENT. ALL SUCH WARRANTIES ARE EXPRESSLY DISCLAIMED.
* TO THE MAXIMUM EXTENT PERMITTED NOT PROHIBITED BY LAW, NEITHER RENESAS
* ELECTRONICS CORPORATION NOR ANY OF ITS AFFILIATED COMPANIES SHALL BE LIABLE
* FOR ANY DIRECT, INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES FOR
* ANY REASON RELATED TO THIS SOFTWARE, EVEN IF RENESAS OR ITS AFFILIATES HAVE
* BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
* Renesas reserves the right, without notice, to make changes to this software
* and to discontinue the availability of this software. By using this software,
* you agree to the additional terms and conditions found by accessing the
* following link:
* http://www.renesas.com/disclaimer*
* Copyright (C) 2015 Renesas Electronics Corporation. All rights reserved.
*******************************************************************************/

#include "mbed.h"
#include "rtos.h"
#include "misratypes.h"

#include "key.h"
#include "key_cmd.h"
#include "system.h"
#include "display.h"
#include "disp_tft.h"

/*--- Macro definition of mbed-rtos mail ---*/
#define MAIL_QUEUE_SIZE     (3)     /* Queue size */
#define MAIL_PARAM_NUM      (1)     /* Elements number of mail parameter array */

/* key_mail_t */
#define MAIL_PARAM0         (0)     /* Index number of mail parameter array */

/* mail_id = KEY_MAILID_DISP_MODE */
#define MAIL_DISPMODE_MODE  (MAIL_PARAM0)   /* Display mode */

#define RECV_MAIL_TIMEOUT_MS (0u)

/*--- Macro definition of key thread ---*/
#define PROC_CYCLE_SW       (10u)   /* The process cycle of SW module */
#define PROC_CYCLE_TFT      (50u)   /* The process cycle of TFT module */
#define PROC_CYCLE_CMD      (20u)   /* The process cycle of command-line module */
#define PROC_CYCLE_REFRESH  (50u)   /* Refresh cycle of counter */
#define UNIT_TIME_MS        (2u)

#define PROC_CNT_SW         (PROC_CYCLE_SW / UNIT_TIME_MS)      /* Counter for 10ms period */
#define PROC_CNT_TFT        (PROC_CYCLE_TFT / UNIT_TIME_MS)     /* Counter for 50ms period */
#define PROC_CNT_CMD        (PROC_CYCLE_CMD / UNIT_TIME_MS)     /* Counter for 2ms period */
#define PROC_CNT_REFRESH    (PROC_CYCLE_REFRESH / UNIT_TIME_MS) /* Counter for 50ms period */

/*--- Macro definition of SW module ---*/
#define SW0_ACTIVE_LEVEL    (0)
#define SW0_DECISION_TIME   (50u)   /* Time until the decision of the input status. */
#define SW0_DECISION_CNT    (SW0_DECISION_TIME / PROC_CYCLE_SW) /* Counter for 50ms period */

/*--- Macro definition of Touch panel module ---*/
#define TP_IIC_DEV_ADR      (0x55 << 1) /* I2C device address */
#define TP_REG_ADR_NUM      (6u)        /* The number of the reading of the register. */

/* Register address */
#define TP_REG_ADR_FINGER   (0u)        /* Number of touch points[3:0] */
#define TP_REG_ADR_STATUS   (2u)        /* 1st Touch valid / 1st Touch X Position[11:8] / 1st Touch Y Position[11:8] */
#define TP_REG_ADR_TOUCH_XL (3u)        /* 1st Touch X Position[7:0] */
#define TP_REG_ADR_TOUCH_YL (4u)        /* 1st Touch Y Position[7:0] */

/* Mask for the register */
#define TP_MASK_FINGER      (0x0Fu)     /* Used in the finger register */
#define TP_MASK_TOUCH_YH    (0x07u)     /* Used in the status register for Y Position */
#define TP_MASK_TOUCH_XH    (0x70u)     /* Used in the status register for X Position */
#define TP_MASK_VALID       (0x80u)     /* Used in the status register for valid */

#define TP_SHIFT_TOUCH_YH   (8u)        /* Used in combination with the upper and lower bits of the Y position */
#define TP_SHIFT_TOUCH_XH   (4u)        /* Used in combination with the upper and lower bits of the X position */

#define TP_TOUCH_MAX_NUM    (1u)
#define TP_DEFAULT_POS      (0u)

#define TP_RELEASE_NUM      (0u)
#define TP_PRESS_NUM        (1u)

#define TP_MAX_WIDHT        (DSP_TFT_WIDTH - 1u)
#define TP_MAX_HEIGHT       (DSP_TFT_HEIGHT - 1u)

/*--- User defined types of mbed-rtos mail ---*/
typedef enum {
    KEY_MAILID_DUMMY = 0,
    KEY_MAILID_DISP_MODE,           /* Change display mode */
    KEY_MAILID_NUM
} KEY_MAIL_ID;

typedef struct {
    KEY_MAIL_ID     mail_id;
    uint32_t        param[MAIL_PARAM_NUM];
} key_mail_t;

/*--- User defined types ---*/
/* Control data of SW module */
typedef struct {
    uint32_t        sampling_count; /* Sampling count for decision of input. */
    bool            current_status; /* Current input status. true=push, false=release. */
} sw_ctrl_t;

/* Control data of TFT module */
typedef struct {
    uint32_t        disp_mode;
    SYS_KeyCode     prev_key_event;
    uint32_t        prev_key_num;
} tft_ctrl_t;

/* Control data of key thread */
typedef struct {
    sw_ctrl_t       sw_data;
    tft_ctrl_t      tft_data;
    cmd_ctrl_t      cmd_data;
} key_ctrl_t;

static Mail<key_mail_t, MAIL_QUEUE_SIZE> mail_box;

static void sw_init_proc(sw_ctrl_t * const p_ctrl);
static SYS_KeyCode sw_main_proc(sw_ctrl_t * const p_ctrl);
static void tft_init_proc(tft_ctrl_t * const p_ctrl);
static SYS_KeyCode tft_main_proc(tft_ctrl_t * const p_ctrl);
static bool tft_get_key_code(const uint32_t disp_mode, SYS_KeyCode * const p_key_ev, uint32_t * const p_key_num);
static bool send_mail(const KEY_MAIL_ID mail_id, const uint32_t param0);
static bool recv_mail(KEY_MAIL_ID * const p_mail_id, uint32_t * const p_param0);

void key_thread(void const *argument)
{
    static key_ctrl_t   key_data;
    SYS_KeyCode         key_ev;
    SYS_KeyCode         tmp_ev;
    uint32_t            cnt = 0u;
    KEY_MAIL_ID         mail_type;
    uint32_t            mail_param[MAIL_PARAM_NUM];
    bool                result;

    UNUSED_ARG(argument);
    
    /* Initializes the control data of key thread. */
    sw_init_proc(&key_data.sw_data);
    tft_init_proc(&key_data.tft_data);
    cmd_init_proc(&key_data.cmd_data);
    while(1) {
        key_ev = SYS_KEYCODE_NON;
        result = recv_mail(&mail_type, &mail_param[MAIL_PARAM0]);
        if (result == true) {
            if (mail_type == KEY_MAILID_DISP_MODE) {
                /* Changes display mode. */
                key_data.tft_data.disp_mode = mail_param[MAIL_DISPMODE_MODE];
            }
        }

        /* Is it a timing of the SW module processing? */
        if((cnt % PROC_CNT_SW) == 0u) {
            /* Executes main process of SW module. */
            tmp_ev = sw_main_proc(&key_data.sw_data);
            if(tmp_ev != SYS_KEYCODE_NON) {
                key_ev = tmp_ev;
            }
        }
        /* Is it a timing of TFT module processing? */
        if((cnt % PROC_CNT_TFT) == 0u) {
            /* Executes main process of TFT module. */
            tmp_ev = tft_main_proc(&key_data.tft_data);
            if(tmp_ev != SYS_KEYCODE_NON) {
                if(key_ev == SYS_KEYCODE_NON) {
                    /* There is no input from other modules. */
                    key_ev = tmp_ev;
                }
            }
        }
        /* Is it a timing of command-line module processing? */
        if((cnt % PROC_CNT_CMD) == 0u) {
            /* Executes main process of command-line module. */
            tmp_ev = cmd_main_proc(&key_data.cmd_data);
            if(tmp_ev != SYS_KEYCODE_NON) {
                if(key_ev == SYS_KEYCODE_NON) {
                    /* There is no input from other modules. */
                    key_ev = tmp_ev;
                }
            }
        }
        /* Is it a refresh timing of the counter? */
        if(cnt >= PROC_CNT_REFRESH) {
            cnt = 0u;
        }
        /* When the event occurs, this mail is sent to main thread. */
        if(key_ev != SYS_KEYCODE_NON) {
            (void) sys_notify_key_input(key_ev);
        }
        Thread::wait(UNIT_TIME_MS);
        cnt++;
    }
}

bool key_notify_disp_mode(const uint32_t disp_mode)
{
    bool ret = false;

    ret = send_mail(KEY_MAILID_DISP_MODE, disp_mode);

    return ret;
}

/** Initialises SW module
 *
 *  @param p_ctrl Pointer to the control data of SW module.
 */
static void sw_init_proc(sw_ctrl_t * const p_ctrl)
{
    if (p_ctrl != NULL) {
        p_ctrl->sampling_count = 0u;
        p_ctrl->current_status = false;
    }
}

/** Executes the main processing of SW module
 *
 *  @param p_ctrl Pointer to the control data of SW module.
 *
 *  @returns 
 *    Key code.
 */
static SYS_KeyCode sw_main_proc(sw_ctrl_t * const p_ctrl)
{
    SYS_KeyCode         key_ev = SYS_KEYCODE_NON;
    int32_t             pin_level;
    static DigitalIn    sw0(P6_0);

    if (p_ctrl != NULL) {
        pin_level = sw0.read();
        if (pin_level == SW0_ACTIVE_LEVEL) {
            /* SW0 is pushed. */
            if (p_ctrl->sampling_count < SW0_DECISION_CNT) {
                p_ctrl->sampling_count++;
                if (p_ctrl->sampling_count == SW0_DECISION_CNT) {
                    key_ev = SYS_KEYCODE_PLAYPAUSE;
                }
            }
            p_ctrl->current_status = true;
        } else {
            /* SW0 is released. */
            p_ctrl->sampling_count = 0u;
            p_ctrl->current_status = false;
        }
    }
    return key_ev;
}

/** Initialises TFT module
 *
 *  @param p_ctrl Pointer to the control data of TFT module.
 */
static void tft_init_proc(tft_ctrl_t * const p_ctrl)
{
    if (p_ctrl != NULL) {
        /* Initialization of the TFT key variables. */
        /* Initializes it to a status pushing down an invalid key. */
        /* This is to detect the edge of pushing down a key after having released all keys. */
        p_ctrl->disp_mode      = 0u;
        p_ctrl->prev_key_event = SYS_KEYCODE_NON;
        p_ctrl->prev_key_num   = TP_PRESS_NUM;
    }
}

/** Executes the main processing of TFT module
 *
 *  @param p_ctrl Pointer to the control data of TFT module.
 *
 *  @returns 
 *    Key code.
 */
static SYS_KeyCode tft_main_proc(tft_ctrl_t * const p_ctrl)
{
    SYS_KeyCode     key_ev = SYS_KEYCODE_NON;
    SYS_KeyCode     cur_key_ev;
    uint32_t        cur_key_num;
    bool            result;

    if (p_ctrl != NULL) {
        result = tft_get_key_code(p_ctrl->disp_mode, &cur_key_ev, &cur_key_num);
        if (result == true) {
            if ((p_ctrl->prev_key_num == cur_key_num) &&
                (p_ctrl->prev_key_event == cur_key_ev)) {
                /* The touch position on the touch panel did not change. */
            } else {
                /* The touch position on the touch panel changed. */
                if ((p_ctrl->prev_key_num == TP_RELEASE_NUM) && (cur_key_num == TP_PRESS_NUM)) {
                    /* Press check */
                    if (cur_key_ev != SYS_KEYCODE_NON) {
                        key_ev = cur_key_ev;
                        /* Notify the touched TFT key information. */
                        (void) dsp_notify_tft_key(key_ev);
                    }
                    p_ctrl->prev_key_event = cur_key_ev;
                } else {
                    /* Release check */
                    key_ev = SYS_KEYCODE_NON;
                    if (p_ctrl->prev_key_event != SYS_KEYCODE_NON) {
                        /* Notify the touched TFT key information. */
                        (void) dsp_notify_tft_key(key_ev);
                    }
                    p_ctrl->prev_key_event = SYS_KEYCODE_NON;
                }
            }
            p_ctrl->prev_key_num   = cur_key_num;
        } else {
            p_ctrl->prev_key_event = SYS_KEYCODE_NON;
            p_ctrl->prev_key_num   = 0u;
        }
    }
    return key_ev;
}

/** Gets the key code of the touch panel via IIC communication.
 *
 *  @param disp_mode Display mode
 *  @param p_key_ev Pointer to variable to store the event code of the detected key
 *  @param p_key_num Pointer to variable to store the number of the detected key
 *
 *  @returns 
 *    Returns true if the API is successful. Returns false if the API fails.
 */
static bool tft_get_key_code(const uint32_t disp_mode, SYS_KeyCode * const p_key_ev, uint32_t * const p_key_num)
{
    bool        ret = false;
    SYS_KeyCode touch_ev;
    uint32_t    touch_num;
    uint32_t    valid;
    uint32_t    data_h;
    uint32_t    data_l;
    uint32_t    pos_x;
    uint32_t    pos_y;
    int32_t     i2c_err;
    char_t      tp_buf[TP_REG_ADR_NUM];
    static I2C  tft_i2c(I2C_SDA, I2C_SCL);
    
    if ((p_key_ev != NULL) && (p_key_num != NULL)) {
        i2c_err = tft_i2c.read(TP_IIC_DEV_ADR, tp_buf, sizeof(tp_buf));
        if (i2c_err == 0) {
            /* Touch num check */
            touch_num = (uint32_t) tp_buf[TP_REG_ADR_FINGER] & TP_MASK_FINGER;
            if (touch_num == TP_TOUCH_MAX_NUM) {
                /* Valid check */
                valid = (uint32_t) tp_buf[TP_REG_ADR_STATUS] & TP_MASK_VALID;
                if (valid == TP_MASK_VALID) {
                    /* X position */
                    data_h = (uint32_t) tp_buf[TP_REG_ADR_STATUS] & TP_MASK_TOUCH_XH;
                    data_l = (uint32_t) tp_buf[TP_REG_ADR_TOUCH_XL];
                    pos_x  = (data_h << TP_SHIFT_TOUCH_XH) | data_l;
                    /* Converts the touch position into TFT display position. */
                    if (pos_x > TP_MAX_WIDHT) {
                        pos_x = TP_DEFAULT_POS;
                    }

                    /* Y position */
                    data_h = (uint32_t) tp_buf[TP_REG_ADR_STATUS] & TP_MASK_TOUCH_YH;
                    data_l = (uint32_t) tp_buf[TP_REG_ADR_TOUCH_YL];
                    pos_y  = (data_h << TP_SHIFT_TOUCH_YH) | data_l;
                    /* Converts the touch position into TFT display position. */
                    if (pos_y > TP_MAX_HEIGHT) {
                        pos_y = TP_DEFAULT_POS;
                    }

                    /* Converts TFT display position into the key code. */
                    touch_ev = dsp_convert_key(disp_mode, pos_x, pos_y);
                } else {
                    touch_ev = SYS_KEYCODE_NON;
                }
            } else {
                touch_ev = SYS_KEYCODE_NON;
            }
            ret = true;
            *p_key_ev  = touch_ev;
            *p_key_num = touch_num;
        }
    }

    return ret;
}

/** Sends the mail to key thread
 *
 *  @param mail_id Mail ID
 *  @param param0 Parameter 0 of this mail
 *
 *  @returns 
 *    Results of process. true is success. false is failure.
 */
static bool send_mail(const KEY_MAIL_ID mail_id, const uint32_t param0)
{
    bool            ret = false;
    osStatus        stat;
    key_mail_t      * const p_mail = mail_box.alloc();

    if (p_mail != NULL) {
        p_mail->mail_id = mail_id;
        p_mail->param[MAIL_PARAM0] = param0;
        stat = mail_box.put(p_mail);
        if (stat == osOK) {
            ret = true;
        } else {
            (void) mail_box.free(p_mail);
        }
    }
    return ret;
}

/** Receives the mail to key thread
 *
 *  @param p_mail_id Pointer to the variable to store the mail ID
 *  @param p_param0 Pointer to the variable to store the parameter 0 of this mail
 *
 *  @returns 
 *    Results of process. true is success. false is failure.
 */
static bool recv_mail(KEY_MAIL_ID * const p_mail_id, uint32_t * const p_param0)
{
    bool            ret = false;
    osEvent         evt;
    key_mail_t      *p_mail;

    if ((p_mail_id != NULL) && (p_param0 != NULL)) {
        evt = mail_box.get(RECV_MAIL_TIMEOUT_MS);
        if (evt.status == osEventMail) {
            p_mail = (key_mail_t *)evt.value.p;
            if (p_mail != NULL) {
                *p_mail_id = p_mail->mail_id;
                *p_param0 = p_mail->param[MAIL_PARAM0];
                ret = true;
            }
            (void) mail_box.free(p_mail);
        }
    }
    return ret;
}
