/*******************************************************************************
* DISCLAIMER
* This software is supplied by Renesas Electronics Corporation and is only
* intended for use with Renesas products. No other uses are authorized. This
* software is owned by Renesas Electronics Corporation and is protected under
* all applicable laws, including copyright laws.
* THIS SOFTWARE IS PROVIDED "AS IS" AND RENESAS MAKES NO WARRANTIES REGARDING
* THIS SOFTWARE, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING BUT NOT
* LIMITED TO WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
* AND NON-INFRINGEMENT. ALL SUCH WARRANTIES ARE EXPRESSLY DISCLAIMED.
* TO THE MAXIMUM EXTENT PERMITTED NOT PROHIBITED BY LAW, NEITHER RENESAS
* ELECTRONICS CORPORATION NOR ANY OF ITS AFFILIATED COMPANIES SHALL BE LIABLE
* FOR ANY DIRECT, INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES FOR
* ANY REASON RELATED TO THIS SOFTWARE, EVEN IF RENESAS OR ITS AFFILIATES HAVE
* BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
* Renesas reserves the right, without notice, to make changes to this software
* and to discontinue the availability of this software. By using this software,
* you agree to the additional terms and conditions found by accessing the
* following link:
* http://www.renesas.com/disclaimer*
* Copyright (C) 2015 Renesas Electronics Corporation. All rights reserved.
*******************************************************************************/

#include "misratypes.h"
#include "display.h"
#include "disp_graphics.h"
#include "img_tbl.h"

/*--- Macro definition ---*/
/* The number of bytes per pixel. */
#define BYTENUM_PER_PIXEL_ARGB8888  (4u)        /* ARGB8888 : 1pixel = 4byte */
#define BYTENUM_PER_PIXEL_RGB888    (3u)        /* RGB888   : 1pixel = 3byte */

/* Alpha blend */
/* Minimum value of alpha blend. */
#define BLEND_MIN_VAL_ARGB8888      (0x00u)
/* Maximum value of alpha blend. */
#define BLEND_MAX_VAL_ARGB8888      (0xFFu)
/* In the case of this value, the background color is not blended. */
#define NOT_BLEND_VAL_ARGB8888      (BLEND_MAX_VAL_ARGB8888)

#define SHIFT_1BIT_SIZE             (1u)
#define SHIFT_1BYTE_SIZE            (8u)
#define SHIFT_2BYTE_SIZE            (16u)
#define SHIFT_3BYTE_SIZE            (24u)

#define MASK_BIT_SIZE               (0x01u)
#define MASK_BYTE_SIZE              (0xFFu)

#define MAX_LOOP_NUM                (128)       /* fail-safe processing. */

#define COL_CLEAR                   (0x00000000u)

typedef uint32_t (*blend_func_t)(const uint32_t bg_col, const uint32_t fg_col);

static void set_bmp_rgb888(const dsp_tftlayer_t * const p_info, 
    const int32_t start_x, const int32_t start_y, const dsp_bmpinf_t * const p_bmpinf);
static void set_bmp_argb8888(const dsp_tftlayer_t * const p_info, const int32_t start_x, 
    const int32_t start_y, const dsp_bmpinf_t * const p_bmpinf, const blend_func_t p_func);
static uint32_t alpha_blend(const uint32_t bg_col, const uint32_t fg_col);
static uint32_t get_fg_col(const uint32_t bg_col, const uint32_t fg_col);
static void set_font(const dsp_tftlayer_t * const p_info, const int32_t start_x, 
                            const int32_t start_y, const uint32_t * const font_data, 
                            const uint32_t font_width, const uint32_t font_height);

void dsp_clear_all(const dsp_tftlayer_t * const p_info)
{
    if (p_info != NULL) {
        dsp_clear_area(p_info, 0, 0, (int32_t)p_info->width, (int32_t)p_info->height);
    }
}

void dsp_clear_area(const dsp_tftlayer_t * const p_info, const int32_t start_x, 
                 const int32_t start_y, const int32_t size_x, const int32_t size_y)
{
    if (p_info != NULL) {
        dsp_fill_rect(p_info, start_x, start_y, size_x, size_y, COL_CLEAR);
    }
}

void dsp_draw_picture(const dsp_tftlayer_t * const p_info, const int32_t start_x, 
                const int32_t start_y, const int32_t pict_id, const int32_t pict_type)
{
    int32_t         type;
    
    if ((p_info != NULL) && (pict_id >= 0) && (pict_id < DSP_IMG_TBL_SIZE)) {
        if (pict_type == DSP_IMG_FORM_AUTOSEL) {
            type = dsp_bitmap_tbl[pict_id].image_type;
        } else {
            type = pict_type;
        }
        
        switch (type) {
            case DSP_IMG_FORM_RGB888:
                set_bmp_rgb888(p_info, start_x, start_y, &dsp_bitmap_tbl[pict_id]);
                break;
            case DSP_IMG_FORM_ARGB8888_NO_BLEND:
                set_bmp_argb8888(p_info, start_x, start_y, &dsp_bitmap_tbl[pict_id], &get_fg_col);
                break;
            case DSP_IMG_FORM_ARGB8888:
            default:
                set_bmp_argb8888(p_info, start_x, start_y, &dsp_bitmap_tbl[pict_id], &alpha_blend);
                break;
        }
    }
    return;
}

void dsp_fill_rect(const dsp_tftlayer_t * const p_info, const int32_t start_x, 
    const int32_t start_y, const int32_t size_x, const int32_t size_y, const uint32_t fg_col)
{
    uint32_t        *p_buff;            /* Pointer to VRAM. */
    uint32_t        stride_pix;         /* The horizontal number of bytes in VRAM. */
    int32_t         pos_x;              /* The X position in VRAM. */
    int32_t         pos_y;              /* The Y position in VRAM. */
    
    if (p_info != NULL) {
        /* Calculates the horizontal number of bytes. */
        /* The horizontal number of bytes is a multiple of 4. */
        /* VRAM */
        stride_pix = p_info->stride / sizeof(uint32_t);
        for (pos_y = start_y; pos_y < (start_y + size_y); pos_y++) {
            if ((pos_y >= 0) && (pos_y < (int32_t)p_info->height)) {
                /* Sets the position of (0, pos_y) in VRAM. */
                p_buff = &p_info->p_back_buf[stride_pix * (uint32_t)pos_y];
                for (pos_x = start_x; pos_x < (start_x + size_x); pos_x++) {
                    if((pos_x >= 0) && (pos_x < (int32_t)p_info->width)) {
                        p_buff[pos_x] = fg_col;
                    }
                }
            }
        }
    }
    return;
}

void dsp_draw_text30x34(const dsp_tftlayer_t * const p_info, const int32_t start_x, 
                                    const int32_t start_y, const char_t * const p_str)
{
    const uint32_t          *p_font_data;
    int32_t                 str_index;
    int32_t                 i;
    int32_t                 pos_x;
    const int32_t           pos_y = start_y;
    const uint32_t          pos_x_offset = DSP_IMG_WS_FONT30X34;
    uint32_t                chr_code;
    const dsp_fntinf_t      *p_font_info;
    
    if ((p_info != NULL) && (p_str != NULL)) {
        str_index = 0;
        pos_x = start_x;
        while ((str_index < MAX_LOOP_NUM) && ((int32_t)p_str[str_index] != '\0')) {
            chr_code = (uint32_t)p_str[str_index];
            for (i = 0; i < (int32_t)(sizeof(dsp_img_fonttbl30x34)/sizeof(dsp_img_fonttbl30x34[0])); i++) {
                p_font_info = &dsp_img_fonttbl30x34[i];
                if ((chr_code >= p_font_info->start_char) && (chr_code <= p_font_info->end_char)) {
                    p_font_data = &p_font_info->p_font[(chr_code - p_font_info->start_char) * DSP_IMG_HS_FONT30X34];
                    set_font(p_info, pos_x, pos_y, p_font_data, DSP_IMG_WS_FONT30X34, DSP_IMG_HS_FONT30X34);
                }
            }
            pos_x += (int32_t)pos_x_offset;
            str_index ++;
        }
    }
}

void dsp_draw_text15x17(const dsp_tftlayer_t * const p_info, const int32_t start_x, 
                                    const int32_t start_y, const char_t * const p_str)
{
    const uint32_t          *p_font_data;
    int32_t                 str_index;
    int32_t                 i;
    int32_t                 pos_x;
    const int32_t           pos_y = start_y;
    const uint32_t          pos_x_offset = DSP_IMG_WS_FONT15X17;
    uint32_t                chr_code;
    const dsp_fntinf_t      *p_font_info;
    
    if ((p_info != NULL) && (p_str != NULL)) {
        str_index = 0;
        pos_x = start_x;
        while ((str_index < MAX_LOOP_NUM) && ((int32_t)p_str[str_index] != '\0')) {
            chr_code = (uint32_t)p_str[str_index];
            for (i = 0; i < (int32_t)(sizeof(dsp_img_fonttbl15x17)/sizeof(dsp_img_fonttbl15x17[0])); i++) {
                p_font_info = &dsp_img_fonttbl15x17[i];
                if ((chr_code >= p_font_info->start_char) && (chr_code <= p_font_info->end_char)) {
                    p_font_data = &p_font_info->p_font[(chr_code - p_font_info->start_char) * DSP_IMG_HS_FONT15X17];
                    set_font(p_info, pos_x, pos_y, p_font_data, DSP_IMG_WS_FONT15X17, DSP_IMG_HS_FONT15X17);
                }
            }
            pos_x += (int32_t)pos_x_offset;
            str_index ++;
        }
    }
}

/** Draws BMP image of the RGB888 format in VRAM.
 *
 *  @param p_info Pointer to VRAM structure
 *  @param start_x Display position X of BMP image
 *  @param start_y Display position Y of BMP image
 *  @param p_bmpinf Pointer to information of BMP image
 */
static void set_bmp_rgb888(const dsp_tftlayer_t * const p_info, 
    const int32_t start_x, const int32_t start_y, const dsp_bmpinf_t * const p_bmpinf)
{
    uint32_t        *p_buff;        /* Pointer to VRAM. */
    uint32_t        stride_pix;     /* The horizontal number of bytes in VRAM. */
    int32_t         pos_x;          /* The X position in VRAM. */
    int32_t         pos_y;          /* The Y position in VRAM. */
    const uint8_t   *p_img;         /* Pointer to BMP image. */
    uint32_t        img_h_byte;     /* The horizontal number of bytes in the BMP image. */
    uint32_t        img_index;
    uint32_t        cnt;
    uint32_t        shift_num;
    uint32_t        col_data;
    
    if ((p_info != NULL) && (p_bmpinf != NULL)) {
        /* Calculates the horizontal number of bytes. */
        /* The horizontal number of bytes is a multiple of 4. */
        /* VRAM */
        stride_pix = p_info->stride / sizeof(uint32_t);
        /* BMP image */
        img_h_byte = (((BYTENUM_PER_PIXEL_RGB888 * p_bmpinf->size_x) + 
                       (sizeof(uint32_t) - 1u)) / sizeof(uint32_t)) * sizeof(uint32_t);
        for (pos_y = start_y; pos_y < (start_y + (int32_t)p_bmpinf->size_y); pos_y++) {
            if ((pos_y >= 0) && (pos_y < (int32_t)p_info->height)) {
                /* Sets the position of (0, pos_y) in VRAM. */
                p_buff = &p_info->p_back_buf[stride_pix * (uint32_t)pos_y];
                /* Sets the position of (start_x, pos_y) in BMP image. */
                p_img = &p_bmpinf->p_image[img_h_byte * ((p_bmpinf->size_y - 1u) - (uint32_t)(pos_y - start_y))];
                img_index = 0u;
                for (pos_x = start_x; pos_x < (start_x + (int32_t)p_bmpinf->size_x); pos_x++) {
                    if((pos_x >= 0) && (pos_x < (int32_t)p_info->width)) {
                        col_data = BLEND_MAX_VAL_ARGB8888 << SHIFT_3BYTE_SIZE;
                        shift_num = 0u;
                        for (cnt = 0u; cnt < BYTENUM_PER_PIXEL_RGB888; cnt++) {
                            col_data |= ((uint32_t)p_img[img_index + cnt] << shift_num);
                            shift_num += SHIFT_1BYTE_SIZE;
                        }
                        p_buff[pos_x] = col_data;
                    }
                    img_index += BYTENUM_PER_PIXEL_RGB888;
                }
            }
        }
    }
}

/** Draws BMP image of the ARGB8888 format in VRAM.
 *
 *  @param p_info Pointer to VRAM structure
 *  @param start_x Display position X of BMP image
 *  @param start_y Display position Y of BMP image
 *  @param p_bmpinf Pointer to information of BMP image
 *  @param p_func Pointer to the data blend function
 */
static void set_bmp_argb8888(const dsp_tftlayer_t * const p_info, const int32_t start_x, 
        const int32_t start_y, const dsp_bmpinf_t * const p_bmpinf, const blend_func_t p_func)
{
    uint32_t        *p_buff;        /* Pointer to VRAM. */
    uint32_t        stride_pix;     /* The horizontal number of bytes in VRAM. */
    int32_t         pos_x;          /* The X position in VRAM. */
    int32_t         pos_y;          /* The Y position in VRAM. */
    const uint8_t   *p_img;         /* Pointer to BMP image. */
    uint32_t        img_h_byte;     /* The horizontal number of bytes in the BMP image. */
    uint32_t        img_index;
    uint32_t        cnt;
    uint32_t        shift_num;
    uint32_t        col_data;
    
    if ((p_info != NULL) && (p_bmpinf != NULL) && (p_func != NULL)) {
        /* Calculates the horizontal number of bytes. */
        /* The horizontal number of bytes is a multiple of 4. */
        /* VRAM */
        stride_pix = p_info->stride / sizeof(uint32_t);
        /* BMP image */
        img_h_byte = BYTENUM_PER_PIXEL_ARGB8888 * p_bmpinf->size_x;
        for (pos_y = start_y; pos_y < (start_y + (int32_t)p_bmpinf->size_y); pos_y++) {
            if ((pos_y >= 0) && (pos_y < (int32_t)p_info->height)) {
                /* Sets the position of (0, pos_y) in VRAM. */
                p_buff = &p_info->p_back_buf[stride_pix * (uint32_t)pos_y];
                /* Sets the position of (start_x, pos_y) in BMP image. */
                p_img = &p_bmpinf->p_image[img_h_byte * ((p_bmpinf->size_y - 1u) - (uint32_t)(pos_y - start_y))];
                img_index = 0u;
                for (pos_x = start_x; pos_x < (start_x + (int32_t)p_bmpinf->size_x); pos_x++) {
                    if((pos_x >= 0) && (pos_x < (int32_t)p_info->width)) {
                        col_data = 0;
                        shift_num = 0u;
                        for (cnt = 0; cnt < BYTENUM_PER_PIXEL_ARGB8888; cnt++) {
                            col_data |= ((uint32_t)p_img[img_index + cnt] << shift_num);
                            shift_num += SHIFT_1BYTE_SIZE;
                        }
                        p_buff[pos_x] = p_func(p_buff[pos_x],col_data);
                    }
                    img_index += BYTENUM_PER_PIXEL_ARGB8888;
                }
            }
        }
    }
}

/** Blends foreground color with background color.
 *
 *  @param bg_col Background color
 *  @param fg_col Foreground color
 */
static uint32_t alpha_blend(const uint32_t bg_col, const uint32_t fg_col)
{
    uint32_t        ret_col;
    /* Blended data */
    uint32_t        new_b;
    uint32_t        new_g;
    uint32_t        new_r;
    uint32_t        new_a;
    /* Background data */
    uint32_t        bg_b;
    uint32_t        bg_g;
    uint32_t        bg_r;
    uint32_t        bg_a;
    /* Foreground data */
    uint32_t        fg_b;
    uint32_t        fg_g;
    uint32_t        fg_r;
    const uint32_t  fg_a  = (fg_col >> SHIFT_3BYTE_SIZE) & MASK_BYTE_SIZE;
    
    if (fg_a == BLEND_MIN_VAL_ARGB8888) {
        /* The data is not blended. A background data is used. */
        ret_col = bg_col;
    } else if (fg_a == BLEND_MAX_VAL_ARGB8888) {
        /* The data is not blended. A foreground data is used. */
        ret_col = fg_col;
    } else {
        /* A background data and a foreground data is blended. */
        /* Foreground data */
        fg_b  = (fg_col                    ) & MASK_BYTE_SIZE;
        fg_g  = (fg_col >> SHIFT_1BYTE_SIZE) & MASK_BYTE_SIZE;
        fg_r  = (fg_col >> SHIFT_2BYTE_SIZE) & MASK_BYTE_SIZE;
        
        /* Background data */
        bg_b  = (bg_col                    ) & MASK_BYTE_SIZE;
        bg_g  = (bg_col >> SHIFT_1BYTE_SIZE) & MASK_BYTE_SIZE;
        bg_r  = (bg_col >> SHIFT_2BYTE_SIZE) & MASK_BYTE_SIZE;
        bg_a  = (~fg_a                     ) & MASK_BYTE_SIZE;
        
        /* Blended data */
        /* The division calculation by 255 needs the long processing time. */
        /* Therefore we execute the division calculation by 256 using shift calculation.  */
        /* An error in calculation occurs to divide it by 256, */
        /* but an error is maximum 1 very small. */
        new_b = ((fg_b * fg_a) + (bg_b * bg_a)) >> SHIFT_1BYTE_SIZE;
        new_g = ((fg_g * fg_a) + (bg_g * bg_a)) >> SHIFT_1BYTE_SIZE;
        new_r = ((fg_r * fg_a) + (bg_r * bg_a)) >> SHIFT_1BYTE_SIZE;
        new_a = fg_a;
        
        ret_col  =  new_b;
        ret_col |= (new_g << SHIFT_1BYTE_SIZE);
        ret_col |= (new_r << SHIFT_2BYTE_SIZE);
        ret_col |= (new_a << SHIFT_3BYTE_SIZE);
    }
    return ret_col;
}

/** Gets the foreground color.
 *
 *  @param bg_col Background color
 *  @param fg_col Foreground color
 */
static uint32_t get_fg_col(const uint32_t bg_col, const uint32_t fg_col)
{
    UNUSED_ARG(bg_col);
    return fg_col;
}

/** Draws font image in VRAM.
 *
 *  @param p_info Pointer to VRAM structure
 *  @param start_x Display position X of font image
 *  @param start_y Display position Y of font image
 *  @param font_data Pointer to font data
 *  @param font_width The width of font data
 *  @param font_height The height of font data
 */
static void set_font(const dsp_tftlayer_t * const p_info, const int32_t start_x, 
                            const int32_t start_y, const uint32_t * const font_data, 
                            const uint32_t font_width, const uint32_t font_height)
{
    uint32_t        *p_buff;
    uint32_t        data_val;
    uint32_t        stride_pix;
    int32_t         pos_x;
    int32_t         pos_y;
    
    if ((p_info != NULL) && (font_data != NULL)) {
        /* Calculates the horizontal number of bytes. */
        /* The horizontal number of bytes is a multiple of 4. */
        stride_pix = p_info->stride / sizeof(uint32_t);
        for (pos_y = start_y; pos_y < (start_y + (int32_t)font_height); pos_y++) {
            if ((pos_y >= 0) && (pos_y < (int32_t)p_info->height)) {
                /* Sets the position of (0, pos_y) in VRAM. */
                p_buff = &p_info->p_back_buf[stride_pix * (uint32_t)pos_y];
                /* Sets the position of (start_x, pos_y) in font image. */
                data_val = font_data[pos_y - start_y];
                for (pos_x = start_x; pos_x < (start_x + (int32_t)font_width); pos_x++) {
                    if ((data_val & MASK_BIT_SIZE) == DSP_IMG_FONT_FG_VAL) {
                        if((pos_x >= 0) && (pos_x < (int32_t)p_info->width)) {
                            p_buff[pos_x] = DSP_IMG_FONT_FG_COL;
                        }
                    }
                    data_val >>= SHIFT_1BIT_SIZE;
                }
            }
        }
    }
}
