/*

Copyright (c) 2012-2014 RedBearLab

Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
and associated documentation files (the "Software"), to deal in the Software without restriction, 
including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, 
and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, 
subject to the following conditions:
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE 
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

/*
 *    SimpleControls
 *
 *    SimpleControls works with the BLEController iOS/Android App.
 *    The mbed's pin can acts as DIGITAL_IN, DIGITAL_OUT, PWM, SERVO, ANALOG_IN.
 *    The sketch is to show you how to control the pin as one of the abilities.
 *    Note that not every pin can own all of the abilities.
 *    You can change the following macro to specify which pin to be controlled.
 *
 *      DigitalInOut    DIGITAL_OUT_PIN( PTD4 );
 *      DigitalInOut    DIGITAL_IN_PIN( PTA12 );
 *      PwmOut          pwm( PTA4 );
 *      Servo           myservo( PTA13 );
 *      AnalogIn        analog( PTB0 );
 *
 *    The application will report the state of DIGITAL_IN_PIN and ANALOG_IN_PIN to App.
 *    The App can control the state of DIGITAL_OUT_PIN / PWM_PIN / SERVO_PIN
 */
 
// Import libraries 
#include "Arduino.h"
#include "BLEPeripheral.h"
#include "Servo.h"

Serial serial(USBTX, USBRX);

DigitalInOut    DIGITAL_OUT_PIN( PTD4 );
DigitalInOut    DIGITAL_IN_PIN( PTA12 );
PwmOut          pwm( PTA4 );
Servo           myservo( PTC9 );
AnalogIn        analog( PTB0 );

// The SPI construct, REQN and RDYN IO construct should be modified manually
// It depend on the board you are using and the REQN&RDYN configuration on BLE Shield
SPI spi(PTD2, PTD3, PTD1);      
DigitalInOut BLE_RDY(PTA13);  
DigitalInOut BLE_REQ(PTD5);   

unsigned char buf[16] = {0};
unsigned char len = 0;

unsigned char analog_enabled = 0;
unsigned char old_state = LOW;

/*----- BLE Utility -------------------------------------------------------------------------*/
// create peripheral instance, see pinouts above
BLEPeripheral            blePeripheral        = BLEPeripheral(&BLE_REQ, &BLE_RDY, NULL);

// create service
BLEService               uartService          = BLEService("713d0000503e4c75ba943148f18d941e");

// create characteristic
BLECharacteristic    txCharacteristic = BLECharacteristic("713d0002503e4c75ba943148f18d941e", BLENotify, 20);
BLECharacteristic    rxCharacteristic = BLECharacteristic("713d0003503e4c75ba943148f18d941e", BLEWriteWithoutResponse, 20);
/*--------------------------------------------------------------------------------------------*/
 
int main()
{    
    serial.baud(115200);
    serial.printf("Serial begin!\r\n");
    
    pinMode(&DIGITAL_OUT_PIN, OUTPUT);
    pinMode(&DIGITAL_IN_PIN, INPUT_PULLUP);
    
    /*----- BLE Utility ---------------------------------------------*/
    // set advertised local name and service UUID
    blePeripheral.setLocalName("BLE Shield");
    
    blePeripheral.setAdvertisedServiceUuid(uartService.uuid());
    
    // add service and characteristic
    blePeripheral.addAttribute(uartService);
    blePeripheral.addAttribute(rxCharacteristic);
    blePeripheral.addAttribute(txCharacteristic);
    
    // begin initialization
    blePeripheral.begin();
    /*---------------------------------------------------------------*/
    
    serial.printf("BLE UART Peripheral begin!\r\n");
    
    while(1)
    {
        BLECentral central = blePeripheral.central();
        
        if (central) 
        {
            // central connected to peripheral
            serial.printf("Connected to central\r\n");
            
            while (central.connected()) 
            {
                // central still connected to peripheral
                if (rxCharacteristic.written()) 
                {
                    unsigned char len = rxCharacteristic.valueLength();
                    const unsigned char *val = rxCharacteristic.value();
                    serial.printf("didCharacteristicWritten, Length: %d\r\n", len); 
                
                    unsigned char i = 0;
                    while(i<len)
                    {
                        unsigned char data0 = val[i++];
                        unsigned char data1 = val[i++];
                        unsigned char data2 = val[i++]; 

                        if (data0 == 0x01)  // Command is to control digital out pin
                        {
                            if (data1 == 0x01)
                                digitalWrite(&DIGITAL_OUT_PIN, HIGH);
                            else
                                digitalWrite(&DIGITAL_OUT_PIN, LOW);
                        }
                        else if (data0 == 0xA0) // Command is to enable analog in reading
                        {
                            if (data1 == 0x01)
                                analog_enabled = 1;
                            else
                                analog_enabled = 0;
                        }
                        else if (data0 == 0x02) // Command is to control PWM pin
                        {
                            float percent = (1.0 / 255) * data1; // Convert 0~255 to 0%~100%
                            pwm.write(percent);
                        }
                        else if (data0 == 0x03)  // Command is to control Servo pin
                        {
                            myservo.write(data1);
                        }
                        else if (data0 == 0x04)
                        {
                            analog_enabled = 0;
                            myservo.write(0);
                            pwm.write(0.0);
                            digitalWrite(&DIGITAL_OUT_PIN, LOW);
                        }
                    } 
                }
                
                if (analog_enabled)  // if analog reading enabled
                {
                    // Read and send out
                    unsigned short value = analog.read_u16();
                    
                    const unsigned char val[3] = {0x0B, value >> 8, value};
                    txCharacteristic.setValue(val, 3);
                }
                
                // If digital in changes, report the state
                if (digitalRead(&DIGITAL_IN_PIN) != old_state)
                {
                    old_state = digitalRead(&DIGITAL_IN_PIN);
                    
                    if (digitalRead(&DIGITAL_IN_PIN) == HIGH)
                    {
                        const unsigned char val[3] = {0x0A, 0x01, 0x00};
                        txCharacteristic.setValue(val, 3);   
                    }
                    else
                    {
                        const unsigned char val[3] = {0x0A, 0x00, 0x00};
                        txCharacteristic.setValue(val, 3);
                    }
                }
            }
            
            analog_enabled = 0;
            digitalWrite(&DIGITAL_OUT_PIN, LOW);
        
            // central disconnected
            serial.printf("Disconnected from central\r\n");
        }
    }
}
