/****************************************************************************************************************************************************
Titre : Full_Sensors
Auteur : Jeanne Baumier, Anaïs Auberval et Thomas Broussard
Date : 29/11/17
Plateforme : STM32L432KC
Projet : PlantSigfox - Monitoring de plante à distance 
--------------------------------
Description : 
Ce programme permet d'effectuer des relèves des différents capteurs météorologiques toute les T secondes. 
Les données reçues par les capteurs sont traitées puis transmises via un module Sigfox TD1208R sur un serveur sigfox (https://backend.sigfox.com)
puis redirigées vers un dashboard (https://plantsigfox.eu-gb.mybluemix.net/ui)

L'ensemble des composants peuvent être configurés dans le fichier Mapping.h, à savoir : 
- l'activation ou la désactivation d'un composant
- les paramètres d'étalonnage et de mesure des capteurs
- les broches programmées pour le microcontrôleur

****************************************************************************************************************************************************/

// ------------------------------------------------------------------------------
//                                  LIBRAIRIES
// ------------------------------------------------------------------------------
#include "mbed.h"       
#include "DS18B20.h"            // OneWire Temperature
#include "DHT.h"                // DHT22
#include "TSL2561.h"            // Lux
#include "TCS34725.h"           // RGB
#include "Adafruit_SSD1306.h"   // OLED
#include "Mapping.h"
#include "WakeUp.h"             // Low Power Energy
// ------------------------------------------------------------------------------
//                          CLASSES ET STRUCTURES
// ------------------------------------------------------------------------------

#ifdef OLED_ACTIF
    // sub-class SPI : 
    // Caractéristiques par défaut de la communication SPI avec le SSD1306
    class SPIPreInit : public SPI
    {
    public:
        SPIPreInit(PinName mosi, PinName miso, PinName clk) : SPI(mosi,miso,clk)
        {
            format(8,3); 
            frequency(2000000);
        };
    };
#endif

// ------------------------------------------------------------------------------
//                    INITIALISATION DES GPIO DU MICROCONTROLEUR
// ------------------------------------------------------------------------------
#ifdef DEBUG_UART_ACTIF
    // Liaison UART pour Debug
    Serial pc(USBTX, USBRX);
#endif 

#ifdef SIGFOX_ACTIF
    // Liaison UART Sigfox
    Serial Sigfox(SIGFOX_RX, SIGFOX_TX); 
#endif

#ifdef OLED_ACTIF
    // Ecran OLED (communication SPI)
    SPIPreInit gSpi(MOSI,MISO,CLK);
    Adafruit_SSD1306_Spi OLED(gSpi,DC,RST_SPI,CS,HAUTEUR_OLED,LARGEUR_OLED);
#endif

#ifdef GROVE_MOIST_ACTIF
    // Capteur d'humidité du sol Grove
    AnalogIn Capteur_Moisture(GROVE_MOIST_PIN);
#endif

#ifdef ONEWIRE_TEMP_ACTIF
    //Capteur de Température du sol One Wire CRC
    DS18B20 Temperature(true, true, false, ONEWIRE_TEMP_PIN); // Temperature( crcOn, useAddress, parasitic, mbed pin )
#endif

#ifdef DHT_ACTIF
    //Capteur Température/Humidité de l'air Grove DHT22
    DHT sensor(DHT_PIN,SEN51035P); // Use the SEN11301P sensor
#endif

#ifdef I2C_ACTIF
    I2C          i2c(I2C_SDA,I2C_SCL);    // SDA, SCL
    //Capteur de Luminosité et RGB Adafruit (tous les deux connectés au bus i2c)
    #ifdef LUX_ACTIF
        TSL2561      lum(i2c);  
    #endif
    #ifdef RGB_ACTIF  
        TCS34725     rgb(i2c);   
    #endif 
#endif

#ifdef BATTERIE_ACTIF
    //Batterie
    AnalogIn Batterie(BATTERIE_PIN);
#endif


// ------------------------------------------------------------------------------
//                           PROGRAMME PRINCIPAL
// ------------------------------------------------------------------------------
int main()
{
/*********************************
         Variables locales
*********************************/    
// Divers
int i;
int line = 0; 

// Variables permettant de stocker les données à relever
float VBAT = 0;
float HumidSol = 0; 
float TempSol  = 0;
float TempAir  = 0;
float HumidAir = 0;
float Lumiere  = 0;
float Niveau_Batterie = 101;
int   Clear    = 0;
int   Red      = 0;
int   Green    = 0;
int   Blue     = 0; 

wait(2);
/*********************************
         Initialisation
*********************************/   
#ifdef BATTERIE_ACTIF
    float BATTERIE_MIN = 33;
    float BATTERIE_MAX = 42; 
#endif

#ifdef OLED_ACTIF
    // Initialisation de l'écran 
    OLED.fillRect(0,0,LARGEUR_OLED,HAUTEUR_OLED,BLACK); 
    OLED.setTextCursor(0,10); OLED.printf("==================="); 
    OLED.setTextCursor(0,20); OLED.printf("    PlantSigfox    ");  
    OLED.setTextCursor(0,30); OLED.printf("   Projet ei2i-4   ");  
    OLED.setTextCursor(0,40); OLED.printf("Jeanne Anais Thomas");
    OLED.setTextCursor(0,50); OLED.printf("==================="); 
    OLED.display();
#endif

#ifdef DEBUG_UART_ACTIF
    // Initialisation des communication UART
    pc.format(8,SerialBase::None,1);
    pc.baud(9600);
#endif

#ifdef SIGFOX_ACTIF
    Sigfox.format(8,SerialBase::None,1);
    Sigfox.baud(9600);
#endif

#ifdef ONEWIRE_TEMP_ACTIF
    // Initialisation du capteur de température
    while (!Temperature.initialize());    // on attend que le capteur soit initialisé avant de pouvoir effectuer des mesures
#endif

#ifdef RGB_ACTIF
    // Initialisation du TCS34725
    rgb.init_RGB();
#endif

// Attente de 5 secondes avant le démarrage du programme
wait(5);


/*********************************
         Boucle principale
*********************************/  
while(1) 
{   
    /*********************************
              Mesure des ADC
    *********************************/  

    // On effectue une moyenne sur une quantité de relevés ( pour avoir une mesure fiable )
    #if defined(GROVE_MOIST_ACTIF) || defined(BATTERIE_ACTIF)
        Niveau_Batterie = 0;    
        HumidSol = 0;
        
        for (i=0; i < NB_MESURES; i++)
        {
            #ifdef BATTERIE_ACTIF
                // Batterie 
                VBAT = 100 * Batterie; // Lecture de l'entrée ADC
                Niveau_Batterie += ((VBAT - BATTERIE_MIN) / (BATTERIE_MAX - BATTERIE_MIN)) * 100 ; 
            #endif
            
            #ifdef GROVE_MOIST_ACTIF
                // Humidité du Sol (Grove)
                HumidSol += ((Capteur_Moisture  - GROVE_MOIST_MIN) / (GROVE_MOIST_MAX - GROVE_MOIST_MIN)) * 100;
            #endif
            
            wait(TEMPS_MESURE);
        }
    #endif
    
    // Moyennage du résultat
    #ifdef GROVE_MOIST_ACTIF
        HumidSol /= NB_MESURES; 
    #endif
    
    #ifdef BATTERIE_ACTIF
        Niveau_Batterie /= NB_MESURES;
        // Limitation de la batterie
        if (Niveau_Batterie >100) Niveau_Batterie = 100;
        // Si la batterie est déconnectée, on doit le faire remarquer (code : 101)
        if (Niveau_Batterie < 0) Niveau_Batterie = 101;
    #endif
    
    /*********************************
        Température du sol (OneWIRE)
    *********************************/  
    #ifdef ONEWIRE_TEMP_ACTIF
        Temperature.setResolution(twelveBit); // Le choix de la résolution se fait avant le relevé des données
        TempSol = Temperature.readTemperature(); // relevé de données 
    #endif
    
    /*********************************
     Température/Humidité Air (Grove)
    *********************************/      
    #ifdef DHT_ACTIF  
        int err;
        wait(1); // wait 1 second for device stable status
        do
        {
            err = sensor.readData();
            wait(1);
        }
        while (err != 0);
        if (err == 0)
        {
            TempAir = sensor.ReadTemperature(CELCIUS);
            HumidAir = sensor.ReadHumidity();
        }
    #endif
    
    /*********************************
          Luminosité / RGB(Adafruit)
    *********************************/  
    #ifdef LUX_ACTIF
        Lumiere = COEF_LUX * lum.lux();
    #endif
    
    #ifdef RGB_ACTIF
        rgb.GET_RGB(&Clear,&Red,&Green,&Blue);
        #ifdef RGB_1_OCTET
            // On réduit la précision des données brutes en divisant par 256 pour n'avoir qu'un octet seulement
            Clear   /= 256;
            Red     /= 256;
            Green   /= 256;
            Blue    /= 256;
        #endif
    #endif
    
    /*********************************
            Affichage OLED
    *********************************/  
    #ifdef OLED_ACTIF
        // OLED Reset
        OLED.fillRect(0,0,LARGEUR_OLED,HAUTEUR_OLED,BLACK);
        line = 0;
        
        // Sol : 
        #if defined(GROVE_MOIST_ACTIF) && defined(ONEWIRE_TEMP_ACTIF)
            OLED.setTextCursor(0,line);
            OLED.printf("Sol = %d \tC // %d \t/\t",(int)TempSol,(int)HumidSol);
            line +=10;
        #endif
        
        // Air :
        #ifdef DHT_ACTIF
            OLED.setTextCursor(0,line);
            OLED.printf("Air = %d \tC // %d \t/\t",(int)TempAir,(int)HumidAir);
            line +=10;
        #endif
        
        // Intensité lumineuse
        #if defined(LUX_ACTIF) && defined(I2C_ACTIF) 
            OLED.setTextCursor(0,line);
            OLED.printf("Light = %d", (int)Lumiere);
            OLED.setTextCursor(110,line);           
            OLED.printf("Lux");
            line +=10;
        #endif
        
        // Couleurs RGB
        #if defined(RGB_ACTIF) && defined(I2C_ACTIF) 
            OLED.setTextCursor(0,line);
            OLED.printf("RGB = %d/%d/%d",Red,Green,Blue);
            line +=10;   
            OLED.setTextCursor(0,line);
            OLED.printf("Clear = %d",Clear);
            line +=10;     
        #endif
        
        // Batterie
        #ifdef BATTERIE_ACTIF
            OLED.setTextCursor(0,line);
            if (((int)Niveau_Batterie >= 0) && ((int)Niveau_Batterie <= 100)) 
            {
                OLED.printf("Batterie : %d", (int)Niveau_Batterie);
                OLED.setTextCursor(110,line);
                OLED.printf("\t/\t");
            }
            else 
            {
                OLED.printf("Prise Secteur");
            }
        #endif
        
        // Display
        OLED.display();
    #endif
    
    /****************************************************************************
         Envoi des données via le module Sigfox (dépend du format des données)
    *****************************************************************************/
    #ifdef SIGFOX_ACTIF
        #if defined(RGB_2_OCTET) && !defined(RGB_1_OCTET)
            // trame 1 : [T°Air , Hum Air , T°Sol , Hum Sol , Lumiere , Batterie]
            Sigfox.printf("AT$SS=%02x %02x %02x %02x %04x %02x \r\n",(int)TempAir,(int)HumidAir,(int)TempSol, (int)HumidSol, (int)Lumiere,(int)Niveau_Batterie);
            // trame 2 : [Clear, Red, Green, Blue]
            Sigfox.printf("AT$SS=%04x %04x %04x %04x \r\n",Clear, Red, Green, Blue);
        #endif
        
        #if defined(RGB_1_OCTET) && !defined(RGB_2_OCTET)
            // trame : [T°Air , Hum Air , T°Sol , Hum Sol , Lumiere , Batterie, Clear, Red, Green, Blue]
            Sigfox.printf("AT$SS=%02x %02x %02x %02x %04x %02x %02x %02x %02x %02x \r\n",(int)TempAir,(int)HumidAir,(int)TempSol, (int)HumidSol, (int)Lumiere,(int)Niveau_Batterie,Clear, Red, Green, Blue);
        #endif
    #endif
    
    // Envoi des données sur le debug UART (optionnel)
    #ifdef DEBUG_UART_ACTIF
        #if defined(RGB_2_OCTET) && !defined(RGB_1_OCTET)
            // trame 1 : [T°Air , Hum Air , T°Sol , Hum Sol , Lumiere , Batterie]
            pc.printf("AT$SS=%02x %02x %02x %02x %04x %02x \r\n",(int)TempAir,(int)HumidAir,(int)TempSol, (int)HumidSol, (int)Lumiere,(int)Niveau_Batterie);
            // trame 2 : [Clear, Red, Green, Blue]
            pc.printf("AT$SS=%04x %04x %04x %04x \r\n",Clear, Red, Green, Blue);
        #endif
        
        #if defined(RGB_1_OCTET) && !defined(RGB_2_OCTET)
            // trame : [T°Air , Hum Air , T°Sol , Hum Sol , Lumiere , Batterie, Clear, Red, Green, Blue]
            pc.printf("AT$SS=%02x %02x %02x %02x %04x %02x %02x %02x %02x %02x \r\n",(int)TempAir,(int)HumidAir,(int)TempSol, (int)HumidSol, (int)Lumiere,(int)Niveau_Batterie,Clear, Red, Green, Blue);
        #endif
    #endif
    
    
    /*********************************
        Mise en veille de l'appareil
    *********************************/  
    #if defined(OLED_ACTIF) && defined(OLED_VEILLE)
        // Test : éteindre l'écran au bout d'un temps donné
        wait (DUREE_AFFICHAGE);
        OLED.fillRect(0,0,LARGEUR_OLED,HAUTEUR_OLED,BLACK);
        OLED.display();
          
        WakeUp::set_ms((TEMPS_RELEVE-DUREE_AFFICHAGE)* 1000);
        deepsleep();
    #else
        WakeUp::set_ms(TEMPS_RELEVE * 1000);
        deepsleep();
    #endif
    
    /***********************************************************************************
            On attend le réveil de l'appareil pour exécuter le prochain relevé
    ***********************************************************************************/ 
   }
}


void my_wakeup()
{
}