/****************************************************************************************************************************************************
Titre : Mapping.h
Auteur : Jeanne Baumier, Anaïs Auberval et Thomas Broussard
Date : 29/11/17
Plateforme : STM32L432KC
Projet : PlantSigfox - Monitoring de plante à distance 
--------------------------------
Description : 
Configuration du programme principal : 
- activation ou désactivation des composants
- paramètres d'étalonnage et de mesure des capteurs
- sélection des broches pour le microcontrôleur
****************************************************************************************************************************************************/

#ifndef __MAPPING_H__
#define __MAPPING_H__

    // ##############################################################################
    //                         DEFINITION DES COMPOSANTS A UTILISER
    // ##############################################################################
    // Utilisation du composant (decommenter pour activer le composant)  
    // Composants généraux (affichage, communication...)
    #define OLED_ACTIF
    #define SIGFOX_ACTIF
    #define I2C_ACTIF
    //#define DEBUG_UART_ACTIF
    
    // Capteurs / Monitoring
    #define DHT_ACTIF
    #define GROVE_MOIST_ACTIF
    #define ONEWIRE_TEMP_ACTIF
    #define LUX_ACTIF
    #define RGB_ACTIF
    #define BATTERIE_ACTIF
    
    // ##############################################################################
    //                   CONFIGURATION ET MAPPING DE LA CARTE STM32L432KC
    // ##############################################################################
    
    // ------------------------------------------------------------------------------
    // Ecran OLED
    // Reference : OLED 0.96" (Adafruit)
    // ------------------------------------------------------------------------------
    
    // Mise en veille de l'écran (commenter pour laisser l'ecran actif tout le temps)
    #define OLED_VEILLE
    
    // Dimensions de l'ecran (pixels)
    #define LARGEUR_OLED       128
    #define HAUTEUR_OLED       64
    
    // Mapping
    #define MISO                NC
    #define MOSI                D2
    #define CLK                 A1
    #define DC                  D9  
    #define RST_SPI             D10
    #define CS                  D11
    
    // ------------------------------------------------------------------------------
    // Nom :  Capteur Luminosite et RGB
    // Reference : TSL2561 et TCS34725 (Adafruit)
    // Communication : i2c
    // ------------------------------------------------------------------------------
    
    // Etalonnage du capteur de luminosite
    #define COEF_LUX            25
    
    // Precision du capteur RGB en octets (decommenter celui a utiliser)
    // il faut oblitagoirement en definir un seul des deux pour utiliser le RGB
    #define RGB_1_OCTET
    //#define RGB_2_OCTET
    
    // Mapping i2c
    #define I2C_SDA             D4
    #define I2C_SCL             D5
    
    // ------------------------------------------------------------------------------
    // Capteur d'Humidite/Temperature de l'air
    // Reference : DHT22 (Seeed)
    // ------------------------------------------------------------------------------
    
    // Mapping
    #define DHT_PIN             A0
    
    // ------------------------------------------------------------------------------
    // Capteur d'Humidite du Sol
    // Reference : Moisture Sensor (Grove)
    // ------------------------------------------------------------------------------
    
    // Etalonnage du capteur
    #define GROVE_MOIST_MAX     0.55
    #define GROVE_MOIST_MIN     0
    
    // Mapping
    #define GROVE_MOIST_PIN     A3
    
    // ------------------------------------------------------------------------------
    // Capteur de Temperature du Sol 
    // Reference : One Wire CRC
    // ------------------------------------------------------------------------------
    
    // Mapping
    #define ONEWIRE_TEMP_PIN    A2
    
    // ------------------------------------------------------------------------------
    // Module Sigfox
    // Reference :  TD1208 (SnootLab)
    // ------------------------------------------------------------------------------
   
    // Mapping
    #define SIGFOX_RX           D1
    #define SIGFOX_TX           D0
    
    // ------------------------------------------------------------------------------
    // Batterie
    // ------------------------------------------------------------------------------
    // Etalonnage : défini dans le programme principal
     
    // Mapping
    #define BATTERIE_PIN        A6   
       
    // ------------------------------------------------------------------------------
    // Mesures realisees par les ADC
    // ------------------------------------------------------------------------------
    // Nombre de point de mesure utilisees pour moyenner les resultats
    #define NB_MESURES          100
    
    // Temps entre chaque point de mesure (Secondes)
    #define TEMPS_MESURE        0.1 
    
    // Temps entre chaque releve de mesure (Secondes)
    #define TEMPS_RELEVE        600
    
    // Temps d'affichage a l'ecran (Secondes)
    // important : doit etre inferieur a TEMPS_RELEVE 
    #define DUREE_AFFICHAGE     30
    
    // ##############################################################################
    //      Verification des paramètres lors de la compilation du programme
    // ##############################################################################
    
    // On s'assure que les timings soient coherents 
    #if defined(OLED_VEILLE) && (TEMPS_RELEVE <= DUREE_AFFICHAGE)
        #error "TEMPS_RELEVE ne peut pas etre inferieur a DUREE_AFFICHAGE !"
    #endif

    // Verification des paramètres RGB
    #if defined(RGB_1_OCTET) && defined(RGB_2_OCTET)
        #error "RGB_1_OCTET et RGB_2_OCTET ne peuvent pas etre definis en meme temps !"
    #endif
    
    #if defined(RGB_ACTIF) && !defined(RGB_1_OCTET) && !defined(RGB_2_OCTET)
        #error "Le capteur RGB est actif mais aucun format de donnees n'a ete choisi !"
        #error "Pensez a decommenter une des lignes RGB_OCTET"
    #endif

#endif