/**
******************************************************************************
* @file    makeFilter.cpp
* @author  Hemant Kapoor
* @date    18-July-2019
* @brief   Make filter class
******************************************************************************
**/
#include <math.h>
#include "./makeFilter.h"

#define PI               3.14159265358979f
using namespace MakeFilterName;

MakeFilter::MakeFilter(int ord, float freq, float sampleFreq)
{
    m_order = ord;
    m_rawAlpha1 = freq / sampleFreq;
    m_poleMask = 0xffffffff;
    m_rawAlpha2 = m_rawAlpha1;
    compute_s();
    normalize();
    compute_z();
    expandpoly();   
}


void MakeFilter::compute_s() 
{ /* compute S-plane poles for prototype LP filter */
    m_numPoles = 0;
    int i;
    for (i = 0; i < 2 * m_order; i++) 
    {
        double img;
        if(m_order == 1)
        {
            img = (i  * PI) / m_order;
        }
        else
        {
            img = ((i + 0.5) * PI) / m_order;
        }
        //complexStruct s = {exp(0.0),exp(img)};
        complexStruct s = {0.0,img};
        complexStruct expS = s.Exp();
        choosepole(expS);
    }
}

void MakeFilter::choosepole(complexStruct z) 
{
    if (z.Real < 0.0) 
    {
        if ((m_poleMask & 0x01) == 0x01)
            m_spoles[m_numPoles++] = z;
        m_poleMask >>= 1;
    }
}

void MakeFilter::normalize() 
{
    int i;
    m_warpedAlpha1 = tan(PI * m_rawAlpha1) / PI;
    m_warpedAlpha2 = tan(PI * m_rawAlpha2) / PI;
    complexStruct w1 = {PI * 2 * m_warpedAlpha1, 0};
    //complexStruct w2 = {PI * 2 * m_warpedAlpha2, 0};
    for (i = 0; i < m_numPoles; i++)
    {
        m_spoles[i] = m_spoles[i] * w1;
    }
}

void MakeFilter::compute_z() /* given S-plane poles, compute Z-plane poles */
{
    int i;
    for (i = 0; i < m_numPoles; i++) 
    { /* use bilinear transform */
        complexStruct top, bot;
        top = m_ctwo + m_spoles[i];
        bot = m_ctwo - m_spoles[i];
        m_zPoles[i] = top / bot;
        m_zZeros[i] = m_cmone;
    }
}


void MakeFilter::expandpoly() /* given Z-plane poles & zeros, compute top & bot polynomials in Z, and then recurrence relation */
{
    complexStruct topcoeffs[MAXPOLES + 1] = {0};
    complexStruct botcoeffs[MAXPOLES + 1] = {0};
    complexStruct zfc;
    int i;

    expand(m_zZeros, topcoeffs);
    expand(m_zPoles, botcoeffs);
    m_dcGain = evaluate(topcoeffs, botcoeffs, m_numPoles, m_cone);
    complexStruct st{0.0, PI * (m_rawAlpha1 + m_rawAlpha2)}; /* "jwT" for centre freq. */
    zfc = st.Exp();
    m_fcGain = evaluate(topcoeffs, botcoeffs, m_numPoles, zfc);
    m_hfGain = evaluate(topcoeffs, botcoeffs, m_numPoles, m_cmone);
    for (i = 0; i <= m_numPoles; i++) 
    {
        m_xcoeffs[i] = topcoeffs[i].Real / botcoeffs[m_numPoles].Real;
        m_ycoeffs[i] = -(botcoeffs[i].Real / botcoeffs[m_numPoles].Real);
    }
}

void MakeFilter::expand(complexStruct pz[], complexStruct coeffs[]) 
{ /* compute product of poles or zeros as a polynomial of z */
    int i;
    coeffs[0] = m_cone;
    for (i = 0; i < m_numPoles; i++)
        coeffs[i + 1] = m_czero;
    for (i = 0; i < m_numPoles; i++)
        multin(pz[i], coeffs);
    #ifdef NEVER
    /* check computed coeffs of z^k are all real */
    for (i = 0; i < m_numPoles + 1; i++) 
    {
        if (Math.Abs(coeffs[i].Imaginary) > 1e-10) 
        {
            MessageBox.Show("mkfilter: coeff of z^" + i.ToString() + " is not real; poles are not complex conjugates\n");
        }
    }
    #endif
}

complexStruct MakeFilter::evaluate(complexStruct topco[], complexStruct botco[], int np, complexStruct z) 
{ /* evaluate response, substituting for z */
        return (eval(topco, np, z) / eval(botco, np, z));
}

complexStruct MakeFilter::eval(complexStruct coeffs[], int np, complexStruct z)
{ /* evaluate polynomial in z, substituting for z */
    complexStruct sum;
    int i;
    sum = m_czero;
    for (i = np; i >= 0; i--)
        sum = (sum * z) + coeffs[i];
    return sum;
}

void MakeFilter::multin(complexStruct w, complexStruct coeffs[]) 
{ /* multiply factor (z-w) into coeffs */
    complexStruct nw; int i;
    nw = w.Negate();
    for (i = m_numPoles; i >= 1; i--)
        coeffs[i] = (nw * coeffs[i]) + coeffs[i - 1];
    coeffs[0] = nw * coeffs[0];
}
