
/*
//To use ARM DSP blocks

// include https://os.mbed.com/users/mbed_official/code/mbed-dsp/ in project


    #include "arm_math.h"
    int CMSIS_Stages = order/2  // (order must be even)
    float m_forwardFilterStateArray[4*CMSIS_Stages];
    float m_filterCoeff[CMSIS_Stages*4+1];
    arm_biquad_casd_df1_inst_f32 filterStructure;

  Filter init for order 2:
    m_filterCoeff[0] = xval[0] / gain;
    m_filterCoeff[1] = xval[1] / gain;
    m_filterCoeff[2] = xval[2] / gain;
    m_filterCoeff[3] = yVal[1];
    m_filterCoeff[4] = yVal[0];
    memset(forwardFilterStateArray, 0, 4 * CMSIS_Stages * sizeof(float32_t));
    arm_biquad_cascade_df1_init_f32(&filterStructure, CMSIS_Stages, m_filterCoeff, forwardFilterStateArray);

  To filter data
    arm_biquad_cascade_df1_f32(&filterStructure, float* input, float* output, numberOfPoints);

// see https://www.keil.com/pack/doc/CMSIS/DSP/html/group__BiquadCascadeDF1.html for details of how these work.


*/

#include "LowPassFilter.h"
#include "makeFilter.h"
#include "mbed.h"
#include "BufferedSerial.h"
extern BufferedSerial pc;

LowPassFilter::LowPassFilter()
{
    m_inValues = NULL;
    m_outValues = NULL;
    m_inScale = NULL;
    m_outValues = NULL;
    m_enable = false;
    m_order = 0;
    m_lastIn = 0;
    m_lastOut =&m_lastIn;
}

bool LowPassFilter::makeFilter(int order, float frequency, float sampleRate)
{
    if (order > MakeFilterName::MAXORDER)
        order = MakeFilterName::MAXORDER;

    if (order != m_order) { // change buffer sizes
        freeBuffers();      // clear existing buffers
        if (order) {        // if need new buffers
            if (!createBuffers(order)) {   // create buffers and check for errors
                return false;              // failed to create buffers.
            }
        }
    }

    // buffers are now correct size for order

    if (order==0) { // order 0 - bypass the filter
        m_enable = false;
        m_order = 0;
        m_lastOut =&m_lastIn;
        return true;
    }

    // have correct size buffers. order != 0

    // calculate filter perameters and copy to local buffers.
    MakeFilter *newFilter = new MakeFilter(order, frequency, sampleRate);
    if (newFilter) {
        double gain = newFilter->getGain();
        for (int i=0; i<(order+1); i++) {
            m_inScale[i] = newFilter->getXCoeff()[i] / gain;
            m_outScale[i] = newFilter->getYCoeff()[i];
            m_inValues[i] = 0;
            m_outValues[i] = 0;
        }
        delete newFilter;
    } else {            // failed to alocate memory needed to calculate filter values.
        freeBuffers();  // free memory.
        return false;
    }

    m_enable = true;
    m_order = order;
    m_lastOut = &m_outValues[m_order];
    return true;
}

// frees dynamically alocated buffers and sets filter to bypass mode.
void LowPassFilter::freeBuffers()
{
    if (m_inValues) {
        free(m_inValues);
        m_inValues = NULL;
    }
    if (m_outValues) {
        free(m_outValues);
        m_outValues = NULL;
    }
    if (m_inScale) {
        free(m_inScale);
        m_inScale = NULL;
    }
    if (m_outScale) {
        free(m_outScale);
        m_outScale = NULL;
    }
    m_order = 0;
    m_enable = false;
    m_lastOut =&m_lastIn;
}

bool LowPassFilter::createBuffers(int order)
{
    m_inValues = (float*)malloc((order+1)*sizeof(float));
    m_outValues = (float*)malloc((order+1)*sizeof(float));
    m_inScale = (float*)malloc((order+1)*sizeof(float));
    m_outScale = (float*)malloc((order+1)*sizeof(float));
    if (!m_outScale) {
        freeBuffers();
        return false;
    }
    return true;
}

LowPassFilter::LowPassFilter(int order, float frequency, float sampleRate)
{
    makeFilter(order, frequency, sampleRate);
}

LowPassFilter::~LowPassFilter()
{
    freeBuffers();
}

float LowPassFilter::addPoint(float input)
{
    m_lastIn = input;
    if (!m_enable) {
        return input;
    }

    for (int i = 0; i < m_order; i++) {
        m_inValues[i] = m_inValues[i + 1];
        m_outValues[i] = m_outValues[i + 1];
    }
    m_inValues[m_order] = input;
    m_outValues[m_order] = 0;

    for (int i = 0; i < m_order + 1; i++) {
        m_outValues[m_order] += m_inValues[i] * m_inScale[i];
    }
    for (int i = 0; i < m_order; i++) {
        m_outValues[m_order] += m_outValues[i] * m_outScale[i];
    }
    return m_outValues[m_order];
}