/* This is a sample application program for WlanBP3595 library. */
/* This program works as TCP socket sever, and this program sends a message */
/* when a connection is accepted. The setting of the WLAN is WPA2+AES. */

/*
This program works with the following libraries.
  mbed-rtos                                : revision 84
  EthernetInterface_GR-PEACH_WlanBP3595STA : revision 50
*/

#include "mbed.h"
#include "rtos.h"
#include "EthernetInterface.h"
#include "WlanBP3595.h"

/* Please change the following macro definition to your setting. */
#define WLAN_SSID               ("SSIDofYourAP")                // SSID
#define WLAN_PSK                ("PSKofYourAP")                 // PSK(Pre-Shared Key)
#define SERVER_IP               ("192.168.1.200")               // Server IP address
#define SERVER_PORT             (50000)                         // TCP server socket port number
#define SUBNET_MASK             ("255.255.255.0")               // Subnet mask
#define DEFAULT_GATEWAY         ("192.168.1.1")                 // Default gateway
#define CONNECT_TIMEOUT         (3*60*1000)                     // Connect-Timeout in ms
#define SEND_MESSAGE            ("Hello, world!\r\n")           // Send-message

static void _wlan_inf_callback(uint8_t ucType, uint16_t usWid, uint16_t usSize, uint8_t *pucData);
static int _wlan_connect(void);

DigitalOut  red_led(LED1);              // On: error
DigitalOut  green_led(LED2);            // On: WLAN has been connected

/** Main function
 *
 */
int main() {
    EthernetInterface   eth;
    TCPSocketServer     server;
    TCPSocketConnection connection;
    uint32_t            status;
    int                 ret;

    /* Initialize WlanBP3595 */
    ret = WlanBP3595_Init(_wlan_inf_callback);
    if (ret != 0) {
        /* error */
        red_led = 1;
        while (1) { Thread::wait(1000); }
    }

    /* Wait until WLAN_BP3595_START */
    while (1) {
        Thread::wait(200);
        status = WlanBP3595_GetWlanSts();
        if (status == WLAN_BP3595_START) {
            break;
        }
    }

    /* Initialize EthernetInterface */
    ret = eth.init(SERVER_IP, SUBNET_MASK, DEFAULT_GATEWAY);
    if (ret != 0) {
        /* error */
        red_led = 1;
        while (1) { Thread::wait(1000); }
    }

    /* Connect(WLAN) */
    ret = _wlan_connect();
    if (ret != 0) {
        /* error */
        red_led = 1;
        while (1) { Thread::wait(1000); }
    }

    /* Connect(EthernetInterface) */
    ret = eth.connect(CONNECT_TIMEOUT);
    if (ret != 0) {
        /* error */
        red_led = 1;
        while (1) { Thread::wait(1000); }
    }

    /* Bind and listen */
    server.bind(SERVER_PORT);
    server.listen();

    /* Loop */
    while (1) {
        /* Accept */
        server.accept(connection);
        printf("Connection from: %s\n", connection.get_address());

        /* Send a message */
        connection.send_all((char *)SEND_MESSAGE, sizeof(SEND_MESSAGE)-1);

        /* Close */
        connection.close();
    }
}

/** WLAN Information callback function
 *
 */
static void _wlan_inf_callback(uint8_t ucType, uint16_t usWid, uint16_t usSize, uint8_t *pucData)
{
    if (ucType == 'I')
    {
        if (usWid == 0x0005)    // WID_STATUS
        {
            if (pucData[0] == 0x01)     // CONNECTED
            {
                /* Notify the EthernetInterface driver that WLAN has been connected */
                WlanBP3595_Connected();
                green_led = 1;
            }
            else
            {
                /* Notify the EthernetInterface driver that WLAN has been disconnected */
                WlanBP3595_Disconnected();
                green_led = 0;
            }
        }
    }
}

/** WLAN connecting function
 *
 */
static int _wlan_connect(void)
{
    grp_u8              ucWidData8;     // 8bit wid data
    grp_wld_byte_array  tBAWidData;     // byte array wid data
    int                 ret;

    /* Set BSS type */
    ucWidData8 = 0x00;  // BSS-STA(Infrastructure)
    ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_BSS_TYPE, &ucWidData8);
    if (ret != 0) {
        /* error */
        return -1;
    }

    /* Set SSID */
    tBAWidData.pucData = (grp_u8 *)WLAN_SSID;
    tBAWidData.ulSize  = strlen((char *)tBAWidData.pucData);
    ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_SSID, &tBAWidData);
    if (ret != 0) {
        /* error */
        return -1;
    }

    /* Set 11i mode */
    ucWidData8 = 0x01 |     // Encryption is enable
                 0x10 |     // WPA2 is enable
                 0x20;      // CCMP(AES) is enable
    ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_11I_MODE, &ucWidData8);
    if (ret != 0) {
        /* error */
        return -1;
    }

    /* Set PSK */
    tBAWidData.pucData = (grp_u8 *)WLAN_PSK;
    tBAWidData.ulSize  = strlen((char *)tBAWidData.pucData);
    ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_11I_PSK, &tBAWidData);
    if (ret != 0) {
        /* error */
        return -1;
    }

    return 0;
}
