#include "cisme.h"
#include "wifi_events.h"
#include "wifi.h"
#include "debug.h"
#include "pb_encode.h"
#include "pb_decode.h"
#include "messages.pb.h"
#include "light.h"
#include "pump.h"
#include "presens.h"
#include "calibration.h"
#include "experiments.h"
#include "battery.h"

#ifdef USE_WIFI

#define FILE_BLOCK_SIZE 128

// Struct to encode payload in generic message
typedef struct {
    const pb_field_t* fields;
    void* msg;
} PbEncodeMessage;

// Struct to get payload from generic message
typedef struct {
    uint8_t* msg;
    size_t msgLen;
} PbDecodeMessage;

typedef bool (*Encoder)(pb_ostream_t *stream, const pb_field_t *field, void * const *arg);

char startErrors = 0;

// Files data
static int32_t fileId;
static int32_t blockId;
static FILE* curFile;
static uint32_t fileLen;
static int32_t numBlocks;

// Experiments data
static int  currentExpId;
static bool isExpExecuting;

static LocalFileSystem local("local");

static void wifiEventsSendFinishExpInd(int32_t id, char* fileName);

static int getNoSetBits(int value)
{
    int result = 0;

    while (value != 0) {
        value &= (value - 1);
        result++;
    }

    return result;
}

static bool pbEncodeMsg(pb_ostream_t* stream, const pb_field_t* field, void* const* arg)
{
    if (!pb_encode_tag_for_field(stream, field)) {
        return false;
    }

    if (*arg == NULL) {
        pb_encode_varint(stream, 0);
        return true;
    }

    PbEncodeMessage* message = (PbEncodeMessage*)*arg;

    if (!pb_encode_submessage(stream, message->fields, message->msg)) {
        return false;
    }

    return true;
}

static bool pbDecodeMsg(pb_istream_t* stream, const pb_field_t* field, void** arg)
{
    uint8_t* src = (uint8_t*)stream->state;

    PbDecodeMessage* message = (PbDecodeMessage*)*arg;
    message->msg = (uint8_t*)stream->state;
    message->msgLen = stream->bytes_left;
    stream->bytes_left = 0;
    stream->state = src + message->msgLen;

    return true;
}

static bool pbEncodeString(pb_ostream_t* stream, const pb_field_t* field, void* const* arg)
{
    char* str = (char*)*arg;

    if (!pb_encode_tag_for_field(stream, field)) {
        return false;
    }

    if (!pb_encode_string(stream, (uint8_t*)str, strlen(str))) {
        return false;
    }

    return true;
}

static bool pbEncodeErrors(pb_ostream_t* stream, const pb_field_t* field, void* const* arg)
{
    static const char* errorsText[4] = {"Low battery level. Device will be disabled soon.",
                                        "Zero pump RPM. Please, check RPM wire.",
                                        "Failed to read isfet.sys file.",
                                        "Failed to read dsparams.sys file."
                                       };
    int  errorNo   = 0;
    char errorList = **((char**)arg);

    while ((1 << errorNo) <= errorList) {
        if ((errorList & (1 << errorNo)) == 0) {
            errorNo++;
            continue;
        }

        pbEncodeString(stream, field, (void**)&(errorsText[errorNo]));
        errorNo++;
    }

    return true;
}

static bool pbEncodeFiles(pb_ostream_t* stream, const pb_field_t* field, void* const* arg)
{
    DIR* dir = opendir("/msc/data/");
    struct dirent* entry = NULL;

    while ((entry = readdir(dir)) != NULL) {
        if (strstr(entry->d_name, ".csv") == NULL) {
            continue;
        }

        if (!pb_encode_tag_for_field(stream, field)) {
            return false;
        }

        if (!pb_encode_string(stream, (uint8_t*)entry->d_name, strlen(entry->d_name))) {
            return false;
        }
    }

    return true;
}

static bool pbEncodeFileBlock(pb_ostream_t* stream, const pb_field_t* field, void* const* arg)
{
    if (!pb_encode_tag_for_field(stream, field)) {
        return false;
    }

    uint64_t size = (blockId < numBlocks) ? FILE_BLOCK_SIZE : (fileLen % FILE_BLOCK_SIZE);

    if (!pb_encode_varint(stream, size)) {
        return false;
    }

    if (stream->callback != NULL) {
        size_t result = fread(stream->state, 1, size, curFile);
        if (result != size) {
            return false;
        }

        uint8_t* src = (uint8_t*)stream->state;
        stream->state = src + size;
    }

    stream->bytes_written += size;

    return true;
}

static bool pbDecodeString(pb_istream_t* stream, const pb_field_t* field, void** arg)
{
    char* str = (char*)*arg;
    size_t len = stream->bytes_left;

    if (!pb_read(stream, (uint8_t*)str, len)) {
        return false;
    }

    str[len] = '\0';

    return true;
}

static bool pbDecodeProgramSteps(pb_istream_t* stream, const pb_field_t* field, void** arg)
{
    uint8_t* stepIx = (uint8_t*)*arg;

    cisme_startExperimentMsg_Step stepMsg = cisme_startExperimentMsg_Step_init_default;

    if (!pb_decode(stream, cisme_startExperimentMsg_Step_fields, &stepMsg)) {
        return false;
    }

    if (!stepMsg.has_duration) {
        return false;
    }

    DEBUG1("Program step: lightLevel=%d, duration=%d, phCutoff=%2.3f, o2Cutoff=%3.0f",
           stepMsg.lightLevel, stepMsg.duration, stepMsg.phCutoff, stepMsg.o2Cutoff);

    pgm[*stepIx][1] = stepMsg.lightLevel;
    pgm[*stepIx][2] = stepMsg.duration;
    pgm[*stepIx][3] = stepMsg.phCutoff;
    pgm[*stepIx][4] = stepMsg.o2Cutoff;
    if (stepMsg.has_pumpSpeed) {
        pgm[*stepIx][0] = stepMsg.pumpSpeed;
    }

    (*stepIx)++;
    return true;
}

// Encode and send message to app
static void wifiEventsSendMsg(cisme_MessageType type, const pb_field_t* fields, void* msg)
{
    static uint8_t msgBuffer[WIFI_MESSAGE_MAX_LENGTH];

    cisme_genericMsg genericMsg = cisme_genericMsg_init_default;
    genericMsg.type = type;
    genericMsg.payload.funcs.encode = &pbEncodeMsg;
    pb_ostream_t stream = pb_ostream_from_buffer(msgBuffer, WIFI_MESSAGE_MAX_LENGTH);

    if (fields != NULL && msg != NULL) {
        PbEncodeMessage message = {fields, msg};
        genericMsg.payload.arg = &message;
    }

    if (!pb_encode(&stream, cisme_genericMsg_fields, &genericMsg)) {
        ERROR("Failed to encode message");
        return;
    }

    if (debugGetCurrLvl() >= DEBUG_LEVEL_2) {
        DEBUG2("Start of message----------------------------------------------");
        for (int sendByteIt= 0; sendByteIt < stream.bytes_written; sendByteIt++) {
            DEBUG2("Byte %d=%d", sendByteIt, msgBuffer[sendByteIt]);
        }
        DEBUG2("End of message----------------------------------------------");
    }

    wifiSendMessage(msgBuffer, stream.bytes_written);

    wait(WIFI_COMM_TIME / 1000.0);
}

// Collect data and send heartbeat message
static void wifiEventsSendBeatInd(void)
{
    if (isExpExecuting) {
        return;
    }

    cisme_beatDataMsg beatMsg = cisme_beatDataMsg_init_default;
    beatMsg.has_battery = true;
    beatMsg.has_temperature = true;
    beatMsg.has_ph = true;
    beatMsg.has_o2 = true;
    beatMsg.has_lightLevel = true;
    beatMsg.has_pumpRpm = true;

    beatMsg.battery = batteryGet();

    float presensData[PRESENS_RES_LENGTH];
    presensGetData(presensData);
    beatMsg.temperature = PHTEMP;
    beatMsg.ph = pHCorrected;
    beatMsg.o2 = presensData[2];
    beatMsg.lightLevel = lightRead();
    beatMsg.pumpRpm = pumpSpeed();

    DEBUG1("Send BEAT_DATA_IND: battery=%3.1f, temperature=%2.1f, ph=%3.2f, o2=%3.0f, lightLevel=%d, pumpRpm=%d",
           beatMsg.battery, beatMsg.temperature, beatMsg.ph, beatMsg.o2, beatMsg.lightLevel, beatMsg.pumpRpm);
    wifiEventsSendMsg(cisme_MessageType_BEAT_DATA_IND, cisme_beatDataMsg_fields, &beatMsg);
}

// Send instrument identifier to app
static void wifiEventsHandleIdReq(void)
{
    DEBUG1("Receive INSTRUMENT_ID_REQ");

    cisme_instrumentIdMsg idRsp = cisme_instrumentIdMsg_init_default;
    idRsp.id.funcs.encode = &pbEncodeString;
    idRsp.id.arg = instrumentId;

    DIR* dir = opendir("/local");
    struct dirent* entry = NULL;
    while ((entry = readdir(dir)) != NULL) {
        DEBUG1("binary=%s", entry->d_name);
        if (strstr(entry->d_name, ".BIN") == NULL) {
            continue;
        }

        idRsp.binary.funcs.encode = &pbEncodeString;
        idRsp.binary.arg = entry->d_name;
        break;
    }


    static char buildTime[sizeof(__DATE__) + sizeof(__TIME__) + 1];
    sprintf(buildTime, "%s %s", __DATE__, __TIME__);
    idRsp.timestamp.funcs.encode = &pbEncodeString;
    idRsp.timestamp.arg = buildTime;

    char* name = entry->d_name;
    DEBUG1("Send INSTRUMENT_ID_RSP: id=%s, binary=%s, timestamp=%s", instrumentId, name, buildTime);

    wifiEventsSendMsg(cisme_MessageType_INSTRUMENT_ID_RSP, cisme_instrumentIdMsg_fields, &idRsp);
    closedir(dir);
}

// Handle time synchronization
static void wifiEventsHandleTimeRsp(uint8_t* msg, size_t msgLen)
{
    if (msg == NULL || msgLen == 0) {
        ERROR("Empty GET_TIME_RSP");
        return;
    }

    cisme_timeMsg timeMsg = cisme_timeMsg_init_default;
    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);

    if (!pb_decode(&stream, cisme_timeMsg_fields, &timeMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    if (!timeMsg.has_time) {
        ERROR("Missing time field in GET_TIME_RSP");
        return;
    }

    DEBUG1("Receive GET_TIME_RSP: time=%d", timeMsg.time);

    INFO("Set time to %s", ctime((time_t*)&timeMsg.time));
    set_time(timeMsg.time);
}

// Fill pump speed message and send to app
static void wifiEventsSendPumpRsp(cisme_MessageType type)
{
    cisme_pumpSpeedMsg speedMsg = cisme_pumpSpeedMsg_init_default;
    speedMsg.has_speed = true;
    speedMsg.speed = pumpCurrentIntensity();

    DEBUG1("Send %s: speed=%d", (type == cisme_MessageType_SET_PUMP_RSP) ? "SET_PUMP_RSP" : "GET_PUMP_RSP", speedMsg.speed);
    wifiEventsSendMsg(type, cisme_pumpSpeedMsg_fields, &speedMsg);
}

// Handle request to set pump speed
static void wifiEventsHandleSetPumpReq(uint8_t* msg, size_t msgLen)
{
    cisme_pumpSpeedMsg pumpMsg = cisme_pumpSpeedMsg_init_default;

    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);

    if (!pb_decode(&stream, cisme_pumpSpeedMsg_fields, &pumpMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    DEBUG1("Receive SET_PUMP_REQ: speed=%d", pumpMsg.speed);

    pumpSetRpm(pumpMsg.speed);

    // Send response
    wifiEventsSendPumpRsp(cisme_MessageType_SET_PUMP_RSP);
}

// Handle request to get pump speed
static void wifiEventsHandleGetPumpReq(void)
{
    DEBUG1("Receive GET_PUMP_REQ");
    // Send response
    wifiEventsSendPumpRsp(cisme_MessageType_GET_PUMP_RSP);
}

// Fill light level message and send to app
static void wifiEventsSendLightRsp(cisme_MessageType type)
{
    cisme_lightLevelMsg levelMsg = cisme_lightLevelMsg_init_default;
    levelMsg.has_level = true;
    levelMsg.level = lightRead();

    DEBUG1("Send %s: level=%d", (type == cisme_MessageType_SET_PUMP_RSP) ? "SET_LIGHT_RSP" : "GET_LIGHT_RSP", levelMsg.level);
    wifiEventsSendMsg(type, cisme_lightLevelMsg_fields, &levelMsg);
}

// Handle request to set light level
static void wifiEventsHandleSetLightReq(uint8_t* msg, size_t msgLen)
{
    cisme_lightLevelMsg levelMsg = cisme_lightLevelMsg_init_default;
    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);

    if (!pb_decode(&stream, cisme_lightLevelMsg_fields, &levelMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    DEBUG1("Receive SET_LIGHT_REQ: level=%d", levelMsg.level);

    if ((levelMsg.level < LIGHT_MIN_INTENSITY) || (levelMsg.level > LIGHT_MAX_INTENSITY)) {
        ERROR("Incorrect light level: %d", levelMsg.level);
        return;
    }

    lightSet(levelMsg.level);

    // Send response
    wifiEventsSendLightRsp(cisme_MessageType_SET_LIGHT_RSP);
}

// Handle request to get light level
static void wifiEventsHandleGetLightReq(void)
{
    DEBUG1("Receive GET_LIGHT_REQ");
    // Send response
    wifiEventsSendLightRsp(cisme_MessageType_GET_LIGHT_RSP);
}

// Send temperature calibration factor
static void wifiEventsSendTempCalibrationRsp(float bValue)
{
    cisme_tempCalibrationRspMsg tempRsp = cisme_tempCalibrationRspMsg_init_default;
    tempRsp.has_bValue = true;
    tempRsp.bValue = bValue;

    DEBUG1("Send TEMP_CALIBRATION_RSP: bValue=%2.6f", tempRsp.bValue);
    wifiEventsSendMsg(cisme_MessageType_TEMP_CALIBRATION_RSP, cisme_tempCalibrationRspMsg_fields, &tempRsp);
}

// Handle request to calibrate temperature
static void wifiEventsHandleTempCalibrationReq(uint8_t* msg, size_t msgLen)
{
    cisme_tempCalibrationMsg tempMsg = cisme_tempCalibrationMsg_init_default;
    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);

    if (!pb_decode(&stream, cisme_tempCalibrationMsg_fields, &tempMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    if (!tempMsg.has_temperature) {
        ERROR("Missing temperature field in TEMP_CALIBRATION_REQ");
        return;
    }

    DEBUG1("Receive TEMP_CALIBRATION_REQ: temperature=%2.3f, lightLevel=%d, pumpSpeed=%d",
           tempMsg.temperature, tempMsg.lightLevel, tempMsg.pumpSpeed);

    // Set light and pump intensity
    if (tempMsg.lightLevel) {
        float TempLight = lightRead();
        lightSet(TempLight);
    } else {
        lightSet(LIGHT_OFF_INTENSITY);
    }

    if (tempMsg.pumpSpeed) {
        //pumpSet(50);
        float TempSpeed = pumpSpeed();
        pumpSetRpm(TempSpeed);
    } else {
        pumpSet(PUMP_OFF_INTENSITY);
    }

    float bValue = calibrationTemperature(tempMsg.temperature);

    // Send response
    wifiEventsSendTempCalibrationRsp(bValue);
}

// Apply O2 calibration parameters
static void wifiEventsHandleO2CalibrationReq(uint8_t* msg, size_t msgLen)
{
    cisme_o2CalibrationMsg o2Msg = cisme_o2CalibrationMsg_init_default;
    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);

    if (!pb_decode(&stream, cisme_o2CalibrationMsg_fields, &o2Msg)) {
        ERROR("Can't decode received message");
        return;
    }

    if (!o2Msg.has_pressure) {
        ERROR("Missing pressure field in O2_CALIBRATION_REQ");
        return;
    }

    if (!o2Msg.has_led) {
        ERROR("Missing led field in O2_CALIBRATION_REQ");
        return;
    }

    if (!o2Msg.has_phase1) {
        ERROR("Missing phase1 field in O2_CALIBRATION_REQ");
        return;
    }

    if (!o2Msg.has_temperature1) {
        ERROR("Missing temperature1 field in O2_CALIBRATION_REQ");
        return;
    }

    if (!o2Msg.has_phase2) {
        ERROR("Missing phase2 field in O2_CALIBRATION_REQ");
        return;
    }

    if (!o2Msg.has_temperature2) {
        ERROR("Missing temperature2 field in O2_CALIBRATION_REQ");
        return;
    }

    DEBUG1("Receive O2_CALIBRATION_REQ: pressure=%1.2f, led=%1.2f, phase1=%1.2f, temperature1=%1.2f, phase2=%1.2f, temperature2=%1.2f",
           o2Msg.pressure, o2Msg.led, o2Msg.phase1, o2Msg.temperature1, o2Msg.phase2, o2Msg.temperature2);

    // Set parameters
    PresCal = o2Msg.pressure;
    PhaseCal1 = o2Msg.phase1;
    TempCal1 = o2Msg.temperature1;
    PhaseCal2 = o2Msg.phase2;
    TempCal2 = o2Msg.temperature2;
    LEDCurrent = o2Msg.led;

    // Apply parameters
    calibrationO2();

    // Send response
    DEBUG1("Send O2_CALIBRATION_RSP: pressure=%1.2f, led=%1.2f, phase1=%1.2f, temperature1=%1.2f, phase2=%1.2f, temperature2=%1.2f",
           o2Msg.pressure, o2Msg.led, o2Msg.phase1, o2Msg.temperature1, o2Msg.phase2, o2Msg.temperature2);
    wifiEventsSendMsg(cisme_MessageType_O2_CALIBRATION_RSP, cisme_o2CalibrationMsg_fields, &o2Msg);
}

// Handle request to calibrate PH
static void wifiEventsHandlePhCalibrationReq(uint8_t* msg, size_t msgLen)
{
    cisme_phCalibrationMsg phMsg = cisme_phCalibrationMsg_init_default;
    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);

    if (!pb_decode(&stream, cisme_phCalibrationMsg_fields, &phMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    if (!phMsg.has_ph) {
        ERROR("Missing ph field in PH_CALIBRATION_REQ");
        return;
    }

    DEBUG1("Receive PH_CALIBRATION_REQ: ph=%2.3f, lightLevel=%d, pumpSpeed=%d",
           phMsg.ph, phMsg.lightLevel, phMsg.pumpSpeed);

    // Set light and pump intensity
    if (phMsg.lightLevel) {
        float pHLight = lightRead();
        lightSet(pHLight);
    } else {
        lightSet(LIGHT_OFF_INTENSITY);
    }

    if (phMsg.pumpSpeed) {
        float pHSpeed = pumpSpeed();
        pumpSetRpm(pHSpeed);
        
    } else {
        pumpSet(PUMP_OFF_INTENSITY);
    }

    // Allocate response
    cisme_phCalibrationRspMsg phRsp = cisme_phCalibrationRspMsg_init_default;
    phRsp.has_phBuffer = true;
    phRsp.has_phTemp = true;
    phRsp.has_slope = true;
    phRsp.has_eo = true;
    phRsp.phBuffer = phMsg.ph;

    calibrationPh(phRsp.phBuffer, &phRsp.phTemp, &phRsp.slope, &phRsp.eo, NULL);

    // Send response
    DEBUG1("Send PH_CALIBRATION_RSP: ph=%2.3f, phTemp=%2.3f, slope=%2.6f, eo=%2.6f",
           phRsp.phBuffer, phRsp.phTemp, phRsp.slope, phRsp.eo);
    wifiEventsSendMsg(cisme_MessageType_PH_CALIBRATION_RSP, cisme_phCalibrationRspMsg_fields, &phRsp);
}

// Send current PH parameters
static void wifiEventsSendCurCalPh(void)
{
    cisme_phCalibrationRspMsg phRsp = cisme_phCalibrationRspMsg_init_default;
    phRsp.has_phBuffer = true;
    phRsp.has_phTemp = true;
    phRsp.has_slope = true;
    phRsp.has_eo = true;
    phRsp.phBuffer = PHBUFFERF;
    phRsp.phTemp = PHTEMPF;
    phRsp.slope = MSINGLEPT;
    phRsp.eo = EOSINGLEPT;

    // Send response
    DEBUG1("Send PH_CALIBRATION_RSP: ph=%2.3f, phTemp=%2.3f, slope=%2.6f, eo=%2.6f",
           phRsp.phBuffer, phRsp.phTemp, phRsp.slope, phRsp.eo);
    wifiEventsSendMsg(cisme_MessageType_PH_CALIBRATION_RSP, cisme_phCalibrationRspMsg_fields, &phRsp);
}

// Send current O2 parameters
static void wifiEventsSendCurCalO2(void)
{
    cisme_o2CalibrationMsg o2Msg = cisme_o2CalibrationMsg_init_default;
    o2Msg.has_pressure = true;
    o2Msg.has_phase1 = true;
    o2Msg.has_temperature1 = true;
    o2Msg.has_phase2 = true;
    o2Msg.has_temperature2 = true;
    o2Msg.has_led = true;
    o2Msg.pressure = PresCal;
    o2Msg.phase1 = PhaseCal1;
    o2Msg.temperature1 = TempCal1;
    o2Msg.phase2 = PhaseCal2;
    o2Msg.temperature2 = TempCal2;
    o2Msg.led = LEDCurrent;

    // Send response
    DEBUG1("Send O2_CALIBRATION_RSP: pressure=%1.2f, led=%1.2f, phase1=%1.2f, temperature1=%1.2f, phase2=%1.2f, temperature2=%1.2f",
           o2Msg.pressure, o2Msg.led, o2Msg.phase1, o2Msg.temperature1, o2Msg.phase2, o2Msg.temperature2);
    wifiEventsSendMsg(cisme_MessageType_O2_CALIBRATION_RSP, cisme_o2CalibrationMsg_fields, &o2Msg);
}

static void sendPumpCalibrationRsp(float aValue, float bValue, float cValue)
{
    cisme_pumpCalibrationMsg rspMsg = cisme_pumpCalibrationMsg_init_default;
    rspMsg.has_a = true;
    rspMsg.has_b = true;
    rspMsg.has_c = true;
    rspMsg.a = aValue;
    rspMsg.b = bValue;
    rspMsg.c = cValue;

    DEBUG1("Send PUMP_CALIBRATION_RSP: a=%f, b=%f, c=%f", rspMsg.a, rspMsg.b, rspMsg.c);
    wifiEventsSendMsg(cisme_MessageType_PUMP_CALIBRATION_RSP, cisme_pumpCalibrationMsg_fields, &rspMsg);
}

// Handle request to get current calibrated parameters
static void wifiEventsHandleCurCalibrationReq(uint8_t* msg, size_t msgLen)
{
    cisme_currentCalibrationMsg calMsg = cisme_currentCalibrationMsg_init_default;
    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);

    if (!pb_decode(&stream, cisme_currentCalibrationMsg_fields, &calMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    if (!calMsg.has_type) {
        calMsg.type = cisme_CalibrationType_PH;
    }

    DEBUG1("Receive CUR_CALIBRATION_REQ: type=%d", calMsg.type);

    switch(calMsg.type) {
        case cisme_CalibrationType_PH:
            wifiEventsSendCurCalPh();
            break;
        case cisme_CalibrationType_TEMPERATURE:
            wifiEventsSendTempCalibrationRsp(B);
            break;
        case cisme_CalibrationType_O2:
            wifiEventsSendCurCalO2();
            break;
        case cisme_CalibrationType_PUMP: {
            float a = 0.0;
            float b = 0.0;
            float c = 0.0;
            pumpGetParams(&a, &b, &c);
            sendPumpCalibrationRsp(a, b, c);
            break;
        }
        default:
            break;
    }
}

// Send calibration accept response
static void wifiEventsSendCalAcceptRsp(cisme_CalibrationType type)
{
    cisme_currentCalibrationMsg calMsg = cisme_currentCalibrationMsg_init_default;
    calMsg.has_type = true;
    calMsg.type = type;

    DEBUG1("Send CALIBRATION_ACCEPT_RSP: type=%d", calMsg.type);
    wifiEventsSendMsg(cisme_MessageType_CALIBRATION_ACCEPT_RSP, cisme_currentCalibrationMsg_fields, &calMsg);
}

// Handle request to save temperature parameters
static void wifiEventsHandleTempCalAcceptReq(uint8_t* msg, size_t msgLen)
{
    cisme_tempCalibrationRspMsg tempMsg = cisme_tempCalibrationRspMsg_init_default;
    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);

    if (!pb_decode(&stream, cisme_tempCalibrationRspMsg_fields, &tempMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    if (!tempMsg.has_bValue) {
        ERROR("Missing bValue field in TEMP_CALIBRATION_ACCEPT_REQ");
        return;
    }

    DEBUG1("Receive TEMP_CALIBRATION_ACCEPT_REQ: bValue=%2.6f", tempMsg.bValue);

    calibrationTemperatureSave(tempMsg.bValue);

    // Send response
    wifiEventsSendCalAcceptRsp(cisme_CalibrationType_TEMPERATURE);
}

// Handle request to save PH parameters
static void wifiEventsHandlePhCalAcceptReq(uint8_t* msg, size_t msgLen)
{
    cisme_phCalibrationRspMsg phMsg = cisme_phCalibrationRspMsg_init_default;
    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);

    if (!pb_decode(&stream, cisme_phCalibrationRspMsg_fields, &phMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    if (!phMsg.has_phBuffer) {
        ERROR("Missing phBuffer field in PH_CALIBRATION_ACCEPT_REQ");
        return;
    }

    if (!phMsg.has_slope) {
        ERROR("Missing slope field in PH_CALIBRATION_ACCEPT_REQ");
        return;
    }

    if (!phMsg.has_phTemp) {
        ERROR("Missing phTemp field in PH_CALIBRATION_ACCEPT_REQ");
        return;
    }

    if (!phMsg.has_eo) {
        ERROR("Missing eo field in PH_CALIBRATION_ACCEPT_REQ");
        return;
    }

    DEBUG1("Receive PH_CALIBRATION_ACCEPT_REQ: ph=%2.3f, phTemp=%2.3f, slope=%2.6f, eo=%2.6f",
           phMsg.phBuffer, phMsg.phTemp, phMsg.slope, phMsg.eo);

    PHBUFFERF = (float)phMsg.phBuffer;
    PHTEMPF = (float)phMsg.phTemp;
    PHTEMPKF = (float)(phMsg.phTemp + 273.15);
    PHVOLTSF = (float)(phMsg.eo + phMsg.slope * phMsg.phBuffer);
    MSINGLEPT = (float)phMsg.slope;
    EOSINGLEPT = (float)phMsg.eo;

    // Send response
    wifiEventsSendCalAcceptRsp(cisme_CalibrationType_PH);

    // Save parameters and restart board
    calibrationPhSave();
}

// Start experiment
static void wifiEventsHandleStartExpInd(uint8_t* msg, size_t msgLen)
{
    cisme_startExperimentMsg expMsg = cisme_startExperimentMsg_init_default;
    static char receivedPrefix[256] = {0};

    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);
    if (!pb_decode(&stream, cisme_startExperimentMsg_fields, &expMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    if (!expMsg.has_id) {
        ERROR("Missing id field in START_EXPERIMENT_IND");
        return;
    }

    if (!expMsg.has_stepsCount) {
        ERROR("Missing stepsCount field in START_EXPERIMENT_IND");
        return;
    }

    if (!expMsg.has_salinity) {
        ERROR("Missing salinity field in START_EXPERIMENT_IND");
        return;
    }

    DEBUG1("Receive START_EXPERIMENT_IND: id=%d, experiment=%d, stepsCount=%d, salinity=%d, pumpSpeed=%d repeats=%d",
           expMsg.id, expMsg.experiment, expMsg.stepsCount, expMsg.salinity, expMsg.pumpSpeed, expMsg.repeats);

    if (isExpExecuting == true && expMsg.id != currentExpId) {
        ERROR("Received START_EXPERIMENT_IND with expMsg.id=%d, but already executing experiment id=%d",
              expMsg.id, currentExpId);
        return;
    }

    if (expMsg.stepsCount * (1 + expMsg.repeats) > MAX_STEPS_NO) {
        expMsg.repeats = (MAX_STEPS_NO / expMsg.stepsCount) - 1;
        ERROR("User wants too many steps. Count of repeats decreased to: %d", expMsg.repeats);
    }

    for (int ix = 0; ix < expMsg.stepsCount * (1 + expMsg.repeats); ix++) {
        pgm[ix][0] = expMsg.pumpSpeed;
    }

    // Decode steps
    uint8_t stepIx = 0;
    expMsg.steps.funcs.decode = &pbDecodeProgramSteps;
    expMsg.steps.arg = &stepIx;

    expMsg.filePrefix.funcs.decode = & pbDecodeString;
    expMsg.filePrefix.arg = receivedPrefix;

    stream = pb_istream_from_buffer(msg, msgLen);
    if (!pb_decode(&stream, cisme_startExperimentMsg_fields, &expMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    if (strcmp(receivedPrefix, "") != 0) {
        /* User wants some prefix for the file name.*/
        strncpy(FilePrefix, receivedPrefix, FILE_PREFIX_LENGTH);
        DEBUG1("User wants prefix:%s", FilePrefix);
    }

    Salinity           = expMsg.salinity;
    numberofprograms   = expMsg.stepsCount;
    currentExpId       = expMsg.id;
    isExpExecuting     = true;
    MeasurementTypeVal = PGM;

    if (expMsg.has_repeats && expMsg.experiment >= cisme_startExperimentMsg_ExperimentType_RESPIRATION &&
            expMsg.experiment < cisme_startExperimentMsg_ExperimentType_MULTI &&
            expMsg.repeats > 0) {
        for (int repeatIx = 0; repeatIx < expMsg.repeats; repeatIx++) {
            memcpy(&(pgm[numberofprograms * (repeatIx + 1)]), pgm, sizeof(float) * 20 * numberofprograms);
        }

        numberofprograms *= expMsg.repeats + 1;
    }

    experimentPrepareUserProgram();
}

// Handle experiment action (next step or stop)
static void wifiEventsHandleActionExpInd(cisme_MessageType type, uint8_t* msg, size_t msgLen)
{
    cisme_experimentMsg expMsg = cisme_experimentMsg_init_default;

    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);
    if (!pb_decode(&stream, cisme_experimentMsg_fields, &expMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    if (!expMsg.has_id) {
        ERROR("Missing id field in %s",
              (type == cisme_MessageType_NEXT_EXPERIMENT_IND) ? "NEXT_EXPERIMENT_IND" : "STOP_EXPERIMENT_IND");
        return;
    }

    if (isExpExecuting == false) {
        /* Not ERROR because this can happen if the message will be received when experiment already finished. */
        INFO("Unexpected actionExpInd. We are not executing experiment right now.");
        return;
    }

    if (expMsg.id != currentExpId) {
        ERROR("Unexpected expId received. Received expId=%d currentExpId=%d", expMsg.id, currentExpId);
        return;
    }

    switch (type) {
        case cisme_MessageType_NEXT_EXPERIMENT_IND:
            SwitchMode = 1;
            INFO("Experiment switched by user.");
            fprintf(currentFile,"User Advanced Step\n");
            break;
        case cisme_MessageType_STOP_EXPERIMENT_IND:

            INFO("User Ended Experiment");
            fprintf(currentFile,"User Ended Experiment\n");
            isExpExecuting = false;

            finishExperiment();

            wifiEventsSendFinishExpInd(currentExpId, FileName);
            break;
        default:
            ERROR("Unexpected message type received:%d", type);
            return;
    }
}

// Send file download response
static void wifiEventsSendFileDownloadRsp(int32_t fileId, int32_t numBlocks)
{
    cisme_fileDownloadRspMsg fileRsp = cisme_fileDownloadRspMsg_init_default;
    fileRsp.has_fileId = true;
    fileRsp.has_numBlocks = true;
    fileRsp.fileId = fileId;
    fileRsp.numBlocks = numBlocks;

    DEBUG1("Send FILE_DOWNLOAD_RSP: fileId=%d, numBlocks=%d", fileRsp.fileId, fileRsp.numBlocks);
    wifiEventsSendMsg(cisme_MessageType_FILE_DOWNLOAD_RSP, cisme_fileDownloadRspMsg_fields, &fileRsp);
}

// Send file block
static void wifiEventsSendFileBlockReq(int32_t fileId)
{
    blockId++;

    if (blockId > numBlocks) {
        ERROR("Excess file block requested");
        return;
    }

    cisme_fileBlockMsg blockReq = cisme_fileBlockMsg_init_default;
    blockReq.has_fileId = true;
    blockReq.has_blockId = true;
    blockReq.has_blockSize = true;
    blockReq.fileId = fileId;
    blockReq.blockId = blockId;
    blockReq.blockSize = (blockId < numBlocks) ? FILE_BLOCK_SIZE : (fileLen % FILE_BLOCK_SIZE);
    blockReq.blockData.funcs.encode = &pbEncodeFileBlock;

    DEBUG1("Send FILE_BLOCK_REQ: fileId=%d, blockId=%d, blockSize=%d",
           blockReq.fileId, blockReq.blockId, blockReq.blockSize);
    wifiEventsSendMsg(cisme_MessageType_FILE_BLOCK_REQ, cisme_fileBlockMsg_fields, &blockReq);
}

// Send file action (finish or abort)
static void wifiEventsSendFileActionInd(cisme_MessageType type, int32_t fileId)
{
    cisme_fileActionMsg actionMsg = cisme_fileActionMsg_init_default;
    actionMsg.has_fileId = true;
    actionMsg.fileId = fileId;

    DEBUG1("Send %s: fileId=%d",
           (type == cisme_MessageType_FILE_FINISH_IND) ? "FILE_FINISH_IND" : "FILE_ABORT_IND",
           actionMsg.fileId);
    wifiEventsSendMsg(type, cisme_fileActionMsg_fields, &actionMsg);
}

// Handle request to download file
static void wifiEventsHandleFileDownloadReq(uint8_t* msg, size_t msgLen)
{
    char fileName[FILE_NAME_LENGTH];

    cisme_fileDownloadMsg fileReq = cisme_fileDownloadMsg_init_default;
    fileReq.name.funcs.decode = &pbDecodeString;
    fileReq.name.arg = fileName;

    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);
    if (!pb_decode(&stream, cisme_fileDownloadMsg_fields, &fileReq)) {
        ERROR("Can't decode received message");
        return;
    }

    DEBUG1("Receive FILE_DOWNLOAD_REQ: name=%s", fileName);

    if (curFile != NULL) {
        INFO("New file download request, abort previous one");
        fclose(curFile);
        wifiEventsSendFileActionInd(cisme_MessageType_FILE_ABORT_IND, fileId);
    }

    char dataname[PATH_TO_FILE_LEN];
    sprintf(dataname, "%s%s", "/" FSNAME "/Data/", fileName);
    curFile = fopen(dataname, "rb");
    if (curFile == NULL) {
        ERROR("File %s not found", fileName);
        return;
    }

    fseek(curFile, 0, SEEK_END);
    fileLen = ftell(curFile);
    rewind(curFile);

    fileId++;
    numBlocks = fileLen / FILE_BLOCK_SIZE;
    if (fileLen % FILE_BLOCK_SIZE > 0) {
        numBlocks++;
    }

    wifiEventsSendFileDownloadRsp(fileId, numBlocks);

    blockId = 0;
    wifiEventsSendFileBlockReq(fileId);
}

// Handle response for block transfer
static void wifiEventsHandleFileBlockRsp(uint8_t* msg, size_t msgLen)
{
    cisme_fileBlockRspMsg blockRsp = cisme_fileBlockRspMsg_init_default;
    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);

    if (!pb_decode(&stream, cisme_fileBlockRspMsg_fields, &blockRsp)) {
        ERROR("Can't decode received message");
        return;
    }

    if (!blockRsp.has_fileId) {
        ERROR("Missing fileId field in FILE_BLOCK_RSP");
        return;
    }

    if (!blockRsp.has_blockId) {
        ERROR("Missing blockId field in FILE_BLOCK_RSP");
        return;
    }

    DEBUG1("Receive FILE_BLOCK_RSP: fileId=%d, blockId=%d",
           blockRsp.fileId, blockRsp.blockId);

    if (curFile == NULL) {
        ERROR("No file transfer ongoing");
        return;
    }

    if (fileId != blockRsp.fileId) {
        ERROR("Unknown fileId=%d, expected one:%d", blockRsp.fileId, fileId);

        // Send abort for both files
        wifiEventsSendFileActionInd(cisme_MessageType_FILE_ABORT_IND, fileId);
        wifiEventsSendFileActionInd(cisme_MessageType_FILE_ABORT_IND, blockRsp.fileId);

        if (curFile != NULL) {
            fclose(curFile);
            curFile = NULL;
        }
        return;
    }

    if (blockId != blockRsp.blockId) {
        ERROR("Unknown blockId=%d, expected one:%d", blockRsp.blockId, blockId);

        // Send abort for both files
        wifiEventsSendFileActionInd(cisme_MessageType_FILE_ABORT_IND, fileId);

        if (curFile != NULL) {
            fclose(curFile);
            curFile = NULL;
        }
        return;
    }

    if (blockId == numBlocks) {
        fclose(curFile);
        curFile = NULL;
        wifiEventsSendFileActionInd(cisme_MessageType_FILE_FINISH_IND, fileId);
        INFO("File transfer (%d) finished", fileId);
        return;
    }

    wifiEventsSendFileBlockReq(fileId);
}

// Handle file abort
static void wifiEventsHandleFileAbortInd(uint8_t* msg, size_t msgLen)
{
    cisme_fileActionMsg actionMsg = cisme_fileActionMsg_init_default;
    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);

    if (!pb_decode(&stream, cisme_fileActionMsg_fields, &actionMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    if (!actionMsg.has_fileId) {
        ERROR("Missing fileId field in FILE_ABORT_IND");
        return;
    }

    DEBUG1("Receive FILE_ABORT_IND: fileId=%d", actionMsg.fileId);

    if (fileId != actionMsg.fileId) {
        ERROR("Unknown fileId=%d, expected one:%d", actionMsg.fileId, fileId);
        return;
    }

    if (curFile == NULL) {
        ERROR("No file transfer ongoing");
        return;
    }

    fclose(curFile);
    curFile = NULL;
}

// Get experiments files
static void wifiEventsHandleGetFilesReq(void)
{
    DEBUG1("Receive GET_FILES_REQ");

    // Fill and send response
    cisme_filesMsg filesMsg = cisme_filesMsg_init_default;
    filesMsg.has_numFiles = true;
    filesMsg.numFiles = 0;
    filesMsg.names.funcs.encode = pbEncodeFiles;

    DIR* dir = opendir("/msc/data/");
    struct dirent* entry = NULL;

    while ((entry = readdir(dir)) != NULL) {
        if (strstr(entry->d_name, ".csv") != NULL) {
            filesMsg.numFiles++;
        }
    }

    DEBUG1("Send GET_FILES_RSP: numFiles=%d", filesMsg.numFiles);
    wifiEventsSendMsg(cisme_MessageType_GET_FILES_RSP, cisme_filesMsg_fields, &filesMsg);
}

// Handle request for raw data
static void wifiEventsHandleRawDataReq(void)
{
    DEBUG1("Receive RAW_DATA_REQ");

    // Prepare and send response
    cisme_rawDataMsg rawMsg = cisme_rawDataMsg_init_default;
    rawMsg.has_o2 = true;
    rawMsg.has_ph = true;
    rawMsg.has_o2Amplitude = true;
    rawMsg.has_o2Phase = true;
    rawMsg.has_phAdcV = true;
    rawMsg.has_temperatureAdcV = true;
    rawMsg.has_temperature = true;

    float presensData[PRESENS_RES_LENGTH];
    presensGetData(presensData);

    rawMsg.o2Amplitude = presensData[0];
    rawMsg.o2Phase = presensData[1];
    rawMsg.o2 = presensData[2];
    rawMsg.ph = pHCorrected;
    rawMsg.phAdcV = pH;
    rawMsg.temperatureAdcV = pHT;
    rawMsg.temperature = PHTEMP;

    DEBUG1("Send RAW_DATA_RSP: o2Amplitude=%d, o2Phase=%3.2f, o2=%3.0f, pH=%3.2f, pHAdcV=%3.3f, temperatureAdcV=%3.3f, temperature=%1.1f",
           rawMsg.o2Amplitude, rawMsg.o2Phase, rawMsg.o2, rawMsg.ph, rawMsg.phAdcV, rawMsg.temperatureAdcV, rawMsg.temperature);
    wifiEventsSendMsg(cisme_MessageType_RAW_DATA_RSP, cisme_rawDataMsg_fields, &rawMsg);
}

// Start pump calibration
static void wifiEventsHandlePumpCalibrationReq(void)
{
    DEBUG1("Receive PUMP_CALIBRATION_REQ");
 
    float a = 0.0;
    float b = 0.0;
    float c = 0.0;
    
    calibrationPump(&a, &b, &c);
    sendPumpCalibrationRsp(a, b, c);
}
 
// Apply pump calibration parameters
static void wifiEventsHandlePumpCalAcceptReq(uint8_t* msg, size_t msgLen)
{
    cisme_pumpCalibrationMsg calMsg = cisme_pumpCalibrationMsg_init_default;
    pb_istream_t stream = pb_istream_from_buffer(msg, msgLen);
 
    if (!pb_decode(&stream, cisme_pumpCalibrationMsg_fields, &calMsg)) {
        ERROR("Can't decode received message");
        return;
    }
 
    DEBUG1("Receive PUMP_CALIBRATION_ACCEPT_REQ: a=%f, b=%f, c=%f", calMsg.a, calMsg.b, calMsg.c);
    pumpSetParams(calMsg.a, calMsg.b, calMsg.c);
    
    // Send response
    wifiEventsSendCalAcceptRsp(cisme_CalibrationType_PUMP);
}

// Handle message from app
static void wifiEventsHandleMsg(void)
{
    uint8_t* buffer;
    unsigned int msgLen = wifiGetMessage(&buffer);

    if (msgLen == 0) {
        return;
    }

    cisme_genericMsg genericMsg = cisme_genericMsg_init_default;
    PbDecodeMessage message = {NULL, 0};
    genericMsg.payload.funcs.decode = pbDecodeMsg;
    genericMsg.payload.arg = &message;

    for (size_t ix = 0; ix < msgLen; ix++) {
        DEBUG1("0x%x", buffer[ix]);
    }

    pb_istream_t stream = pb_istream_from_buffer(buffer, msgLen);

    if (!pb_decode(&stream, cisme_genericMsg_fields, &genericMsg)) {
        ERROR("Can't decode received message");
        return;
    }

    switch (genericMsg.type) {
        case cisme_MessageType_INSTRUMENT_ID_REQ:
            wifiEventsHandleIdReq();
            break;
        case cisme_MessageType_START_EXPERIMENT_IND:
            wifiEventsHandleStartExpInd(message.msg, message.msgLen);
            break;
        case cisme_MessageType_NEXT_EXPERIMENT_IND:
        case cisme_MessageType_STOP_EXPERIMENT_IND:
            wifiEventsHandleActionExpInd(genericMsg.type, message.msg, message.msgLen);
            break;
        case cisme_MessageType_TEMP_CALIBRATION_REQ:
            wifiEventsHandleTempCalibrationReq(message.msg, message.msgLen);
            break;
        case cisme_MessageType_O2_CALIBRATION_REQ:
            wifiEventsHandleO2CalibrationReq(message.msg, message.msgLen);
            break;
        case cisme_MessageType_PH_CALIBRATION_REQ:
            wifiEventsHandlePhCalibrationReq(message.msg, message.msgLen);
            break;
        case cisme_MessageType_CUR_CALIBRATION_REQ:
            wifiEventsHandleCurCalibrationReq(message.msg, message.msgLen);
            break;
        case cisme_MessageType_TEMP_CALIBRATION_ACCEPT_REQ:
            wifiEventsHandleTempCalAcceptReq(message.msg, message.msgLen);
            break;
        case cisme_MessageType_PH_CALIBRATION_ACCEPT_REQ:
            wifiEventsHandlePhCalAcceptReq(message.msg, message.msgLen);
            break;
        case cisme_MessageType_SET_PUMP_REQ:
            wifiEventsHandleSetPumpReq(message.msg, message.msgLen);
            break;
        case cisme_MessageType_GET_PUMP_REQ:
            wifiEventsHandleGetPumpReq();
            break;
        case cisme_MessageType_SET_LIGHT_REQ:
            wifiEventsHandleSetLightReq(message.msg, message.msgLen);
            break;
        case cisme_MessageType_GET_LIGHT_REQ:
            wifiEventsHandleGetLightReq();
            break;
        case cisme_MessageType_GET_TIME_RSP:
            wifiEventsHandleTimeRsp(message.msg, message.msgLen);
            break;
        case cisme_MessageType_GET_FILES_REQ:
            wifiEventsHandleGetFilesReq();
            break;
        case cisme_MessageType_FILE_DOWNLOAD_REQ:
            wifiEventsHandleFileDownloadReq(message.msg, message.msgLen);
            break;
        case cisme_MessageType_FILE_BLOCK_RSP:
            wifiEventsHandleFileBlockRsp(message.msg, message.msgLen);
            break;
        case cisme_MessageType_FILE_ABORT_IND:
            wifiEventsHandleFileAbortInd(message.msg, message.msgLen);
            break;
        case cisme_MessageType_RAW_DATA_REQ:
            wifiEventsHandleRawDataReq();
            break;
        case cisme_MessageType_PUMP_CALIBRATION_REQ:
            wifiEventsHandlePumpCalibrationReq();
            break;
        case cisme_MessageType_PUMP_CALIBRATION_ACCEPT_REQ:
            wifiEventsHandlePumpCalAcceptReq(message.msg, message.msgLen);
            break;
        default:
            ERROR("Unknown message type:%d", genericMsg.type);
            break;
    }
}

static void wifiEventsSendFinishExpInd(int32_t id, char* fileName)
{
    if (!wifiTcpConnectionActive()) {
        return;
    }

    cisme_experimentMsg expMsg = cisme_experimentMsg_init_default;
    expMsg.has_id = true;
    expMsg.id = id;
    expMsg.fileName.funcs.encode = &pbEncodeString;
    expMsg.fileName.arg = fileName;

    DEBUG1("Send FINISH_EXPERIMENT_IND: id=%d, fileName=%s", expMsg.id, fileName);
    wifiEventsSendMsg(cisme_MessageType_FINISH_EXPERIMENT_IND, cisme_experimentMsg_fields, &expMsg);

}

void wifiEventsSendErrorsInd(char errors)
{
    if (getNoSetBits(errors) == 0) {
        return;
    }

    cisme_errorMsg errorMsg = cisme_errorMsg_init_default;
    errorMsg.has_numErrors = true;
    errorMsg.numErrors = getNoSetBits(errors);
    errorMsg.errorText.arg = &errors;
    errorMsg.errorText.funcs.encode = pbEncodeErrors;

    DEBUG1("Send ERRORS_IND: numErrors=%d", errorMsg.numErrors);
    wifiEventsSendMsg(cisme_MessageType_ERRORS_IND, cisme_errorMsg_fields, &errorMsg);
}

void wifiEventsSendDataExpInd(int32_t step, int32_t timeRemaining, double ph, double o2, double battery, double temperature)
{
    if (!wifiTcpConnectionActive()) {
        return;
    }

    cisme_experimentDataMsg expMsg = cisme_experimentDataMsg_init_default;
    expMsg.has_id = true;
    expMsg.has_step = true;
    expMsg.has_timeRemaining = true;
    expMsg.has_ph = true;
    expMsg.has_o2 = true;
    expMsg.has_battery = true;
    expMsg.has_temperature = true;
    expMsg.has_lightLevel = true;
    expMsg.has_pumpRpm = true;
    expMsg.id = currentExpId;
    expMsg.step = step;
    expMsg.timeRemaining = timeRemaining;
    expMsg.ph = ph;
    expMsg.o2 = o2;
    expMsg.battery = battery;
    expMsg.temperature = temperature;
    expMsg.lightLevel = lightRead();
    expMsg.pumpRpm = pumpSpeed();

    DEBUG1("Send EXPERIMENT_DATA_IND: id=%d, step=%d, timeRemaining=%d, ph=%3.2f, o2=%3.0f, battery=%3.1f, temperature=%2.1f, lightLevel=%d, pumpRpm=%d",
           expMsg.id, expMsg.step, expMsg.timeRemaining, expMsg.ph, expMsg.o2, expMsg.battery, expMsg.temperature, expMsg.lightLevel, expMsg.pumpRpm);
    wifiEventsSendMsg(cisme_MessageType_EXPERIMENT_DATA_IND, cisme_experimentDataMsg_fields, &expMsg);

}

void wifiEventsSendTempCalDataInd(float tempBuffer, float tempVolts)
{
    cisme_tempCalibrationDataMsg dataMsg = cisme_tempCalibrationDataMsg_init_default;
    dataMsg.has_tempBuffer = true;
    dataMsg.tempBuffer = tempBuffer;
    dataMsg.has_tempVolts = true;
    dataMsg.tempVolts = tempVolts;

    DEBUG1("Send TEMP_CALIBRATION_DATA_IND: tempBuffer=%2.4f, tempVolts=%2.6f", dataMsg.tempBuffer, dataMsg.tempVolts);
    wifiEventsSendMsg(cisme_MessageType_TEMP_CALIBRATION_DATA_IND, cisme_tempCalibrationDataMsg_fields, &dataMsg);
}

void wifiEventsSendPhCalDataInd(float phBuffer, float phVoltsCurrent, float phVoltsAverage)
{
    cisme_phCalibrationDataMsg dataMsg = cisme_phCalibrationDataMsg_init_default;
    dataMsg.has_phBuffer = true;
    dataMsg.phBuffer = phBuffer;
    dataMsg.has_phVoltsCurrent = true;
    dataMsg.phVoltsCurrent = phVoltsCurrent;
    dataMsg.has_phVoltsAverage = true;
    dataMsg.phVoltsAverage = phVoltsAverage;

    DEBUG1("Send PH_CALIBRATION_DATA_IND: phBuffer=%2.4f, phVoltsCurrent=%2.6f, phVoltsAverage=%2.6f",
           dataMsg.phBuffer, dataMsg.phVoltsCurrent, dataMsg.phVoltsAverage);
    wifiEventsSendMsg(cisme_MessageType_PH_CALIBRATION_DATA_IND, cisme_phCalibrationDataMsg_fields, &dataMsg);
}

void wifiEventsSendPumpCalDataInd(int intensity, int rpm)
{
    cisme_pumpCalibrationDataMsg dataMsg = cisme_pumpCalibrationDataMsg_init_default;
    dataMsg.has_intensity = true;
    dataMsg.intensity = intensity;
    dataMsg.has_rpm = true;
    dataMsg.rpm = rpm;
 
    DEBUG1("Send PUMP_CALIBRATION_DATA_IND: intensity=%d, rpm=%d", dataMsg.intensity, dataMsg.rpm);
    wifiEventsSendMsg(cisme_MessageType_PUMP_CALIBRATION_DATA_IND, cisme_pumpCalibrationDataMsg_fields, &dataMsg);
}

void wifiEventsStart(void)
{
    bool isTimeRequested = false;

    while(true) {

        if (isExpExecuting && experimentRunUserProgram()) {
            INFO("Experiment completed.");
            fprintf(currentFile, "Experiment Completed\n");
            isExpExecuting = false;
            finishExperiment();

            wifiEventsSendFinishExpInd(currentExpId, FileName);
        }

        if (!wifiTcpConnectionActive()) {
            wait(0.1);
            L4=0;
            continue;
        }

        if (batteryGet() < BATTERY_MIN_LEVEL) {
            if (isExpExecuting) {
                fprintf(currentFile, "Experiment Finished due to Low Battery Level\n");
                finishExperiment();
                wifiEventsSendFinishExpInd(currentExpId, FileName);
            }

            wifiEventsSendErrorsInd(ERROR_BATTERY_LOW);
            lightSet(LIGHT_OFF_INTENSITY);
            pumpSet(PUMP_OFF_INTENSITY);
            return;
        }

        if (!isTimeRequested) {
            // Request current time
            DEBUG1("Send GET_TIME_REQ");
            wifiEventsSendMsg(cisme_MessageType_GET_TIME_REQ, NULL, NULL);
            wifiEventsSendErrorsInd(startErrors);
            isTimeRequested = true;
        }

        wifiEventsSendBeatInd();

        wifiEventsHandleMsg();
    }
}

#endif // USE_WIFI
