#include "mbed.h"
#include "light.h"
#include "debug.h"

// PWM out to handle light
static PwmOut lightPwm(p24);

// Light PWM period is 10 ms changed to 1 ms for use with new LED driver Aug. 2018
#define LIGHT_PWM_PERIOD_US 1000

// 100% duty cycle is used to disable light changed to 0% equals off for new LED driver Aug. 2018
#define LIGHT_OFF_DUTY_CYCLE 0.0f

void lightInit(void)
{
    // Turn light off
    // Don't need to set period as 100% duty cycle generates
    // the same signal for all periods
    lightPwm.write(LIGHT_OFF_DUTY_CYCLE);
}

void lightSet(unsigned char intensity)
{
    if (intensity > LIGHT_MAX_INTENSITY) {
        // Don't support intensity more than 100%
        ERROR("Request to set too big light intensity %u", intensity);
        return;
    }

//    if (lightRead() == intensity) {
//        INFO("Discard request to set light intensity, intensity %u already applied", intensity);
//        return;
//    }

    if (intensity != LIGHT_OFF_INTENSITY) {
        lightPwm.period_us(LIGHT_PWM_PERIOD_US);
    }

    // Duty cycle 0% turns light off, 100% - full brightness changed Aug. 2018 for new LED driver
    float dutyCycle = LIGHT_OFF_DUTY_CYCLE + (float)intensity / LIGHT_MAX_INTENSITY;
    lightPwm.write(dutyCycle);
    
    DEBUG1("Light intensity %u was successfully applied.", intensity);
}

unsigned char lightRead(void)
{
    float dutyCycle = lightPwm.read();
    unsigned char intensity = (LIGHT_OFF_DUTY_CYCLE + dutyCycle) * LIGHT_MAX_INTENSITY;
    return intensity;
}

bool lightOn(void)
{
    return (lightRead() > LIGHT_OFF_INTENSITY);
}
    
