#include "cisme.h"
#include "keys.h"
#include "buzzer.h"
#include "debug.h"

#define KEY_RSP_LENGHT 6
#define KEY_RSP_TYPE_OFFSET 2
#define KEY_RSP_SYMBOL_OFFSET 3

static char inBuffer[KEY_RSP_LENGHT];
static unsigned char bufferIx;

// Get numeric key from char
static Key keyNumberFromChar(char symbol)
{
    static const char map[] = "0123456789";

    for (char ix = 0; ix < (sizeof(map) - 1); ix++) {
        if (map[ix] == symbol) {
            return (Key)ix;
        }
    }

    return KEY_NONE;
}

// Get special key from char
static Key keySpecialFromChar(char symbol)
{
    switch (symbol) {
        case 'A':
            return KEY_ASTERISK;
        case '3':
            return KEY_HASH;
        default:
            return KEY_NONE;
    }
}

// Get key from response
static Key keyGetFromResponse(void)
{
    Key keyPressed = KEY_NONE;

    switch (inBuffer[KEY_RSP_TYPE_OFFSET]) {
        case '2':
            keyPressed = keySpecialFromChar(inBuffer[KEY_RSP_SYMBOL_OFFSET]);
            break;
        case '3':
            keyPressed = keyNumberFromChar(inBuffer[KEY_RSP_SYMBOL_OFFSET]);
            break;
        default:
            keyPressed = KEY_NONE;
            break;
    }

    // Clear buffer
    bufferIx = 0;
    memset(inBuffer, 0, KEY_RSP_LENGHT);

    if (keyPressed != KEY_NONE) {
        buzzerBeep(0.1);
    }

    return keyPressed;
}

// Check that key is in range
static bool keyInRange(Key key, const Key* range)
{
    for (char keyIx = 0; range[keyIx] != KEY_END; keyIx++) {
        if (key == range[keyIx]) {
            return true;
        }
    }
    return false;
}

void keyEvent(char byte)
{
    inBuffer[bufferIx++] = byte;
    if (bufferIx == KEY_RSP_LENGHT) bufferIx = 0;
}

Key keyWait(void)
{
    // Wait for keypress response
    while (bufferIx != (KEY_RSP_LENGHT - 1)) {
        wait(0.1);
    }

    return keyGetFromResponse();
}

Key keyWaitInRange(const Key* range)
{
    Key key = KEY_NONE;
    while (!keyInRange(key, range)) {
        key = keyWait();
    }

    return key;
}

Key keyGet(void)
{
    if (bufferIx != (KEY_RSP_LENGHT - 1)) {
        wait(0.1);
    }

    return keyGetFromResponse();
}
