#include "cisme.h"
#include "calibration.h"
#include "lcd.h"
#include "adc.h"
#include "debug.h"
#include "presens.h"
#include "pump.h"
#include "wifi_events.h"

#define PUMP_NUM_LEVELS 10
#define PUMP_NUM_MEASUREMENTS 10
#define PUMP_MATRIX_NUM_ROWS 3

float calibrationTemperature(float temperature)
{
    NVIC_EnableIRQ(UART1_IRQn);

    lcdClear();
    lcdWrite(0, 0, JUSTIFICATION_CENTER, "Temp Calibration");
    lcdWrite(1, 0, JUSTIFICATION_CENTER, "Single Point");
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "ISFET Temp");
    lcdWrite(3, 0, JUSTIFICATION_CENTER, "When Stable");
    lcdWrite(4, 0, JUSTIFICATION_CENTER, "Voltage");

    for (unsigned char n = 0; n < 4; n++) {
        adcGetData();
        lcdWrite(5, 0, JUSTIFICATION_CENTER, "%2.6f", pHT);
        wait(0.2);
    }

    lcdClear();
    lcdWrite(0, 0, JUSTIFICATION_CENTER, "Temp Calibration");
    lcdWrite(1, 0, JUSTIFICATION_CENTER, "Single Point");
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "100 points, 15 seconds");
    lcdWrite(3, 0, JUSTIFICATION_CENTER, "Temp(C) Value");
    lcdWrite(4, 0, JUSTIFICATION_CENTER, "%2.4f", temperature);
    lcdWrite(5, 0, JUSTIFICATION_CENTER, "Temp Voltage");

    float PHTEMPa = 0;
    for (unsigned char i = 1; i < 101; i++) {
        adcGetData();
        PHTEMPa += pHT;
        lcdWrite(6, 0, JUSTIFICATION_CENTER, "%2.6f", PHTEMPa / i);
        wifiEventsSendTempCalDataInd(temperature, PHTEMPa / i);
        DEBUG1("PHTEMPa %f", PHTEMPa);
    }

    PHTEMPa /= 100;
    NVIC_EnableIRQ(UART1_IRQn);
    float Rthermc = 20000.0 / ((2.5 / PHTEMPa) - 1.0);
    INFO("PHTEMP %f", Rthermc);

    lcdWrite(6, 0, JUSTIFICATION_CENTER, "%2.6f", pHT);
    INFO("PHTEMPave %1.6f", PHTEMPa);

    float bValue = (temperature / Rthermc) - (A / Rthermc) - (( C / Rthermc) * log10(Rthermc)) - ((D / Rthermc) * pow(log10(Rthermc), 3));

    INFO("Bval %1.6f", bValue);
    lcdWrite(7, 0, JUSTIFICATION_CENTER, "%2.6f", bValue);

    return bValue;
}

void calibrationTemperatureSave(float bValue)
{
    B = bValue;

    FILE* isfet = fopen("/" FSNAME "/Parameters/isfet.sys","w");
    if (isfet == NULL) {
        lcdClear();
        ERROR("Could not open file for write");
        lcdWrite(4, 0, JUSTIFICATION_CENTER, "isfet File Missing");
    }

    INFO("Aval %1.6f Bval %1.6f Cval %1.6f Dval %1.6f", A, B, C, D);
    fprintf(isfet, "%f,%f,%f,%f", A, B, C, D);
    fclose(isfet);
}

void calibrationO2(void)
{
    PRESENS_SET_SACU(LEDCurrent);

    // Open Parameter File
    FILE* param = fopen("/" FSNAME "/Parameters/dsparams.sys", "w");

    fprintf(param, "%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%d",
            a, b, c, PresCal, PhaseCal2, TempCal2, PhaseCal1, TempCal1, LEDCurrent,
            PHBUFFERF, PHTEMPF, PHTEMPKF, PHVOLTSF, MSINGLEPT, EOSINGLEPT, UNIT, debugGetCurrLvl());
    fclose(param);
}

void calibrationPh(float phBuffer, float* phTemp, float* slope, float* eo, float* phVolts)
{
    const double F = 96487.0;
    const double R = 8.31451;

    lcdClear();
    lcdWrite(0, 0, JUSTIFICATION_CENTER, "Temp Calibration");
    lcdWrite(1, 0, JUSTIFICATION_CENTER, "Single Point");
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "ISFET pH");
    lcdWrite(3, 0, JUSTIFICATION_CENTER, "When Stable");
    lcdWrite(4, 0, JUSTIFICATION_CENTER, "Voltage");

    for (int m = 0; m < 4; m++) {
        adcGetData();
        lcdWrite(5, 0, JUSTIFICATION_CENTER, "%2.6f", pHNoGain);
        DEBUG1("pHNoGain %f", pHNoGain);
    }

    // Calibrate
    lcdClear();
    lcdWrite(0, 0,   JUSTIFICATION_CENTER, "pH Auto Calibration");
    lcdWrite(1, 0,   JUSTIFICATION_CENTER, "Single Point");
    lcdWrite(2, 0,   JUSTIFICATION_CENTER, "100 points, 15 seconds");
    lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "pH Buffer Value");
    lcdWrite(4, 0,   JUSTIFICATION_CENTER, " %2.4f", phBuffer);
    lcdWrite(5, 0, JUSTIFICATION_ABSOLUTE, "pH Measured Voltage");

    float PHValue1 = 0;
    float PHTEMPa = 0;
    for (int i = 1; i < 101; i++) {
        wait (0.1);
        adcGetData();
        float pHActual = pHNoGain;
        PHValue1 += pHNoGain;
        PHTEMPa += PHTEMP;

        lcdWrite(6, 0, JUSTIFICATION_CENTER," %2.6f", PHValue1 / i);
        lcdWrite(7, 0, JUSTIFICATION_CENTER, "%2.6f", pHNoGain);
        wifiEventsSendPhCalDataInd(phBuffer, pHNoGain, PHValue1 / i);
        DEBUG2("pHActual %f PHValue1 %f PHTEMP %f PHTEMPa %f",
               pHActual, PHValue1, PHTEMP, PHTEMPa);
    }

    PHValue1 /= 100;
    PHTEMPa /= 100;

    lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "%2.6f", PHValue1);

    INFO("PHTEMPave %1.6f PHValue1 ave %1.6f", PHTEMPa, PHValue1);

    *phTemp = PHTEMPa;
    *slope = R * (PHTEMPa + 273.15) / F * 2.302585;
    *eo = (double)PHValue1 - (*slope) * (double)phBuffer;

    if (phVolts != NULL) {
        *phVolts = PHValue1;
    }
}

void calibrationPhSave(void)
{
    // Open Parameter File
    FILE* param = fopen("/" FSNAME "/Parameters/dsparams.sys", "w");
    if (param == NULL) {
        lcdClear();
        lcdWrite(4, 0, JUSTIFICATION_CENTER, "Dsprams File Missing");
        return;
    }

    fprintf(param, "%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%f,%d",
            a, b, c, PresCal, PhaseCal2, TempCal2, PhaseCal1, TempCal1, LEDCurrent,
            PHBUFFERF, PHTEMPF, PHTEMPKF, PHVOLTSF, MSINGLEPT, EOSINGLEPT, UNIT, debugGetCurrLvl());
    fclose(param);
    lcdWrite(0, 0, JUSTIFICATION_CENTER, "Wait 2 seconds");
    wait(2.0);
    lcdClear();
    lcdWrite(4, 0, JUSTIFICATION_CENTER, "Restarting");
    wait(0.5);
    pumpSet(PUMP_OFF_INTENSITY);
    mbed_reset();
}

void calibrationPump(float* a, float* b, float* c)
{
    static const uint8_t intensity[PUMP_NUM_LEVELS] = {10, 20, 30, 40, 50, 60, 70, 80, 90, 99};

    double matrix[PUMP_MATRIX_NUM_ROWS][PUMP_MATRIX_NUM_ROWS + 1] = {0.0};

    for (int intenIx = 0; intenIx < PUMP_NUM_LEVELS; intenIx++) {

        double rpm = 0.0;
        pumpSet(intensity[intenIx]);
        for (int measIx = 0; measIx < PUMP_NUM_MEASUREMENTS; measIx++) {
            rpm += pumpSpeed();
            DEBUG1("Pump calibration: intensity=%u, average RPM=%f", intensity[intenIx], rpm / (measIx + 1));
            wifiEventsSendPumpCalDataInd(intensity[intenIx], rpm / (measIx + 1));
        }

        rpm /= PUMP_NUM_MEASUREMENTS;

        matrix[0][1] += rpm;
        matrix[0][2] += rpm * rpm;
        matrix[1][2] += rpm * rpm * rpm;
        matrix[2][2] += rpm * rpm * rpm * rpm;
        matrix[0][3] += intensity[intenIx];
        matrix[1][3] += intensity[intenIx] * rpm;
        matrix[2][3] += intensity[intenIx] * rpm * rpm;
    }

    pumpSet(PUMP_OFF_INTENSITY);

    matrix[0][0] = PUMP_NUM_LEVELS;
    matrix[1][0] = matrix[0][1];
    matrix[1][1] = matrix[0][2];
    matrix[2][0] = matrix[1][1];
    matrix[2][1] = matrix[1][2];

    // Pivoting
    for (int col = 0; col + 1 < PUMP_MATRIX_NUM_ROWS; col++) {
        if (matrix[col][col] == 0) {
            // Find non-zero coefficient
            int swapRow = col + 1;
            for (; swapRow < PUMP_MATRIX_NUM_ROWS; swapRow++) {
                if (matrix[swapRow][col] != 0) {
                    break;
                }
            }

            if (matrix[swapRow][col] != 0) { // Found a non-zero coefficient?
                // Yes, then swap it with the above
                for (int i = 0; i < PUMP_MATRIX_NUM_ROWS + 1; i++) {
                    double tmp = matrix[swapRow][i];
                    matrix[swapRow][i] = matrix[col][i];
                    matrix[col][i] = tmp;
                }
            } else {
                // No, then the matrix has no unique solution
                ERROR("Can't resolve pump matrix");
                return;
            }
        }
    }

    // Elimination
    for (int sourceRow = 0; sourceRow + 1 < PUMP_MATRIX_NUM_ROWS; sourceRow++) {
        for (int destRow = sourceRow + 1; destRow < PUMP_MATRIX_NUM_ROWS; destRow++) {
            float df = matrix[sourceRow][sourceRow];
            float sf = matrix[destRow][sourceRow];
            for (int i = 0; i < PUMP_MATRIX_NUM_ROWS + 1; i++) {
                matrix[destRow][i] = matrix[destRow][i] * df - matrix[sourceRow][i] * sf;
            }
        }
    }

    // Back-insertion
    for (int row = PUMP_MATRIX_NUM_ROWS - 1; row >= 0; row--) {
        double f = matrix[row][row];
        if (f == 0) {
            ERROR("Can't resolve pump matrix");
            return;
        }

        for (int ix = 0; ix < PUMP_MATRIX_NUM_ROWS + 1; ix++) {
            matrix[row][ix] /= f;
        }

        for (int destRow = 0; destRow < row; destRow++) {
            matrix[destRow][PUMP_MATRIX_NUM_ROWS] -= matrix[destRow][row] * matrix[row][PUMP_MATRIX_NUM_ROWS];
            matrix[destRow][row] = 0;
        }
    }

    *a = matrix[0][PUMP_MATRIX_NUM_ROWS];
    *b = matrix[1][PUMP_MATRIX_NUM_ROWS];
    *c = matrix[2][PUMP_MATRIX_NUM_ROWS];

    INFO("Pump calibration coefficients: a=%f, b=%f, c=%f", *a, *b, *c);
}