#ifndef WIFI_H
#define WIFI_H

#include <stddef.h>
#include "cisme.h"

#ifdef USE_WIFI
#define WIFI_MESSAGE_MAX_LENGTH 256

/*
 * Current wifi board will send TCP message in the following 3 cases:
 * 1) When it receives some specific symbol specified during
 *    reconfiguration. This case is disabled currently.
 * 2) When the flush buffer if full. Currently during
 *    reconfiguration we set flush buffer to max value
 *    ~= 1500 bytes.
 * 3) When it doesn't receive any new data during some
 *    timer. During reconfiguration timer length is set
 *    to this value. Default value is 0.005 seconds. With
 *    such timer length sometimes we observing situation when
 *    one message is split on 2-3 TCP messages. With this timer
 *    length this is not observed. But sometimes we can concatenate
 *    some messages into one TCP message. To avoid this the program
 *    should wait this timer value (probably twice) before sending of
 *    the next message.
*/
#define WIFI_COMM_TIME    100

/**
 * Initialize Wi-Fi.
 */
void wifiInit(void);

/**
 * Get received message.
 *
 * @param msg Received message.
 * @return message length.
 */
size_t wifiGetMessage(uint8_t** msg);

/**
 * Send message.
 *
 * @param msg Message to be sent.
 * @param len Message length.
 */
void wifiSendMessage(const uint8_t* msg, size_t len);

/**
 * Get state of TCP connection.
 *
 * @return true if there is active TCP connection, false otherwise.
 */
bool wifiTcpConnectionActive(void);

#else // USE_WIFI

static inline void wifiInit(void)
{
}

static inline size_t wifiGetMessage(uint8_t** msg)
{
    return 0;
}

static inline void wifiSendMessage(const uint8_t* msg, size_t len)
{
}

static inline bool wifiTcpConnectionActive(void)
{
    return false;
}

#endif // USE_WIFI

#endif // WIFI_H