#include <time.h>
#include "cisme.h"
#include "lcd.h"
#include "debug.h"
#include "light.h"
#include "pump.h"
#include "buzzer.h"
#include "presens.h"
#include "keys.h"
#include "debug.h"
#include "wifi_events.h"
#include "experiments.h"
#include "battery.h"

#define PRINT_RESULT_TO_FILE(_modeStepNo, _deltaTime, _02Float, _uMolO2, _phase, _amplitude)\
fprintf(currentFile,"%d,%2.5f,%c,%2.2f,%2.1f,%2.4f,%2.2f,%c,%2.6f,%2.6f,%2.2f,%2.2f,%d\n",\
        _modeStepNo, _deltaTime / 60, ' ', _02Float, _uMolO2, pHCorrected, PHTEMP, ' ', pH, pHT, batteryGet(), _phase, _amplitude);

/* Variables neede in other modules: (mostly lcd_events.h)*/
FILE*  currentFile                    = NULL;
char   FileName[FILE_NAME_LENGTH]     = {0};
char   FilePrefix[FILE_PREFIX_LENGTH] = {0};
int    StartTime          = 0;
int    SwitchMode         = 0;
float  pgm[MAX_STEPS_NO][20];

/* Local variables for module. */
/* Variables for user program. */
static int    stepIx;
static double phold;
static float  O2old;

/* Varibles for defined programs. */
static int  mode     = 0;
static int  minEXPER = 0;
static int  minMODE  = 0;
static int  minTOTAL = 0;
static bool isModeSwitched = false;

static void getFileName(void)
{
    const char* months[12] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

    struct tm    *tim;
    time_t        now;
    int           numSymb = 0;
    int           numSymbToPrint;

    /* get time to make filename */
    now = time(NULL);
    tim = localtime(&now);

    /*
     * Make filename from date and time
     * format will be: C1_Mdd_hhmm.csv
     * M - 3 letters representing month
     * dd - day of month,
     * hh - hour,
     * mm - minutes.
     */

    // Set the Cisme unit name
    if (FilePrefix[0] == 0) {
        /* Prefix is not set. sprintf is used here to receive number of printed symbols. */
        numSymb = sprintf(&FileName[0], "%s", instrumentId);
    } else {
        numSymb = snprintf(&FileName[0], FILE_NAME_LENGTH, "%s_", FilePrefix);
    }

    numSymbToPrint = ((FILE_NAME_LENGTH - numSymb) < 0)? 0 : FILE_NAME_LENGTH - numSymb;
    numSymb += snprintf( &FileName[numSymb], numSymbToPrint, months[tim->tm_mon]);
    numSymbToPrint = ((FILE_NAME_LENGTH - numSymb) < 0)? 0 : FILE_NAME_LENGTH - numSymb;
    numSymb += strftime(&FileName[numSymb], numSymbToPrint, "%d", tim);
    numSymbToPrint = ((FILE_NAME_LENGTH - numSymb) < 0)? 0 : FILE_NAME_LENGTH - numSymb;
    numSymb += snprintf( &FileName[numSymb], numSymbToPrint, "_");
    numSymbToPrint = ((FILE_NAME_LENGTH - numSymb) < 0)? 0 : FILE_NAME_LENGTH - numSymb;
    numSymb += strftime(&FileName[numSymb], numSymbToPrint, "%H%M", tim);
    numSymbToPrint = ((FILE_NAME_LENGTH - numSymb) < 0)? 0 : FILE_NAME_LENGTH - numSymb;

    if (numSymbToPrint < 5) {
        snprintf(&FileName[FILE_NAME_LENGTH - 5], 5, ".csv");
    } else {
        snprintf(&FileName[numSymb], numSymbToPrint, ".csv");
    }
}

static float getO2InMicromole(float O2Float)
{
    const float A0 = 5.80871;
    const float A1 = 3.20291;
    const float A2 = 4.17887;
    const float A3 = 5.10006;
    const float A4 = -9.86643e-2;
    const float A5 = 3.80369;
    const float B0 = -7.01577e-3;
    const float B1 = -7.70028e-3;
    const float B2 = -1.13864e-2;
    const float B3 = -9.51519e-3;
    const float C0 = -2.75915e-7;
    float scaledT;
    float uMolO2;
    double salinDouble = double(Salinity);

    // Calculate O2 in micromole per kilogram
    scaledT  =  298.15 - PHTEMP;
    scaledT /= (273.15 + PHTEMP);
    scaledT  = log(scaledT);
    uMolO2   = (A0 + (A1 * scaledT) + ((A2 * scaledT * scaledT) + (A3 * scaledT * scaledT * scaledT) + (A4 * scaledT * scaledT * scaledT * scaledT) +
                                       (A5 * scaledT * scaledT * scaledT * scaledT * scaledT) + ((Salinity * (B0 + (B1 * scaledT) + (B2 * scaledT * scaledT) +
                                               (B3 * scaledT * scaledT * scaledT))))+(C0 * salinDouble * salinDouble)));
    uMolO2  = exp(uMolO2);
    uMolO2 *= O2Float / 100;

    return uMolO2;
}

static float getDeltaTime(void)
{
    struct tm *TimeNow;
    time_t now;
    float currentTime;
    float deltaTime;

    time(&now);
    TimeNow = localtime(&now);
    // Calculate the current time to seconds
    currentTime = TimeNow->tm_sec + ((TimeNow->tm_min) * 60) + ((TimeNow->tm_hour) * 3600);

    // Calculate the delta time from the start of the experiment or mode
    deltaTime = currentTime - StartTime;

    return deltaTime;
}

static void calculateStartTime(void)
{
    struct tm *TimeNow;
    time_t now;

    time(&now);
    TimeNow = localtime(&now);
    StartTime = (TimeNow->tm_sec) + ((TimeNow->tm_min) * 60) + ((TimeNow->tm_hour) * 3600);
}

#ifdef USE_LCD
void readUserPrograms(void)
{
    FILE *programFile;

    // Read predefined programs.
    programFile = fopen("/" FSNAME "/program.sys", "r");

    if (programFile == NULL) {
        lcdClear();
        lcdWrite(5, 0, JUSTIFICATION_CENTER, "Program File Missing");
    }

    fscanf(programFile, "%d", &numberofprograms);
    DEBUG1("numberofprograms=%d", numberofprograms);

    for (int progamIx = 0; progamIx < numberofprograms; ++progamIx) {
        DEBUG1("progamIx=%d", progamIx);

        for (int paramIx = 0; paramIx < 5; ++paramIx) {
            fscanf(programFile, "%f", &pgm[progamIx][paramIx]);
            DEBUG1("paramIx=%d programParameter=%f", paramIx, pgm[progamIx][paramIx]);
        }

        /* Saving number of seconds. */
        pgm[progamIx][2] *= 60;
    }

    fclose(programFile);
}
#endif

void experimentPrepareDefinedProgram(void)
{
    static char dataname[PATH_TO_FILE_LEN];

    DEBUG1("Salinity = %d", Salinity);

    calculateStartTime();
#ifdef USE_WIFI
    pumpSetRpm(MeasurementFlow);
#else
    pumpSet(MeasurementFlow);
#endif
    // create the filename to write
    getFileName();
    DEBUG1("create name");

    if (MeasurementTypeVal == R) {
        MeasurementLight = 0;
    }

    sprintf(dataname, "%s%s", "/" FSNAME "/Data/", FileName);

    DEBUG1("datename %s", dataname);

    currentFile = fopen(dataname, "w");

    switch (MeasurementTypeVal) {
        case R:
            lightSet(LIGHT_OFF_INTENSITY);
            fprintf(currentFile, "Respiration Time: %d\n", minRESP);
            minEXPER       = minRESP;
            minTOTAL       = minEXPER;
            minMODE        = minEXPER;
            mode           = R;
            isModeSwitched = true;
            SwitchMode     = 0;
            break;
        case P:
            lightSet(MeasurementLight); // turn on light
            fprintf(currentFile, "Photosynthesis Time: %d\n", minPHOTO);
            minEXPER       = minPHOTO;
            minTOTAL       = minEXPER;
            minMODE        = minEXPER;
            mode           = P;
            isModeSwitched = true;
            SwitchMode     = 0;
            break;
        case RP:
            lightSet(LIGHT_OFF_INTENSITY);
            fprintf(currentFile, "Respiration Time: %d\n",    minRESP);
            fprintf(currentFile, "Photosynthesis Time: %d\n", minPHOTO);
            minEXPER       = minRESP + minPHOTO;
            minTOTAL       = minEXPER;
            minMODE        = minRESP;
            mode           = R;
            isModeSwitched = false;
            SwitchMode     = 0;
            lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "3 - Change Mode");
            break;
        case PR:
            lightSet(MeasurementLight);// turn on light
            fprintf(currentFile, "Respiration Time: %d\n",    minRESP);
            fprintf(currentFile, "Photosynthesis Time: %d\n", minPHOTO);
            minEXPER       = minRESP + minPHOTO;
            minTOTAL       = minEXPER;
            minMODE        = minPHOTO;
            mode           = P;
            isModeSwitched = false;
            SwitchMode     = 0;
            lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "3 - Change Mode");

            minEXPER = minRESP + minPHOTO;
            break;

        default:
            ERROR("Unknown experiment type=%d", MeasurementTypeVal);
            return;
    }

    // write file header
    fprintf(currentFile, "Filename:%s\n",     dataname);
    fprintf(currentFile, "Type: %d\n",        MeasurementTypeVal + 1);
    fprintf(currentFile, "Flow Rate: %d\n",   MeasurementFlow);
    fprintf(currentFile, "Light Level: %d\n", MeasurementLight);
    fprintf(currentFile, "Salinity: %d\n",    Salinity);
    fprintf(currentFile, "Total Time: %d\n",  minEXPER);

    fprintf(currentFile, "O2 0 Phase: %2.2f\n",      PhaseCal1);
    fprintf(currentFile, "O2 0 Temp: %2.2f\n",       TempCal1);
    fprintf(currentFile, "O2 100 Phase: %2.2f\n",    PhaseCal2);
    fprintf(currentFile, "O2 100 Temp: %2.2f\n",     TempCal2);
    fprintf(currentFile, "O2 LED Current: %3.0f\n",  LEDCurrent);
    fprintf(currentFile, "O2 Pressure Cal: %3.2f\n", PresCal);
    fprintf(currentFile, "Mode 0 = Light Off, Mode 1 = Light On \n");
    fprintf(currentFile, "Mode, Time(m), Per. Sat. 02, O2 in uMol/kg, ISFET pH, ISFET Temp,, pH Sensor Voltage, ISFET Temp Sensor Voltage, Battery, Phase 02, Amplitude O2\n");

    lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "4 - Stop");
}

bool experimentRunDefinedProgram(void)
{
    static const char* expType[4] = {"R", "P", "R+P", "P+R"};

    float deltaTime;
    float TotalTimeRemaining = 0;
    int   tremmin;
    int   tremsec;
    int   mremmin;
    int   mremsec;
    float presensData[PRESENS_RES_LENGTH];

    float uMolO2 = 0;
    float PHASE;
    int   Amplitude;
    float O2Float;

    DEBUG1("Running experiment MeasurementTypeVal=%d, minRESP=%d, minPHOTO=%d",
           MeasurementTypeVal, minRESP, minPHOTO);

    presensGetData(presensData);

    Amplitude = presensData[0];
    PHASE     = presensData[1];
    O2Float   = presensData[2];

    // get amplitude and phase to store
    lcdWrite(0,  0, JUSTIFICATION_ABSOLUTE, "Run Exp");
    lcdWrite(0, 11, JUSTIFICATION_ABSOLUTE, "Bat=%3.1f", batteryGet());        // Battery Voltage
    lcdWrite(1,  0, JUSTIFICATION_ABSOLUTE, "Type_%-3s", expType[MeasurementTypeVal]);
    lcdWrite(1, 11, JUSTIFICATION_ABSOLUTE, "pH_%3.2f", pHCorrected);

    /* get time to make filename */
    deltaTime = getDeltaTime();
    uMolO2 = getO2InMicromole(O2Float);

    TotalTimeRemaining = minTOTAL * 60 - deltaTime;        // Total time remaining
    mtimeRemaining     = minMODE  * 60 - deltaTime;        // Mode time remaining

    /* Switch mode for R+P or P+R experiemnts type. */
    if (isModeSwitched == false &&
            (SwitchMode == 1 || mtimeRemaining <= 0)) {
        isModeSwitched = true;
        SwitchMode     = 1;
        mode           = (mode == R) ? P : R;
        minTOTAL       = mode == P ? minPHOTO : minRESP;
        minMODE        = minTOTAL;
        /* Reset start time, since we switched to next step. */
        calculateStartTime();

        deltaTime = 0;
        TotalTimeRemaining = minTOTAL * 60 - deltaTime;
        mtimeRemaining     = minMODE  * 60 - deltaTime;

        lightSet(mode == R ? LIGHT_OFF_INTENSITY : MeasurementLight); // turn on light
        // clean line with instruction for mode changing
        lcdClearLine(6);
        INFO("Mode switched: minRESP=%d deltaTime=%f. New mode %s", minRESP, deltaTime, expType[mode]);
        lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "4 - Stop");
    }

    tremmin = TotalTimeRemaining / 60;
    tremsec = TotalTimeRemaining - tremmin * 60;
    mremmin = mtimeRemaining / 60;
    mremsec = mtimeRemaining - mremmin * 60;

    lcdWrite(2,  0, JUSTIFICATION_ABSOLUTE, "Mode_%-3s", expType[mode]);
    lcdWrite(2, 11, JUSTIFICATION_ABSOLUTE, "Temp_%2.1f", (float)PHTEMP);
    lcdWrite(3,  0, JUSTIFICATION_ABSOLUTE, "T Time_%2d", minEXPER);
    lcdWrite(3, 11, JUSTIFICATION_ABSOLUTE, "%%O2_%3.2f", O2Float);
    lcdWrite(4,  0, JUSTIFICATION_ABSOLUTE, "T Rem_%2d:%-.2d", tremmin, tremsec);
    lcdWrite(4, 11, JUSTIFICATION_ABSOLUTE, "F_%3d", MeasurementFlow);
    lcdWrite(5,  0, JUSTIFICATION_ABSOLUTE, "M Rem_%2d:%-.2d", mremmin, mremsec);
    lcdWrite(5, 11, JUSTIFICATION_ABSOLUTE, "L_%3d", (mode == R? 0 : MeasurementLight));

    PRINT_RESULT_TO_FILE(mode, deltaTime, O2Float, uMolO2, PHASE, Amplitude);

    NVIC_EnableIRQ(UART1_IRQn);

    // this is where we switch to shut down
    if (TotalTimeRemaining <= 0) {
        return true;
    }

    return false;
}

void experimentPrepareUserProgram(void)
{
    static char resultsFileName[PATH_TO_FILE_LEN] = {0};

    float  presensData[PRESENS_RES_LENGTH];
    float  O2Float;

    lightSet(LIGHT_OFF_INTENSITY);
    pumpSet(PUMP_OFF_INTENSITY);

    // Create the filename to write
    getFileName();

    sprintf(resultsFileName, "%s%s", "/" FSNAME "/Data/", FileName);

    DEBUG1("datename %s", resultsFileName);

    currentFile = fopen(resultsFileName,"w");
    // Write the header file if this is the first time this program is run
    fprintf(currentFile, "DeviceId:%s\r\n",             instrumentId);
    fprintf(currentFile, "Filename:%s\r\n",             resultsFileName);
    fprintf(currentFile, "pH Slope(m): %1.5f\n",        MSINGLEPT);
    fprintf(currentFile, "pH Eo (b): %1.5f\n",          EOSINGLEPT);
    fprintf(currentFile, "cal. Buffer pH: %1.5f\n",     PHBUFFERF);
    fprintf(currentFile, "pH Vo (buffer): %1.5f\n",     PHVOLTSF);
    fprintf(currentFile, "cal. Buffer temp. %1.5f\n",   PHTEMPF);
    fprintf(currentFile, "Salinity: %d\n",              Salinity);
    fprintf(currentFile, "O2 0 Phase: %1.5f\n",         PhaseCal1);
    fprintf(currentFile, "O2 0 Temp: %1.5f\n",          TempCal1);
    fprintf(currentFile, "O2 100 Phase: %1.5f\n",       PhaseCal2);
    fprintf(currentFile, "O2 100 Temp: %1.5f\n",        TempCal2);
    fprintf(currentFile, "O2 LED Current: %1.5f\n",     LEDCurrent);
    fprintf(currentFile, "O2 Pressure Cal: %1.5f\n",    PresCal);
    fprintf(currentFile, "Total number of steps: %d\n", numberofprograms);
    wait (0.1);

    stepIx = 0;

    calculateStartTime();

    presensGetData(presensData);
    O2Float      = presensData[2];

    O2old   = O2Float;
    phold   = 1; /* In the old version it was 1 in the beggining. Probably this is error. */

    SwitchMode = 0; // instead of isGoToNextProgram

    lightSet(pgm[stepIx][1]);
#ifdef USE_WIFI
    pumpSetRpm(pgm[stepIx][0]);
#else
    pumpSet(pgm[stepIx][0]);
#endif

    buzzerBeep(0.1);
    wait(0.2);
    buzzerBeep(0.1);
    wait(0.2);
    buzzerBeep(0.1);

    fprintf(currentFile, "Step Number: %d\n",      stepIx + 1);
    fprintf(currentFile, "Time Threshold: %3.0f\n",pgm[stepIx][2]);
    fprintf(currentFile, "O2 Threshold: %3.0f\n",  pgm[stepIx][4]);
    fprintf(currentFile, "pH Threshold: %2.3f\n",  pgm[stepIx][3]);
    fprintf(currentFile, "Flow Rate: %f\n",        pgm[stepIx][0]);
    fprintf(currentFile, "Light Level: %f\n",      pgm[stepIx][1]);
    fprintf(currentFile, "Step No., Time(m), Total T(m), O2 PerCent Sat., O2 (umol/kg), ISFET pH, ISFET Temp,, pH raw (V), ISFET Temp raw (V), Battery (V), Phase O2, Amplitude O2\n");
}

bool experimentRunUserProgram(void)
{
    int    TotalTimeRemaining;
    int    AmplitudeInt;
    int    tremmin;
    int    tremsec;

    float  deltaTime;
    float  O2Float;
    float  O2thres;
    double phthres;
    float  PHASEFloat;
    float  presensData[PRESENS_RES_LENGTH];
    float  uMolO2;

    deltaTime = getDeltaTime();

    TotalTimeRemaining = pgm[stepIx][2] - deltaTime;

    DEBUG1("total time remaining %d", TotalTimeRemaining);
    presensGetData(presensData);

    phthres = (pHCorrected - pgm[stepIx][3]) * (phold - pgm[stepIx][3]);
    phold   = pHCorrected;

    AmplitudeInt = presensData[0];
    PHASEFloat   = presensData[1];
    O2Float      = presensData[2];
    DEBUG1("amp %d phase float %2.2f", AmplitudeInt, PHASEFloat);

    O2thres = (O2Float - pgm[stepIx][4]) * (O2old - pgm[stepIx][4]);
    O2old   = O2Float;

    tremmin = TotalTimeRemaining / 60;
    tremsec = TotalTimeRemaining - tremmin * 60;

    if (SwitchMode == 1 || /* switching because user push button. */
            TotalTimeRemaining <= 0 || phthres < 0 || O2thres < 0) { /* legacy switch conditions. */
        lcdClear();

        DEBUG1("Switching step: SwitchMode=%d TotalTimeRemaining=%4.0f phthres=%f O2thres=%f"
               "switch in cases: SwitchMode==1 time<=0 phthres<0, O2thres<0",
               SwitchMode, TotalTimeRemaining, phthres, O2thres);
        stepIx++;
        SwitchMode = 0;

        if (stepIx >= numberofprograms) {
            /* The last step was finished. */
            return true;
        }

        INFO("going to step stepIx=%d", stepIx);

        calculateStartTime();
        deltaTime = 0;
        TotalTimeRemaining = pgm[stepIx][2] - deltaTime;

        tremmin = TotalTimeRemaining / 60;
        tremsec = TotalTimeRemaining - tremmin * 60;

        lightSet(pgm[stepIx][1]);
#ifdef USE_WIFI
        pumpSetRpm(pgm[stepIx][0]);
#else
        pumpSet(pgm[stepIx][0]);
#endif

        buzzerBeep(0.1);
        wait(0.2);
        buzzerBeep(0.1);
        wait(0.2);
        buzzerBeep(0.1);

        fprintf(currentFile, "Step Number: %d\n",      stepIx + 1);
        fprintf(currentFile, "Time Threshold: %3.0f\n",pgm[stepIx][2]);
        fprintf(currentFile, "O2 Threshold: %3.0f\n",  pgm[stepIx][4]);
        fprintf(currentFile, "pH Threshold: %2.3f\n",  pgm[stepIx][3]);
        fprintf(currentFile, "Flow Rate: %f\n",        pgm[stepIx][0]);
        fprintf(currentFile, "Light Level: %f\n",      pgm[stepIx][1]);
        fprintf(currentFile, "Step No., Time(m), Total T(m), 02 PerCent Sat., O2 umol/kg, ISFET pH, ISFET Temp,, pH raw (V), ISFET raw Temp (V), Ext. Battery (V), Phase O2, Amplitude O2\n");
    }

    lcdWrite(0, 0, JUSTIFICATION_ABSOLUTE, "Step# %2d        Bat_%3.1f", stepIx + 1, batteryGet());
    lcdWrite(1, 0, JUSTIFICATION_ABSOLUTE, "T Rem _ %2d:%-.2d    %2.0f:00", tremmin, tremsec, pgm[stepIx][2] / 60);
    lcdWrite(2, 0, JUSTIFICATION_ABSOLUTE, "%%a.s O2_%3.2f  %2.2f", O2Float, pgm[stepIx][4]);
    lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "pH_%3.3f  %2.3f", pHCorrected, pgm[stepIx][3]);
    lcdWrite(4, 0, JUSTIFICATION_ABSOLUTE, "      T_C _  %2.3f", PHTEMP);
    lcdWrite(5, 0, JUSTIFICATION_ABSOLUTE, "      F - %3d  L - %3d", (int)pgm[stepIx][0], (int)pgm[stepIx][1]);
    lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "3 - Next Program Step");
    lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "4 - Stop");

    uMolO2 = getO2InMicromole(O2Float);

    wifiEventsSendDataExpInd(stepIx + 1, TotalTimeRemaining, pHCorrected, O2Float, batteryGet(), PHTEMP);
    PRINT_RESULT_TO_FILE(stepIx+1, deltaTime, O2Float, uMolO2, PHASEFloat, AmplitudeInt);

    wait(.1);

    NVIC_EnableIRQ(UART1_IRQn);

    return false;
}

void finishExperiment(void)
{
    if (MeasurementTypeVal == PGM) {
        buzzerBeep(0.7);
        wait(0.2);
        buzzerBeep(0.7);
        wait(0.2);
        buzzerBeep(0.7);

        /* Clen up filePrefix. It will be new, if user will want this. */
        memset(FilePrefix, 0, FILE_PREFIX_LENGTH);
    }

    /* Experiment finilisation. */
    lightSet(LIGHT_OFF_INTENSITY);
    pumpSet(PUMP_OFF_INTENSITY);

    fclose(currentFile);

}