#include "OxygenSensor.h"
#include <cmath>

static const float DEG2RAD = 3.141592654 / 180.;

/**
 * constructs an OxygenSensor object.
 */
OxygenSensor::OxygenSensor(const float temp0,
               const float temp100,
               const float phase0,
               const float phase100,
               const float pressure,
               const float salinity)
  : m_temp0(temp0), m_temp100(temp100), m_phase0(phase0),
    m_phase100(phase100), m_pressure(pressure), m_salinity(salinity) {
}

/**
 * calculates the percent of oxygen air saturation.
 *
 * @param temp  temperature, in degrees Celsius.
 * @param phase measured phase angle.
 * @return      percentage of air saturated with oxygen.
 */
float
OxygenSensor::calculate(const float temp, const float phase) const {
  const float f1 = 0.808;
  const float delphik = -0.0803;
  const float delksvk = 0.000433;
  const float m = 29.87;

  const float tanphi0_100 = std::tan((m_phase0 + delphik*(m_temp100 - m_temp0))
                * DEG2RAD);
  const float tanphi0_m   = std::tan((m_phase0 + delphik*(   temp   - m_temp0))
                * DEG2RAD);
  const float tanphi100_100 = std::tan(m_phase100 * DEG2RAD);
  const float tanphim_m     = std::tan(  phase    * DEG2RAD);

  const float A = tanphi100_100 / tanphi0_100 / m * 100*100;
  const float B = (tanphi100_100 / tanphi0_100
           + tanphi100_100 / tanphi0_100 / m
           - f1 / m - 1 + f1) * 100;
  const float C = tanphi100_100 / tanphi0_100 - 1;

  const float ksv100 = (-B + std::sqrt(B*B - 4*A*C)) / (2 * A);
  const float ksvm = ksv100 + delksvk*(temp - m_temp100);

  const float a = tanphim_m / tanphi0_m / m * ksvm * ksvm;
  const float b = (tanphim_m / tanphi0_m
           + tanphim_m / tanphi0_m / m 
           - f1 / m - 1 + f1) * ksvm;
  const float c = tanphim_m / tanphi0_m - 1;

  //  const float chlorinity = (m_salinity - 0.03) / 1.805;

  const float airsat = (-b + std::sqrt(b*b - 4*a*c)) / (2 * a);

  return airsat;
}
