#ifndef WIFI_EVENTS_H
#define WIFI_EVENTS_H

#include "cisme.h"

typedef enum {
    ERROR_BATTERY_LOW     = 0x1,
    ERROR_PUMP_RPM_NULL   = 0x2,
    ERROR_MISSED_ISFET    = 0x4,
    ERROR_MISSED_DSPARAMS = 0x8,
} Errors;

#ifdef USE_WIFI
 
#define ADD_START_ERROR(_startErr) startErrors |= _startErr

extern char startErrors;

/**
 * Start handling of Wi-Fi events.
 */
void wifiEventsStart(void);

/**
 * Send experiment data.
 *
 * @param id Experiment identifier
 * @param step Current step index
 * @param timeRemaining Remainig experiment time
 * @param ph PH value
 * @param o2 O2 value
 * @param battery Battery level
 * @param temperature Temperature (C)
 */
void wifiEventsSendDataExpInd(int32_t step, int32_t timeRemaining, double ph, double o2, double battery, double temperature);

/**
 * Send temperature calibration data.
 *
 * @param tempBuffer Temperature value entered by user
 * @param tempVolts Average volts value
 */
void wifiEventsSendTempCalDataInd(float tempBuffer, float tempVolts);

/**
 * Send PH calibration data.
 *
 * @param phBuffer PH value entered by user
 * @param phVoltsCurrent Current volts value
 * @param phVoltsAverage Average volts value
 */
void wifiEventsSendPhCalDataInd(float phBuffer, float phVoltsCurrent, float phVoltsAverage);

/**
 * Send errors.
 *
 * @param errors Errors bitmask (see Errors)
 */
void wifiEventsSendErrorsInd(char errors);

/**
 * Send pump calibration data.
 *
 * @param intensity Pump intensity in percents
 * @param rpm Pump RPM
 */
void wifiEventsSendPumpCalDataInd(int intensity, int rpm);

#else // USE_WIFI

#define ADD_START_ERROR(_startErr)

static inline void wifiEventsStart(void)
{
}

static inline void wifiEventsSendDataExpInd(int32_t step, int32_t timeRemaining, double ph, double o2, double battery, double temperature)
{
}

static inline void wifiEventsSendTempCalDataInd(float tempBuffer, float tempVolts)
{
}

static inline void wifiEventsSendPhCalDataInd(float phBuffer, float phVoltsCurrent, float phVoltsAverage)
{
}

static inline void wifiEventsSendErrorsInd(char errors)
{
}

static inline void wifiEventsSendPumpCalDataInd(int intensity, int rpm)
{
}

#endif // USE_WIFI

#endif // WIFI_EVENTS_H