#include "mbed.h"
#include "pump.h"
#include "light.h"
#include "debug.h"
#include "wifi_events.h"

// According to specification for DM212 pump PWM period is 15 ms. new pump driver period 4-2019
#define PUMP_PWM_PERIOD_US 15000

// Pulse width for DM212 pump can be 1-2 ms,
// where 1 ms - pump is off, 2 ms  - pump is on with maximum speed. new pump driver parameters 4-2019
#define PUMP_PULSE_WIDTH_MIN_US 950
#define PUMP_PULSE_WIDTH_MAX_US 2000

// Zero pulse width is used to disable pulse
#define PUMP_DISABLE_PULSE 0

// Time to wait to be sure that signal sent 3 times.
// Measured during experiments, can be tuned in the future.
#define PUMP_DELAY_MS 500

// File with pump parameters
#define PUMP_FILE "/" FSNAME "/Parameters/pump.sys"

// Measurements number for pump feedback
#define PUMP_FEEDBACK_NUM_MEASUREMENTS 10

// Maximum period for RPM feedback
#define PUMP_FEEDBACK_MAX_PERIOD 0.5

// Pump polynom coefficients
// Default values for ideal pump (linear dependency)
static float a = -50.0;
static float b = 0.02;
static float c = 0.0;

// PWM out to handle pump
static PwmOut pumpPwm(p26);

// Feedback timer
static Timer feedbackTimer;

// Feedback interrupt
static InterruptIn feedbackInterrupt(p18);

// Latest applied pump PercentPump
static float lastPercentPump;


// Latest average feedback period
static float feedbackPeriod;

static void feedbackRise(void)
{
    static float periodSum = 0;
    static int numMeasureaments = 0;

    // Read period
    float period = feedbackTimer.read();
    feedbackTimer.reset();

    if (period > PUMP_FEEDBACK_MAX_PERIOD) {
        periodSum = 0;
        numMeasureaments = 0;
        feedbackPeriod = 0;
        return;
    }

    periodSum += period;
    numMeasureaments++;

    if (numMeasureaments >= PUMP_FEEDBACK_NUM_MEASUREMENTS) {
        feedbackPeriod = periodSum / numMeasureaments;
        periodSum = 0;
        numMeasureaments = 0;
    }
}

void pumpInit(void)
{
    // Configure PWM period
    pumpPwm.period_us(PUMP_PWM_PERIOD_US);

    // Turn pump off
    pumpPwm.pulsewidth_us(PUMP_PULSE_WIDTH_MIN_US);
    
    // Wait to be sure that signal sent 3 times. 
    wait_ms(PUMP_DELAY_MS); 
    
    // Disable pump pulse. 
    pumpPwm.pulsewidth_us(PUMP_DISABLE_PULSE);
    

    // Read parameters
    FILE* paramFile = fopen(PUMP_FILE, "r");
    if (paramFile != NULL) {
        fscanf(paramFile, "%f,%f,%f", &a, &b, &c);
        fclose(paramFile);
    }
    
    INFO("Used pump coefficients: a=%f, b=%0.6f, c=%0.9f", a, b, c);

    // Set callback for feedback
    feedbackInterrupt.rise(&feedbackRise);
    feedbackTimer.start();
}

void pumpSet(float PercentPump)
{
    if (PercentPump > PUMP_MAX_INTENSITY) {
        // Don't support PercentPump more than 100%
        ERROR("Request to set too big pump PercentPump %f", PercentPump);
        return;
    }

//    if (lastPercentPump == PercentPump) {
//        INFO("Discard request to set pump PercentPump, PercentPump %f already applied", PercentPump);
//        return;
//    }

    lastPercentPump = PercentPump;
    

    // As PWM interface for LPC1768 uses the same period for all PWM outs
    // read lignt intensity, turn light off and set it back when pump operation finished.
    unsigned char lightIntensity = lightRead();

    // Configure PWM period and set pulse for pump.
    pumpPwm.period_us(PUMP_PWM_PERIOD_US);

    // Set light off
    lightSet(LIGHT_OFF_INTENSITY);

    int pulseWidth = PUMP_PULSE_WIDTH_MIN_US +
                     (PUMP_PULSE_WIDTH_MAX_US - PUMP_PULSE_WIDTH_MIN_US) * PercentPump / PUMP_MAX_INTENSITY;
    pumpPwm.pulsewidth_us(pulseWidth);

    // Wait to be sure that signal sent 3 times. 
    wait_ms(PUMP_DELAY_MS); 
    
    // Disable pump pulse. 
    pumpPwm.pulsewidth_us(PUMP_DISABLE_PULSE);

    // Return light intensity back.
    lightSet(lightIntensity);
}

void pumpSetRpm(int rpm)
{
    // Calculate PercentPump according to provided RPM
    float PercentPump = a + b * rpm + c * rpm * rpm;

    if (PercentPump < PUMP_MIN_INTENSITY) {
        PercentPump = PUMP_OFF_INTENSITY;
    }
    if (PercentPump > PUMP_MAX_INTENSITY) {
        PercentPump = PUMP_MAX_INTENSITY;
    }

    DEBUG1("Set pump PercentPump: %f", PercentPump);
    pumpSet(PercentPump);
}

bool pumpOn(void)
{
    return (lastPercentPump > PUMP_OFF_INTENSITY);
}

unsigned char pumpCurrentIntensity(void)
{
    return lastPercentPump;
}

unsigned int pumpSpeed(void)
{
    // Calculate pump speed
    unsigned int speed = 0;

    if (pumpCurrentIntensity() == 0) {
        return 0;
    }

    if (feedbackPeriod > 0.0) {
        speed = 60 / feedbackPeriod;

    }
    DEBUG1("Pump speed: %u", speed);

    static bool pumpErrorSent = false;
    if (speed != 0) {
        pumpErrorSent = false;
    } else if (!pumpErrorSent) {
        wifiEventsSendErrorsInd(ERROR_PUMP_RPM_NULL);
        pumpErrorSent = true;
    }

    return speed;
}

void pumpGetParams(float* aValue, float* bValue, float* cValue)
{
    *aValue = a;
    *bValue = b;
    *cValue = c;
}

void pumpSetParams(float aValue, float bValue, float cValue)
{
    a = aValue;
    b = bValue;
    c = cValue;

    FILE* paramFile = fopen(PUMP_FILE, "w");
    fprintf(paramFile, "%f,%0.6f,%0.9f", a, b, c);
    fclose(paramFile);
}