#include "cisme.h"
#include "lcd.h"
#include "lcd_events.h"
#include "keys.h"
#include "debug.h"
#include "light.h"
#include "pump.h"
#include "buzzer.h"
#include "presens.h"
#include "adc.h"
#include "experiments.h"
#include "calibration.h"
#include "battery.h"

#ifdef USE_LCD

typedef enum {
    HOME,
    EXP_PARAMS,
    UTILITIES,
    SYS_PARAMS,
    NUM_STATES,
} LcdState;

typedef void (*EventHandler)(void);
typedef void (*PrepareExperiment)(void);
typedef bool (*RunExperiment)(void);

static LcdState state;

// Event handlers
static void home(void);
static void expParams(void);
static void utilities(void);
static void dateTime(void);
static void light(void);
static void pump(void);
static void sysParams(void);
static void purge(void);
static void about(void);
static void phCal(void);
static void o2Cal(void);
static void rawData(void);
static void salinity(void);
static void expType(void);
static void expTime(void);
static void expPump(void);
static void expLight(void);
static void expRun(void);
static void toggleLight(void);
static void togglePump(void);

static const EventHandler events[NUM_STATES][NUM_KEYS] = {
    //ZERO  ONE         TWO         THREE       FOUR        FIVE        SIX         SEVEN           EIGHT   NINE        ASTERISK    HASH    NONE
    {home,  expParams,  utilities,  home,       home,       home,       home,       toggleLight,    home,   togglePump, home,       home,   home}, // HOME
    {NULL,  expType,    expTime,    expPump,    expLight,   expRun,     home,       NULL,           NULL,   NULL,       NULL,       NULL,   NULL}, // EXP_PARAMS
    {NULL,  dateTime,   light,      pump,       sysParams,  purge,      home,       NULL,           NULL,   NULL,       NULL,       NULL,   NULL}, // UTILITIES
    {NULL,  about,      phCal,      o2Cal,      rawData,    salinity,   utilities,  NULL,           NULL,   NULL,       NULL,       NULL,   NULL}, // SYS_PARAMS
};

static void home(void)
{
    static bool startup = true;
    float presensData[PRESENS_RES_LENGTH];

    if (state != HOME) {
        lcdClear();
    }

    state = HOME;

    presensGetData(presensData);
    float O2Float = presensData[2];

    if (startup) {
        L4 = 1;
        buzzerBeep(0.1);
        buzzerBeep(0.1);
        lcdClear();
        L1 = 0;
        L2 = 0;
        L3 = 0;
        L4 = 0;
        startup = false;
    }

    lcdWrite(0, 0,   JUSTIFICATION_CENTER, " CISME 10.0     Bat _ %3.1f", batteryGet());        // Battery Voltage
    lcdWrite(1, 0,   JUSTIFICATION_CENTER, "-----------------------");
    lcdWrite(2, 0, JUSTIFICATION_ABSOLUTE, "       %%a.s O2 _  %3.0f", O2Float);        // O2
    lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "              pH _   %3.2f", pHCorrected);
    lcdWrite(4, 0, JUSTIFICATION_ABSOLUTE, "             T_C _  %2.1f", PHTEMP);        // Temp);
    lcdWrite(5, 0,   JUSTIFICATION_CENTER, "---%s--%s---", (lightOn() ? "Light-On" : "--------"),
                                                           ( pumpOn() ? "Pump-On" : "-------"));
    lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "1 - Run Experiment");
    lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "2 - Utilities");

    NVIC_EnableIRQ(UART1_IRQn);

    Key key = keyGet();
    events[state][key]();
}

static void expParams(void)
{
    state = EXP_PARAMS;

    lcdClear();
    lcdWrite(0, 0,   JUSTIFICATION_CENTER, "Params Edit");
    lcdWrite(1, 0,   JUSTIFICATION_CENTER, "-----------------------");
    lcdWrite(2, 0, JUSTIFICATION_ABSOLUTE, "1 - Type");
    lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "2 - Duration");
    lcdWrite(4, 0, JUSTIFICATION_ABSOLUTE, "3 - Flow");
    lcdWrite(5, 0, JUSTIFICATION_ABSOLUTE, "4 - Light");
    lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "5 - RUN");
    lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "6 - Home Screen");

    // Write current values
    switch (MeasurementTypeVal) {
        case 9:
            lcdWrite(2, 16, JUSTIFICATION_ABSOLUTE, "-");
            lcdWrite(3, 16, JUSTIFICATION_ABSOLUTE, "-");
            break;
        case 0:
            lcdWrite(2, 16, JUSTIFICATION_ABSOLUTE, "R");
            lcdWrite(3, 16, JUSTIFICATION_ABSOLUTE, "%d", minRESP);
            break;
        case 1:
            lcdWrite(2, 16, JUSTIFICATION_ABSOLUTE, "P");
            lcdWrite(3, 16, JUSTIFICATION_ABSOLUTE, "%d", minPHOTO);
            break;
        case 2:
            lcdWrite(2, 16, JUSTIFICATION_ABSOLUTE, "R+P");
            lcdWrite(3, 16, JUSTIFICATION_ABSOLUTE, "%d %d", minRESP, minPHOTO);
            break;
        case 3:
            lcdWrite(2, 16, JUSTIFICATION_ABSOLUTE, "P+R");
            lcdWrite(3, 16, JUSTIFICATION_ABSOLUTE, "%d %d", minPHOTO, minRESP);
            break;
        case 4:
            lcdWrite(2, 13, JUSTIFICATION_ABSOLUTE, "Program");
            lcdWrite(3, 13, JUSTIFICATION_ABSOLUTE, "Program");
            lcdWrite(4, 13, JUSTIFICATION_ABSOLUTE, "Program");
            lcdWrite(5, 13, JUSTIFICATION_ABSOLUTE, "Program");
            break;
        default:
            break;
    }

    // Write current values for flow and time
    if (MeasurementTypeVal != 4) {
        lcdWrite(4, 16, JUSTIFICATION_ABSOLUTE, "%d", flowintime);
        lcdWrite(5, 16, JUSTIFICATION_ABSOLUTE, "%d", lightintime);
    }
}

static void utilities(void)
{
    state = UTILITIES;

    lcdClear();
    lcdWrite(0, 0,   JUSTIFICATION_CENTER, "Utilities Menu");
    lcdWrite(1, 0,   JUSTIFICATION_CENTER, "-----------------------");
    lcdWrite(2, 0, JUSTIFICATION_ABSOLUTE, "1 - Date & Time");
    lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "2 - Toggle Light");
    lcdWrite(4, 0, JUSTIFICATION_ABSOLUTE, "3 - Toggle Pump");
    lcdWrite(5, 0, JUSTIFICATION_ABSOLUTE, "4 - System Parameters");
    lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "5 - Purge");
    lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "6 - Exit");
}

static void setDateTime(void)
{
    struct tm dateTime = {0};

    lcdClear();
    lcdWrite(0, 1, JUSTIFICATION_ABSOLUTE, "SET DATE AND TIME");
    lcdWrite(2, 0, JUSTIFICATION_ABSOLUTE, "MONTH [01-12]");
    dateTime.tm_mon = lcdGetParam(1, 12, 2, 15) - 1;
    lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "DAY [01-31]");
    dateTime.tm_mday = lcdGetParam(1, 31, 3, 15);
    lcdWrite(4, 0, JUSTIFICATION_ABSOLUTE, "YEAR [20##]");
    dateTime.tm_year = lcdGetParam(0, 99, 4, 15) + 100;
    lcdWrite(5, 0, JUSTIFICATION_ABSOLUTE, "HOUR [00-23]");
    dateTime.tm_hour = lcdGetParam(0, 23, 5, 15);
    lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "MINUTE [00-59]");
    dateTime.tm_min = lcdGetParam(0, 59, 6, 15);

    // Send Date Command to PICODos
    INFO("Date Command = DATE %02d-%02d-%02d %02d:%02d:00",
         (dateTime.tm_mon + 1), dateTime.tm_mday, (dateTime.tm_year - 100),
         dateTime.tm_hour, dateTime.tm_min);
    set_time(mktime(&dateTime));
}

static void dateTime(void)
{
    Key key = KEY_NONE;
    time_t now;
    struct tm* timeNow;

    do {
        time_t seconds = time(NULL);

        /* get time to make filename */
        time(&now);
        timeNow = localtime(&now);

        if (timeNow->tm_year < 100) {
            timeNow->tm_year = 100;
        }

        lcdClear();
        lcdWrite(0, 0, JUSTIFICATION_CENTER, "CISME 10.0");
        lcdWrite(2, 0, JUSTIFICATION_CENTER, "Verify Date and Time");
        lcdWrite(4, 0, JUSTIFICATION_CENTER, "%02d/%02d/20%02d %02d:%02d:%02d",
                (timeNow->tm_mon + 1), timeNow->tm_mday, (timeNow->tm_year - 100),
                timeNow->tm_hour, timeNow->tm_min, timeNow->tm_sec);
        lcdWrite(6, 0, JUSTIFICATION_CENTER, "(*)Change        (#)OK");

        DEBUG1("%s", ctime(&seconds));

        key = keyWait();
        if (key == KEY_ASTERISK) {
            setDateTime();
        }
    } while (key != KEY_HASH);

    utilities();
}

static void currentNew(int value, const char* rangeStr)
{
    lcdWrite(1,  0,   JUSTIFICATION_CENTER, "-----------------------");
    lcdWrite(2,  1,   JUSTIFICATION_CENTER, "(%s)", rangeStr);
    lcdWrite(4,  1, JUSTIFICATION_ABSOLUTE, "Current");
    lcdWrite(4,  9, JUSTIFICATION_ABSOLUTE, "-");
    lcdWrite(4, 11, JUSTIFICATION_ABSOLUTE, "[");
    lcdWrite(4, 12, JUSTIFICATION_ABSOLUTE,  "%03d", value);
    lcdWrite(4, 16, JUSTIFICATION_ABSOLUTE, "]");
    lcdWrite(5,  1, JUSTIFICATION_ABSOLUTE, "New");
    lcdWrite(5, 11, JUSTIFICATION_ABSOLUTE, "[");
    lcdWrite(5, 16, JUSTIFICATION_ABSOLUTE, "]");
    lcdWrite(7,  1,   JUSTIFICATION_CENTER, "* backspace  # save");
}

static int getPump(void)
{
    // Set pump speed
    lcdClear();
    lcdWrite(0, 1, JUSTIFICATION_CENTER, "Measurement Flow");
    currentNew(flowintime, "0, 5-100%");

    // Now allow user to edit value
    flowintime = lcdGetParam(0, 100, 5, 12);
    if (flowintime < 5) {
        flowintime = 0;
    }
    return (flowintime);
}

static int getLight(void)
{
    // Set light level
    lcdClear();
    lcdWrite(0, 1, JUSTIFICATION_CENTER, "Measurement Light");
    currentNew(lightintime, "0, 6-100%");

    // Now allow user to edit value
    lightintime = lcdGetParam(0, 100, 5, 12);
    if (lightintime < 6) {
        lightintime = 0;
    }
    return (lightintime);
}

static void light(void)
{
    MeasurementLight = getLight();

    lightSet(MeasurementLight);
    INFO("Measurement light %d", MeasurementLight);

    utilities();
}

static void pumpFlowSpeed(void)
{
    lcdClear();
    lcdWrite(0, 1, JUSTIFICATION_ABSOLUTE, "Current flow:");
    lcdWrite(1, 1, JUSTIFICATION_ABSOLUTE, "Current speed:");
    lcdWrite(6, 0,   JUSTIFICATION_CENTER, "(*)Change        (#)OK");
}

static void pump(void)
{
    Key key = KEY_NONE;
    pumpFlowSpeed();

    do {
        unsigned char flow = pumpCurrentIntensity();
        lcdWrite(0, 15, JUSTIFICATION_ABSOLUTE, "%5u", flow);

        unsigned int speed = pumpSpeed();
        lcdWrite(1, 15, JUSTIFICATION_ABSOLUTE, "%5u", speed);

        key = keyGet();
        if (key == KEY_ASTERISK) {
            MeasurementFlow = getPump();
            pumpSet(MeasurementFlow);
            INFO("Measurement flow %d", MeasurementFlow);
            pumpFlowSpeed();
        }
        wait(0.5);

    } while (key != KEY_HASH);

    utilities();
}

static void sysParams(void)
{
    state = SYS_PARAMS;

    lcdClear();
    lcdWrite(0, 0,   JUSTIFICATION_CENTER, "Set System Parameters");
    lcdWrite(1, 0,   JUSTIFICATION_CENTER, "-----------------------");
    lcdWrite(2, 0, JUSTIFICATION_ABSOLUTE, "1 - About");
    lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "2 - pH Auto Cal");
    lcdWrite(4, 0, JUSTIFICATION_ABSOLUTE, "3 - O2 Cal ");
    lcdWrite(5, 0, JUSTIFICATION_ABSOLUTE, "4 - Raw Data Streams ");
    lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "5 - Salinity");
    lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "6 - Exit");
}

static void purge(void)
{
    lcdClear();
    lcdWrite(0, 0, JUSTIFICATION_CENTER, "Purge");
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "Cycles the pump between");
    lcdWrite(3, 0, JUSTIFICATION_CENTER, "0%% and 100%%");
    lcdWrite(4, 0, JUSTIFICATION_CENTER, "4 sec 100%%, 2 sec 0%%");
    lcdWrite(7, 0, JUSTIFICATION_CENTER, "* - Exit,  # - Start");

    if (keyWaitInRange(rangeAsteriskHash) == KEY_ASTERISK) {
        utilities();
        return;
    }

    lcdClear();
    lcdWrite(5, 0, JUSTIFICATION_ABSOLUTE, "Press * to exit Purge");

    do {
        lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "Purge Cycle: Pump On");
        pumpSet(PUMP_MAX_INTENSITY);
        wait(4);
        lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "Purge Cycle: Pump Off");
        pumpSet(PUMP_OFF_INTENSITY);
        wait(2);
    }  while (keyGet() != KEY_ASTERISK);

    utilities();
}

static int getExpDuration(int MeasurementTypeVal)
{
    lcdClear();
    lcdWrite(0, 0, JUSTIFICATION_CENTER, "SET DURATION");
    lcdWrite(1, 0, JUSTIFICATION_CENTER, "-----------------------");

    if (MeasurementTypeVal == 0) {
        lcdWrite(2, 0, JUSTIFICATION_CENTER, "TYPE   - R (Total Time)");
    } else if (MeasurementTypeVal == 1) {
        lcdWrite(2, 0, JUSTIFICATION_CENTER, "TYPE   - P (Total Time)");
    } else if (MeasurementTypeVal == 2) {
        lcdWrite(2, 0, JUSTIFICATION_CENTER, "TYPE   - R + P");
    } else if (MeasurementTypeVal == 3) {
        lcdWrite(2, 0, JUSTIFICATION_CENTER, "TYPE   - P + R");
    }

    lcdWrite(7, 1, JUSTIFICATION_CENTER, "* backspace  # Enter");

    if ((MeasurementTypeVal == 0) || (MeasurementTypeVal == 1)) {
        lcdWrite(4, 0, JUSTIFICATION_ABSOLUTE, "MINUTE [00-59]");
        unsigned char minutes = lcdGetParam(0, 59, 4, 19);
        printf("Minute = %u\n", minutes);

        if (MeasurementTypeVal == 0) {
            minRESP = minutes;
        } else {
            minPHOTO = minutes;
        }
    } else if (MeasurementTypeVal == 2) {
        lcdWrite(4, 0, JUSTIFICATION_ABSOLUTE, "RESP MINUTE [00-59]");
        minRESP = lcdGetParam(0, 59, 4, 19);
        printf("Minute = %u\n", minRESP);

        lcdWrite(5, 0, JUSTIFICATION_ABSOLUTE, "PHOTO MINUTE [00-59]");
        minPHOTO = lcdGetParam(0, 59, 5, 19);
        printf("Minute = %u\n", minPHOTO);
    } else if (MeasurementTypeVal == 3) {
        lcdWrite(4, 0, JUSTIFICATION_ABSOLUTE, "PHOTO MINUTE [00-59]");
        minPHOTO = lcdGetParam(0, 59, 4, 19);
        printf("Minute = %u\n", minPHOTO);

        lcdWrite(5, 0, JUSTIFICATION_ABSOLUTE, "RESP MINUTE [00-59]");
        minRESP = lcdGetParam(0, 59, 4, 19);
        printf("Minute = %u\n", minRESP);
    }

    return (minRESP);
}

static int getExpType(void)
{
    lcdClear();
    lcdWrite(0, 0,   JUSTIFICATION_CENTER, "Measurement Type");
    lcdWrite(1, 0,   JUSTIFICATION_CENTER, "-----------------------");
    lcdWrite(2, 0, JUSTIFICATION_ABSOLUTE, "1 - R");
    lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "2 - P");
    lcdWrite(4, 0, JUSTIFICATION_ABSOLUTE, "3 - R + P");
    lcdWrite(5, 0, JUSTIFICATION_ABSOLUTE, "4 - P + R");
    lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "5 - Saved Program");

    Key key = keyWait();
    INFO("Choice = %u", key);

    return (key - 1);
}

static void expType(void)
{
    MeasurementTypeVal = getExpType();
    expParams();
}

static void expTime(void)
{
    getExpDuration(MeasurementTypeVal);
    expParams();
}

static void expPump(void)
{
    MeasurementFlow = getPump();
    expParams();
}

static void expLight(void)
{
    MeasurementLight = getLight();
    expParams();
}

static void expRun(void)
{
    PrepareExperiment preparation;
    RunExperiment     experimentStep;

    lcdClear();

    INFO("Experiment started.");

    if (MeasurementTypeVal == PGM) {

        preparation    = experimentPrepareUserProgram;
        experimentStep = experimentRunUserProgram;
    } else {
        preparation    = experimentPrepareDefinedProgram;
        experimentStep = experimentRunDefinedProgram;
    }

    preparation();

    while (true) {
        if (experimentStep()) {
            INFO("Experiment completed.");
            fprintf(currentFile, "Experiment Completed\n");
            lcdClear();
            break;
        }

        Key choice = keyGet();

        if (choice == KEY_THREE && SwitchMode == 0 &&
                MeasurementTypeVal != R && MeasurementTypeVal != P) {     // Option 3 - change mode in R+P experiment
            lcdClear();
            lcdWrite(0, 0,   JUSTIFICATION_CENTER, "ARE YOU SURE?");
            lcdWrite(2, 0, JUSTIFICATION_ABSOLUTE, "1 - Yes");
            lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "2 - No");

            if (keyWaitInRange(rangeOneTwo) == KEY_ONE) {

                lcdClear();             // if yes want to change mode
                SwitchMode = 1;
                INFO("Experiment switched by user.");
                fprintf(currentFile,"User Advanced Step\n");
                continue;
            }

            lcdClear();
            lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "3 - Change Mode");
            lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "4 - Stop");

        } else if (choice == KEY_FOUR) { // Option 4 - Stop the experiment
            lcdClear();
            lcdWrite(0, 0,   JUSTIFICATION_CENTER, "ARE YOU SURE?");
            lcdWrite(2, 0, JUSTIFICATION_ABSOLUTE, "1 - Yes");
            lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "2 - No");

            if (keyWaitInRange(rangeOneTwo) == KEY_ONE) {

                INFO("User Ended Experiment");
                fprintf(currentFile,"User Ended Experiment\n");
                lcdClear();
                break;

            }

            lcdClear();
            lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "4 - Stop");

            if ((MeasurementTypeVal == PR || MeasurementTypeVal == RP) &&
                    SwitchMode == 0) {
                lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "3 - Change Mode");
            }
        }
    }

    finishExperiment();

    lcdWrite(2, 0, JUSTIFICATION_CENTER, FileName);
    lcdWrite(4, 0, JUSTIFICATION_CENTER, "Press # to continue");
    Key key = KEY_NONE;
    while (key != KEY_HASH) {
        buzzerBeep(2);
        key = keyWait();
    }
    lcdClear();

    home();
}

static void about(void)
{
    lcdClear();
    lcdWrite(0, 0, JUSTIFICATION_CENTER, "CISME 10.0 February 2016");
    lcdWrite(1, 0, JUSTIFICATION_CENTER, "Coral InSituMEtabolism");
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "Built by Mera for UNCW");
    lcdWrite(3, 0, JUSTIFICATION_CENTER, "Brian Murphy");
    lcdWrite(4, 0, JUSTIFICATION_CENTER, "Robert Whitehead");
    lcdWrite(5, 0, JUSTIFICATION_CENTER, "Charles Mazel");
    lcdWrite(6, 0, JUSTIFICATION_CENTER, "Alina Szmant");

    wait(6);
    sysParams();
}

static void phCal(void)
{
    Key key = KEY_NONE;
    float pH1Float, PHValue1;
    float PHTEMPK, PHTEMPa;
    float mSINGLEPT, eoSINGLEPT;

    lcdClear();
    lcdWrite(1, 0,   JUSTIFICATION_CENTER, "Calibrate ISFET");
    lcdWrite(2, 0,   JUSTIFICATION_CENTER, "Temperature");
    lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "Bvalue - %3.6f", B);
    lcdWrite(7, 0,   JUSTIFICATION_CENTER, "Change? 1-Yes, 2-No");
    wait(.1);

    if (keyWaitInRange(rangeOneTwo) == KEY_ONE) {              // if yes want to change mode
        //ph temp cal
        lcdClear();
        lcdWrite(1, 0, JUSTIFICATION_CENTER, "Calibrate ISFET");
        lcdWrite(2, 0, JUSTIFICATION_CENTER, "Temperature");
        lcdWrite(3, 0, JUSTIFICATION_CENTER, "With Pump");
        lcdWrite(7, 0, JUSTIFICATION_CENTER, "7-On ,9-Off");

        key = keyWaitInRange(rangeSevenOrNine);
        if (key == KEY_SEVEN) {              // if yes want to change mode
            pumpSet(50);
        } else { // KEY_NINE
            pumpSet(PUMP_OFF_INTENSITY);
        }

        lcdClear();
        lcdWrite(0, 0,   JUSTIFICATION_CENTER, "Temp Calibration");
        lcdWrite(1, 0,   JUSTIFICATION_CENTER, "Single Point");
        lcdWrite(2, 0, JUSTIFICATION_ABSOLUTE, "ISFET Temp");
        lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "Temp(C) - XX.XXX");
        lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "Place in Temp");
        lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "Press # to start");
        float TempC = (float)lcdGetParam(0, 99999, 4, 15) / 1000;

        float bValue = calibrationTemperature(TempC);
        calibrationTemperatureSave(bValue);

        wait(1.0);
    }

    lcdClear();
    lcdWrite(1, 0, JUSTIFICATION_ABSOLUTE, "pH Buffer - %3.3f", PHBUFFERF);
    lcdWrite(2, 0, JUSTIFICATION_ABSOLUTE, "pH Temp (C) - %3.5f", PHTEMPF);
    lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "pH Temp (K) - %f", PHTEMPKF);
    lcdWrite(4, 0, JUSTIFICATION_ABSOLUTE, "pH Volts - %f", PHVOLTSF);
    lcdWrite(5, 0, JUSTIFICATION_ABSOLUTE, "Slope - %2.6f", MSINGLEPT);
    lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "Eo - %2.6f", EOSINGLEPT);
    lcdWrite(7, 0,   JUSTIFICATION_CENTER, "Change? 1-Yes,2-No");

    if (keyWaitInRange(rangeOneTwo) == KEY_TWO) {              // if yes want to change mode
        sysParams();
        return;
    }

    lcdClear();
    lcdWrite(1, 0, JUSTIFICATION_CENTER, "Calibrate ISFET");
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "pH");
    lcdWrite(3, 0, JUSTIFICATION_CENTER, "With Pump");
    lcdWrite(7, 0, JUSTIFICATION_CENTER, "7-On ,9-Off");
    wait(0.2);

    if (keyWaitInRange(rangeSevenOrNine) == KEY_SEVEN) {              // if yes want to change mode
        pumpSet(50);
    } else { // Nine
        pumpSet(PUMP_OFF_INTENSITY);
    }

    lcdClear();
    lcdWrite(0, 0,   JUSTIFICATION_CENTER, "pH Auto Calibration");
    lcdWrite(1, 0,   JUSTIFICATION_CENTER, "Single Point");
    lcdWrite(2, 0, JUSTIFICATION_ABSOLUTE, "pH Buffer Value");
    lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "Place Buffer");
    lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "Press # to start");
    lcdWrite(3, 0, JUSTIFICATION_ABSOLUTE, "pH - XX.XXX");
    pH1Float = (float)lcdGetParam(0, 99999, 4, 15) / 1000;

    calibrationPh(pH1Float, &PHTEMPa, &mSINGLEPT, &eoSINGLEPT, &PHValue1);
    PHTEMPK = PHTEMPa + 273.15;
    wait(0.5);

    lcdClear();
    lcdWrite(1, 0, JUSTIFICATION_CENTER, "pH Buffer - %2.3f", pH1Float);
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "pH Temp (C) - %2.3f", PHTEMP);
    lcdWrite(3, 0, JUSTIFICATION_CENTER, "pH Temp (K) - %2.3f", PHTEMPK);
    lcdWrite(4, 0, JUSTIFICATION_CENTER, "pH Volts - %2.6f", PHValue1);
    lcdWrite(5, 0, JUSTIFICATION_CENTER, "Slope - %2.6f", mSINGLEPT);
    lcdWrite(6, 0, JUSTIFICATION_CENTER, "Eo - %2.6f", eoSINGLEPT);

    INFO("PHBUFFERF %2.6f pH1Float %2.6f", PHBUFFERF, pH1Float);
    PHBUFFERF = (float)pH1Float;
    PHTEMPF = (float)PHTEMP;

    INFO("PHTEMPF %2.6f",PHTEMPF);

    PHTEMPKF = (float)PHTEMPK;
    PHVOLTSF = (float)PHValue1;
    MSINGLEPT = (float)mSINGLEPT;
    EOSINGLEPT = (float)eoSINGLEPT;

    calibrationPhSave();
}

static void o2Cal(void)
{
    lcdClear();
    lcdWrite(0, 0, JUSTIFICATION_CENTER, "Pressure Cal - %1.2f", PresCal);
    lcdWrite(1, 0, JUSTIFICATION_CENTER, "0%% Phase - %1.2f", PhaseCal1);
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "0%% Temp - %1.2f", TempCal1);
    lcdWrite(3, 0, JUSTIFICATION_CENTER, "100%% Phase - %1.2f", PhaseCal2);
    lcdWrite(4, 0, JUSTIFICATION_CENTER, "100%% Temp - %1.2f", TempCal2);

    lcdWrite(5, 0,   JUSTIFICATION_CENTER, "Change values?");
    lcdWrite(6, 0, JUSTIFICATION_ABSOLUTE, "1 - Yes");
    lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "2 - No");

    if (keyWaitInRange(rangeOneTwo) == KEY_TWO) {              // if yes want to change mode
        sysParams();
        return;
    }

    lcdClear();
    lcdWrite(0, 0, JUSTIFICATION_CENTER, "Oxygen Calibration");

    // Pressure Calibration
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "Pres Cal [500-2000]");
    PresCal = lcdGetParam(500, 2000, 4, 17);
    INFO("Pres Cal: %f", PresCal);

    // 0% Phase Calibration
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "0%% P Ang [0-90.00]");
    PhaseCal1 = (float)lcdGetParam(0, 9000, 4, 17) / 100;
    INFO("PhaseCal1: %f", PhaseCal1);

    // 0% Temp Calibration
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "0%% Temp [0-50.00]     ");
    TempCal1 = (float)lcdGetParam(0, 5000, 4, 17) / 100;

    INFO("TempCal1: %f", TempCal1);

    // 100% Phase Calibration
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "100%% P. Ang. [0-90.00]");
    PhaseCal2 = (float)lcdGetParam(0, 9000, 4, 17) / 100;
    INFO("PhaseCal2: %f", PhaseCal2);

    // 100% Phase Calibration
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "100%% Temp [0-50.00]");
    TempCal2 = (float)lcdGetParam(0, 5000, 4, 17) / 100;
    INFO("TempCal2: %f", TempCal2);

    // LED Current
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "LED Current [0-255]");
    LEDCurrent = lcdGetParam(0, 255, 4, 17);
    INFO("LEDCurrent: %f", LEDCurrent);

    calibrationO2();

    sysParams();
}

static void rawData(void)
{
    float AmplitudeFloat, PhaseFloat;
    float presensData[PRESENS_RES_LENGTH];

    float O2Float = 0;

    lcdClear();

    do {
        presensGetData(presensData);
        AmplitudeFloat  = presensData[0];
        PhaseFloat      = presensData[1];
        O2Float         = presensData[2];

        lcdWrite(0, 0,   JUSTIFICATION_CENTER, "Raw Data     Bat _  %3.2f", batteryGet()); // Battery Voltage
        lcdWrite(1, 0,   JUSTIFICATION_CENTER, "%%a.s O2_ %3.0f pH_ %3.2f", O2Float, pHCorrected); // O2
        lcdWrite(2, 0,   JUSTIFICATION_CENTER, "o2 Amp _ %d", (int)AmplitudeFloat); // Amp
        lcdWrite(3, 0,   JUSTIFICATION_CENTER, "o2 Phase _ %3.2f", PhaseFloat);
        lcdWrite(4, 0,   JUSTIFICATION_CENTER, "pH adc V _ %3.3f", pH);
        lcdWrite(5, 0,   JUSTIFICATION_CENTER, "T adc V _ %3.3f", pHT);
        lcdWrite(6, 0,   JUSTIFICATION_CENTER, "T C _ %1.1f", PHTEMP);
        lcdWrite(7, 0, JUSTIFICATION_ABSOLUTE, "# - Back");

        NVIC_EnableIRQ(UART1_IRQn);
    } while (keyGet() != KEY_HASH);

    sysParams();
}

static void salinity(void)
{
    lcdClear();
    lcdWrite(0,  0,   JUSTIFICATION_CENTER, " Enter Salinity:");
    lcdWrite(1,  0,   JUSTIFICATION_CENTER, "-----------------------");
    lcdWrite(2,  1,   JUSTIFICATION_CENTER, "(31 - 39)");
    lcdWrite(4,  1, JUSTIFICATION_ABSOLUTE, "Current");
    lcdWrite(4,  9, JUSTIFICATION_ABSOLUTE, "-");
    lcdWrite(4, 11, JUSTIFICATION_ABSOLUTE, "[");
    lcdWrite(4, 12, JUSTIFICATION_ABSOLUTE,  "%02d", Salinity);
    lcdWrite(4, 15, JUSTIFICATION_ABSOLUTE, "]");
    lcdWrite(5,  1, JUSTIFICATION_ABSOLUTE, "New");
    lcdWrite(5, 11, JUSTIFICATION_ABSOLUTE, "[");
    lcdWrite(5, 15, JUSTIFICATION_ABSOLUTE, "]");
    lcdWrite(7,  1,   JUSTIFICATION_CENTER, "* backspace  # save");

    Salinity = lcdGetParam(31, 39, 5, 12);

    sysParams();
}

static void toggleLight(void)
{
    if (lightOn() == false) {
        lightSet(25);
    } else {
        lightSet(LIGHT_OFF_INTENSITY);
    }

    home();
}

static void togglePump(void)
{
    if (pumpOn() == false) {
        pumpSet(60);
    } else {
        pumpSet(PUMP_OFF_INTENSITY);
    }

    home();
}

void lcdEventSplash(void)
{
    lcdClear();
    lcdWrite(1, 0, JUSTIFICATION_CENTER, "--------------------");
    lcdWrite(2, 0, JUSTIFICATION_CENTER, "CISME 10.0");
    lcdWrite(3, 0, JUSTIFICATION_CENTER, "February 2016");
    lcdWrite(4, 0, JUSTIFICATION_CENTER, "--------------------");
    lcdWrite(6, 0, JUSTIFICATION_CENTER, "Mera Software Services");
    lcdWrite(7, 0, JUSTIFICATION_CENTER, "978-689-0003");
}

void lcdEventsStart(void)
{
    home();

    while (true) {
        Key key = keyWait();
        if (events[state][key] != NULL) {
            events[state][key]();
        }
    }
}

#endif // USE_LCD
