#include "cisme.h"
#include "lcd.h"
#include "debug.h"
#include "keys.h"

#ifdef USE_LCD

#define LCD_CMD_LEN 40
#define LCD_MAX_ROW 7
#define LCD_VALUE_MAX_RANK 5
#define NUMBER_TO_ASCII(val) (val + 0x30)

#define SOH    0x01
#define ETX    0x03

static Serial lcdComm(p13, p14);

static void convertToHexStr(unsigned char value, char* dst)
{
    static const char map[] = "0123456789abcdef";
    dst[0] = map[(value / 16)];
    dst[1] = map[(value % 16)];
}

static void lcdCallback(void)
{
    keyEvent(lcdComm.getc());
}

static unsigned char lcdValueRank(unsigned long value)
{
    unsigned char rank = 1;
    for (unsigned long rankIx = 10; (value / rankIx) != 0; rankIx *= 10, rank++);
    return rank;
}

static void lcdValueDiscardHighestRank(unsigned long* value)
{
    unsigned char rank = lcdValueRank(*value);
    unsigned long divider = 1;
    
    for (; rank != 1; rank--) {
        divider *= 10;
    }
    
    *value %= divider;
} 

void lcdInit(void)
{
    // Set the baud rate for the LCD
    lcdComm.baud(9600);
    // Set the serial parameters for the LCD
    lcdComm.format(8, Serial::None, 1);
    lcdComm.attach(lcdCallback, Serial::RxIrq);
}

void lcdClear(void)
{
    lcdComm.attach(NULL);
    lcdComm.printf("%cCFF007F%c", SOH, ETX);
    lcdComm.attach(lcdCallback, Serial::RxIrq);
}

void lcdClearLine(unsigned char row)
{
    static const unsigned char line[] = {1, 2, 4, 8, 16, 32, 64, 128};
    static char cmd[2]; //needed only for hex representation of line
    
    if (row > LCD_MAX_ROW) {
        ERROR("Too big row number. Received: %u, maximum is %u", row, LCD_MAX_ROW);
        return;
    }
    
    convertToHexStr(line[row], cmd);
    
    lcdComm.attach(NULL);
    lcdComm.printf("%cC%s007F%c", SOH, cmd, ETX);
    lcdComm.attach(lcdCallback, Serial::RxIrq);
}

#define LCD_CMD_COUNT_MGMT_BYTES 11

void lcdWrite(unsigned char row, unsigned char col, Justification just, const char* format, ...)
{
    static char cmd[LCD_CMD_LEN];
    static const unsigned char line[] = {1, 2, 4, 8, 16, 32, 64, 128};
    static char txt[LCD_CMD_LEN - LCD_CMD_COUNT_MGMT_BYTES] = {0};
    va_list params;
    int charactersWritten;
    
    if (row > LCD_MAX_ROW) {
        ERROR("Too big row number. Received: %u, maximum is %u", row, LCD_MAX_ROW);
        return;
    }
    
    va_start(params, format);
 
    charactersWritten = vsnprintf(txt, LCD_CMD_LEN - LCD_CMD_COUNT_MGMT_BYTES, format, params);
 
    va_end(params);
    
    /*
     * The return value is more than possible count of characters.
     * (excluding terminating NULL byte)
     * According to the man page, this means that output was truncated.
    */
    if (charactersWritten >= LCD_CMD_LEN - LCD_CMD_COUNT_MGMT_BYTES)
    {
        ERROR("Too big text length. textLength is %u max length is %u", charactersWritten, LCD_CMD_LEN - LCD_CMD_COUNT_MGMT_BYTES - 1);
        return;
    }

    unsigned char cmdIx = 0;
    cmd[cmdIx++] = 1;                           // SOH Byte
    cmd[cmdIx++] = 'P';                         // Print Command
    convertToHexStr(line[row], cmd + cmdIx);    // Row
    cmdIx += 2;
    convertToHexStr(col * 6, cmd + cmdIx);      // Column
    cmdIx += 2;
    cmd[cmdIx++] = '0';                         // Font - Single Line 5x7 Char
    cmd[cmdIx++] = '1';                         // Type - Black Chars on White Background
    cmd[cmdIx++] = NUMBER_TO_ASCII(just);       // Justification

    // Add message text to command string
    size_t txtLen = strlen(txt);
    memcpy(cmd + cmdIx, txt, txtLen);
    cmdIx += txtLen;    
        
    cmd[cmdIx++] = ' ';
    // Add ETX Character to End of Command String
    cmd[cmdIx++] = 3;

    lcdComm.attach(NULL);
    lcdComm.printf("%s\n", cmd);
    lcdComm.attach(lcdCallback, Serial::RxIrq);
}

unsigned long lcdGetParam(unsigned long min, unsigned long max, unsigned char row, unsigned char col)
{
    Key key = KEY_NONE;
    unsigned long value = 0;
    unsigned char maxRank = lcdValueRank(max);
    char valueStr[LCD_VALUE_MAX_RANK + 1];

    while (key != KEY_HASH || value > max || value < min) {
        unsigned char strOffset = maxRank - lcdValueRank(value);
        memset(valueStr, '0', LCD_VALUE_MAX_RANK);        
        sprintf(valueStr + strOffset, "%u", value);
        lcdWrite(row, col, JUSTIFICATION_ABSOLUTE, valueStr);

        key = keyWait();
        if (key == KEY_ASTERISK) { // backspace
            value /= 10;
        }
        else if (key <= KEY_NINE) { // numeric values
            if (lcdValueRank(value) == maxRank) {
                lcdValueDiscardHighestRank(&value);
            }
            value = value * 10 + key;
        }
    }

    return value;
}

#endif // USE_LCD
