/*
Copyright (c) 2016, Dust Networks. All rights reserved.

SimplePublish example application for the mbed OS.

\license See attached DN_LICENSE.txt.
*/

#include "mbed.h"
#include "millis.h"
#include "dn_qsl_api.h"     // Only really need this include from QSL
#include "dn_debug.h"       // Included to borrow debug macros
#include "dn_endianness.h"  // Included to borrow array copying
#include "dn_time.h"        // Included to borrow sleep function

#define NETID           0       // Factory default value used if zero (1229)
#define JOINKEY         NULL    // Factory default value used if NULL (44 55 53 54 4E 45 54 57 4F 52 4B 53 52 4F 43 4B)
#define BANDWIDTH_MS    5000    // Not changed if zero (default base bandwidth given by manager is 9 s)
#define SRC_PORT        60000   // Default port used if zero (0xf0b8)
#define DEST_PORT       0       // Default port used if zero (0xf0b8)
#define DATA_PERIOD_MS  5000    // Should be longer than (or equal to) bandwidth

// We can use debug macros from dn_debug, as stdio defaults to this serial
Serial serialDebug(SERIAL_TX, SERIAL_RX); 
// LED2 is the green LED on the NUCLEO-L053R8; might change for other boards
DigitalOut myled(LED2);

static uint16_t randomWalk(void);
static void parsePayload(const uint8_t *payload, uint8_t size);
 
int main()
{
    uint8_t payload[3];
    uint8_t inboxBuf[DN_DEFAULT_PAYLOAD_SIZE_LIMIT];
    uint8_t bytesRead;
    uint8_t i;
    
    // Configure SysTick for timing
    millisStart();
    
    // Set PC debug serial baudrate
    serialDebug.baud(115200);
    
    log_info("Initializing...");
    dn_qsl_init();
    
    // Flash LED to indicate start-up complete
    for (i = 0; i < 10; i++)
    {
        myled = !myled;
        dn_sleep_ms(50);
    }

    while(TRUE) {
        if (dn_qsl_isConnected())
        {
            uint16_t val = randomWalk();
            static uint8_t count = 0;
            myled = 0; // Turn off LED during send/read
            
            dn_write_uint16_t(payload, val);
            payload[2] = count;
            
            if (dn_qsl_send(payload, sizeof (payload), DEST_PORT))
            {
                log_info("Sent message nr %u: %u", count, val);
                count++;
            } else
            {
                log_info("Send failed");
            }

            do
            {
                bytesRead = dn_qsl_read(inboxBuf);
                parsePayload(inboxBuf, bytesRead);
            } while (bytesRead > 0);
            
            myled = 1; // Turn on LED
            dn_sleep_ms(DATA_PERIOD_MS);
        } else
        {
            log_info("Connecting...");
            myled = 0; // Not connected; turn off LED
            if (dn_qsl_connect(NETID, JOINKEY, SRC_PORT, BANDWIDTH_MS))
            {
                myled = 1; // Connected; turn on LED
                log_info("Connected to network");
            } else
            {
                log_info("Failed to connect");
            }
        }

    }
}

static uint16_t randomWalk(void)
{
    static bool first = TRUE;
    static uint16_t lastValue = 0x7fff; // Start in middle of uint16 range
    const int powerLevel = 9001;
    
    // Seed random number generator on first call
    if (first)
    {
        first = FALSE;
        srand(dn_time_ms());
    }
    
    // Random walk within +/- powerLevel
    lastValue += rand() / (RAND_MAX / (2*powerLevel) + 1) - powerLevel;
    return lastValue;
}

static void parsePayload(const uint8_t *payload, uint8_t size)
{
    uint8_t i;
    char msg[size + 1];
    
    if (size == 0)
    {
        // Nothing to parse
        return;
    }
    
    // Parse bytes individually as well as together as a string
    log_info("Received downstream payload of %u bytes:", size);
    for (i = 0; i < size; i++)
    {
        msg[i] = payload[i];
        log_info("\tByte# %03u: %#.2x (%u)", i, payload[i], payload[i]);
    }
    msg[size] = '\0';
    log_info("\tMessage: %s", msg);
} 