#ifndef __TTSDISPLAY_H__
#define __TTSDISPLAY_H__

#ifndef uchar
#define uchar char 
#endif

// definitions for TM1636
#define ADDR_AUTO  0x40
#define ADDR_FIXED 0x44
#define STARTADDR  0xc0

// definitions for the clock point of the digit tube
#define POINT_ON   1
#define POINT_OFF  0

// definitions for brightness
#define  BRIGHT_DARKEST 0
#define  BRIGHT_TYPICAL 2
#define  BRIGHTEST      7

//Special characters index of tube table
#define INDEX_NEGATIVE_SIGH    16
#define INDEX_BLANK            17


class TTSDisplay{


private:

    uchar Cmd_SetData;
    uchar Cmd_SetAddr;
    uchar Cmd_Dispdisplay;
    uchar _PointFlag;                                   //_PointFlag=1:the clock point on
    uchar _brightness;
    uchar Clkpin;
    uchar Datapin;
    uchar dtaDisplay[4];
    
private:

    void writeByte(uchar wr_data);                      //write 8bit data to tm1637
    void start(void);                                   //send start bits
    void stop(void);                                    //send stop bits
    void set(uchar = BRIGHT_TYPICAL, uchar = 0x40, uchar = 0xc0);       //To take effect the next time it displays. 
    uchar coding(uchar DispData);


//********************************************************************************************************** 
//*************************** USER INTERFACE ***************************************************************
//********************************************************************************************************** 
public:

    TTSDisplay();

    void display(uchar loca, uchar dta);                // display a single num(0-9)
    void num(int dta);                                  // display a num (0-9999)    
    void time(uchar hour, uchar min);                   // display time, such as: 11:26
    void clear();                                       // clear display
    
    void pointOn();                                     // display :
    void pointOff();                                    // undisplay :

};

#endif

//#include <Arduino.h>
#include<mbed.h>
//#include <Streaming.h>

//#include "TTSDisplay.h"

//0~9,A,b,C,d,E,F,"-"," "
const uchar TubeTab[] = 
{
    0x3f,0x06,0x5b,0x4f,
    0x66,0x6d,0x7d,0x07,
    0x7f,0x6f,0x77,0x7c,
    0x39,0x5e,0x79,0x71,
    0x40,0x00
};
    
//#define PINCLK      7                   // pin of clk 
//#define PINDTA      8                   // pin of data

TTSDisplay::TTSDisplay()
{
    DigitalOut Clkpin(D7);
    DigitalOut Datapin(D8);
    /*Clkpin  = PINCLK;
    Datapin = PINDTA;
    
    pinMode(Clkpin,OUTPUT);
    pinMode(Datapin,OUTPUT);*/
    
    for(int i=0; i<4; i++)
    {
        dtaDisplay[i] = 0x00;
    }
    
    set();
    //clear();
}

/*********************************************************************************************************
 * Function Name: display
 * Description:  display a num in certain location
 * Parameters: loca - location, 3-2-1-0
 *             num - number to display
 * Return: None
*********************************************************************************************************/
void TTSDisplay::display(uchar loca, uchar dta)
{

    //if(loca > 3 || loca < 0) return;

    if(loca > 3) return;
    dtaDisplay[loca] = dta;

    loca = 3-loca;

    uchar segData = coding(dta);
    
    start();                            //start signal sent to TM1637 from MCU
    writeByte(ADDR_FIXED);
    stop();
    
    start();
    writeByte(loca|0xc0);
    writeByte(segData);
    stop();
    
    start();
    writeByte(Cmd_Dispdisplay);
    stop();

}

/*********************************************************************************************************
 * Function Name: num
 * Description:  display a num  that 0 - 9999
 * Parameters: num - number to display
 * Return: None
*********************************************************************************************************/
void TTSDisplay::num(int dta)
{
    if(dta < 0 || dta > 9999) return;           // bad data
    
    //clear();
    
    pointOff();
    if(dta < 10)
    {
        display(0, dta);
        display(1, 0x7f);
        display(2, 0x7f);
        display(3, 0x7f);
    }
    else if(dta < 100)
    {
        display(1, dta/10);
        display(0, dta%10);
        display(2, 0x7f);
        display(3, 0x7f);
    }
    else if(dta < 1000)
    {
        display(2, dta/100);
        display(1, (dta/10)%10);
        display(0, dta%10);
        display(3, 0x7f);
    }
    else
    {
        display(3, dta/1000);
        display(2, (dta/100)%10);
        display(1, (dta/10)%10);
        display(0, dta%10);
    }
}

/*********************************************************************************************************
 * Function Name: time
 * Description:  display time
 * Parameters: hour - hour
 *             min - minutes
 * Return: None
*********************************************************************************************************/
void TTSDisplay::time(uchar hour, uchar min)
{
   // if(hour > 24 || hour < 0) return;           // bad data
    if(hour > 24) return;           // bad data
   // if(min > 60 || min < 0) return;             // bad data
    if(min > 60) return;             // bad data
    
    
    display(3, hour/10);
    display(2, hour%10);
    display(1, min/10);
    display(0, min%10);
}
    
/*********************************************************************************************************
 * Function Name: clear
 * Description:  clear all 
 * Parameters: None
 * Return: None
*********************************************************************************************************/
void TTSDisplay::clear()
{
    display(0x00,0x7f);
    display(0x01,0x7f);
    display(0x02,0x7f);
    display(0x03,0x7f);
}

/*********************************************************************************************************
 * Function Name: writeByte
 * Description:  write a byte to tm1636
 * Parameters: wr_data: data to write
 * Return: None
*********************************************************************************************************/
void TTSDisplay::writeByte(uchar wr_data)
{
    uchar i,count1;
    for(i=0;i<8;i++)                                        // sent 8bit data
    {
        Clkpin=0;
        //digitalWrite(Clkpin,LOW);
        //if(wr_data & 0x01)digitalWrite(Datapin,HIGH);       // LSB first
        if(wr_data & 0x01)Datapin=1;       // LSB first
        //else digitalWrite(Datapin,LOW);
        else Datapin=0;
        wr_data >>= 1;
        Clkpin=1;
        //digitalWrite(Clkpin,HIGH);
    }
    
    Clkpin=0;
    Datapin=1;
    Clkpin=1;
    DigitalIn Datapin(D8);
    
    /*digitalWrite(Clkpin,LOW);                               // wait for the ACK
    digitalWrite(Datapin,HIGH);
    digitalWrite(Clkpin,HIGH);
    pinMode(Datapin,INPUT);*/
    
    //while(digitalRead(Datapin))
    while(Datapin)
    {
        count1 += 1;
        if(200 == count1)
        {
            //pinMode(Datapin,OUTPUT);
            DigitalOut Datapin(D8);
            //digitalWrite(Datapin,LOW);
            Datapin=0;
            count1 =0;
        }
        //pinMode(Datapin,INPUT);
        DigitalIn Datapin(D8);
    }
    //pinMode(Datapin,OUTPUT);
    //DigitalOut Datapin(D8);

}

/*********************************************************************************************************
 * Function Name: start
 * Description:  send start signal to TTSDisplay
 * Parameters: None
 * Return: None
*********************************************************************************************************/
void TTSDisplay::start(void)
{
    /*digitalWrite(Clkpin,HIGH);                              //send start signal to TM1637
    digitalWrite(Datapin,HIGH);
    digitalWrite(Datapin,LOW);
    digitalWrite(Clkpin,LOW);*/
    Clkpin=1;
    Datapin=1;
    Datapin=0;
    Clkpin=0;
}

/*********************************************************************************************************
 * Function Name: stop
 * Description:  send end signal
 * Parameters: None
 * Return: None
*********************************************************************************************************/
void TTSDisplay::stop(void)
{/*
    digitalWrite(Clkpin,LOW);
    digitalWrite(Datapin,LOW);
    digitalWrite(Clkpin,HIGH);
    digitalWrite(Datapin,HIGH);*/
    Clkpin=0;
    Datapin=0;
    Clkpin=1;
    Datapin=1;
}

/*********************************************************************************************************
 * Function Name: set
 * Description:  init
 * Parameters: brightness - brightness
 *             SetDta - data
 *             SetAddr - address
 * Return: None
*********************************************************************************************************/
void TTSDisplay::set(uchar brightness,uchar SetData,uchar SetAddr)
{
    _brightness = brightness;
    Cmd_SetData = SetData;
    Cmd_SetAddr = SetAddr;
    Cmd_Dispdisplay = 0x88 + brightness;
}

/*********************************************************************************************************
 * Function Name: pointOn
 * Description:  display :
 * Parameters: None
 * Return: None
*********************************************************************************************************/
void TTSDisplay::pointOn()
{
    _PointFlag = 1;
    
    for(int i=0; i<4; i++)
    {
        display(i, dtaDisplay[i]);
    }
}

/*********************************************************************************************************
 * Function Name: pointOff
 * Description:  no :
 * Parameters: None
 * Return: None
*********************************************************************************************************/
void TTSDisplay::pointOff()
{
    _PointFlag = 0;
    
    for(int i=0; i<4; i++)
    {
        display(i, dtaDisplay[i]);
    }
}

/*********************************************************************************************************
 * Function Name: coding
 * Description:  coding
 * Parameters: None
 * Return: None
*********************************************************************************************************/
uchar TTSDisplay::coding(uchar DispData)
{

    uchar PointData = _PointFlag ? 0x80 : 0x00;
    DispData = (0x7f == DispData) ? PointData : (TubeTab[DispData]+PointData);
    return DispData;
}
    
   
/*********************************************************************************************************
  END FILE
*********************************************************************************************************/


//#include <Wire.h>
//#include <TTSDisplay.h>


//TTSDisplay disp;                                    // instantiate an object


/*void setup()
{
    // nothing to setup
}*/

int main()
{
    TTSDisplay disp;  
    while(1)
        {
            disp.time(11,26);                              // display time, 11:26
            disp.pointOn();                                 // display :
            wait_ms(1000);                                    // wait 1s
        }
}

/*********************************************************************************************************
  END FILE
*********************************************************************************************************/