#ifndef LED_H
#define LED_H

/**
 *  The Led class contains the functions to control the robots LEDs
 */
class Led
{
public:

    /** 
     *  Sends the reset command to the reserved i2c address for the TLC59116 LED driver
     *
     *  As the led driver is powered independently from the MBED, and the MBED can be reset externally, this reset operation should
     *  be run as part of the initialisation routine to switch off the LEDs and restore them to the initial state.
     *
     * @return A zero if acknowledge recceived from LED driver chip, otherwise a non-zero
     */
    int reset_led_driver(void);
    
    /** 
     *  Turns on the oscillator and enables all the LED outputs on the TLC59116 LED driver
     *
     *
     * @return A zero if acknowledge recceived from LED driver chip, otherwise a non-zero
     */
    int init_led_driver(void);
    
    /** 
     *  Restores the currently stored states for the LEDs; used at the end of animation routines
     *  to return the LED states to as they were.  Also used when set_green_led_stored_state and
     *  set_red_led_stored_state as called.
     */
    void restore_states(void);
    
    /**
     *  Set the brightness of all 8 LEDs to the same red, green brightness values
     *  
     *  @param red_brightness - The PWM duty cycle from 0 to 255 for the red LEDs
     *  @param green_brightness - The PWM duty cycle from 0 to 255 for the green LEDs
     */
    void set_all (char red_brightness, char green_brightness);
    
    /**
     *  Set the brightness of all 8 LEDs to the same red, green brightness values
     *  
     *  @param * red_brightness - Pointer to an 8-char array for the PWM duty cycle from 0 to 255 for the red LEDs
     *  @param * green_brightness - Pointer to an 8-char array for the PWM duty cycle from 0 to 255 for the green LEDs
     */
    void set_leds (char red1, char red2, char red3, char red4, char red5, char red6, char red7, char red8, char grn1, char grn2, char grn3, char grn4, char grn5, char grn6, char grn7, char grn8);
    void set_leds (char * red_brightness, char * green_brightness);

    /**
     *  Set the brightness of all case LEDs to zero
     */
    void all_off(void);
    
    /**
     *  Set an individual green led to the brightness specified; this function updates the
     *  stored state for an individual LED and sets an I2C command to update the LED level
     *
     *  Consider using set_X_led_stored_state and restore_states when many changes are to
     *  be made at once to reduce number of i2c messages.
     * 
     *  @param led - The index of the LED from 0 to 7 (corresponds the 1 to 8 on the PCB)
     *  @param brightness - The PWM duty cycle from 0 to 255
     */
    void set_green_led (char led, char brightness);
    
    /**
     *  Set an individual red led to the brightness specified; this function updates the
     *  stored state for an individual LED and sets an I2C command to update the LED level
     *
     *  Consider using set_X_led_stored_state and restore_states when many changes are to
     *  be made at once to reduce number of i2c messages.
     * 
     *  @param led - The index of the LED from 0 to 7 (corresponds the 1 to 8 on the PCB)
     *  @param brightness - The PWM duty cycle from 0 to 255
     */
    void set_red_led (char led, char brightness);

    /**
     *  Sets the stored brightness value for an individual green led without sending 
     *  actual i2c command; use restore_states to flush changes.
     * 
     *  @param led - The index of the LED from 0 to 7 (corresponds the 1 to 8 on the PCB)
     *  @param brightness - The PWM duty cycle from 0 to 255
     */
    void set_green_led_stored_state (char led, char brightness);
    
    /**
     *  Sets the stored brightness value for an individual red led without sending 
     *  actual i2c command; use restore_states to flush changes.
     * 
     *  @param led - The index of the LED from 0 to 7 (corresponds the 1 to 8 on the PCB)
     *  @param brightness - The PWM duty cycle from 0 to 255
     */
    void set_red_led_stored_state (char led, char brightness);
  
    
    /**
     *  Stops the ticker-based routine
     */    
    void stop_animation(void);
    
    /**
     *  Starts a ticker based LED animation
     *
     */
    void start_animation(char mode, char speed);
    
    private:
    
    void animation_cycle();
    
    // The following do not stored`
    void _set_green_led (char led, char brightness);
    void _set_red_led (char led, char brightness);
    void _set_all (char red_brightness, char green_brightness);
    void _set_leds (char * red_brightness, char * green_brightness);
    void _set_leds (char red1, char red2, char red3, char red4, char red5, char red6, char red7, char red8, char grn1, char grn2, char grn3, char grn4, char grn5, char grn6, char grn7, char grn8);


};

#endif
