#ifndef MOTORS_H
#define MOTORS_H

/**
 *  The Motors class contains the functions to control the robots motors
 */
class Motors
{
public:
    /**
    * Setup the PWM based H-Bridge drivers for the motors
    */
    void init(void);

    /**
    * Returns the current [adjusted] value for the target left motor speed
    *
    * @returns The speed value from -1.0 to 1.0
    */
    float get_left_motor_speed(void);

    /**
    * Returns the current [adjusted] value for the target right motor speed
    *
    * @returns The speed value from -1.0 to 1.0
    */
    float get_right_motor_speed(void);

    /**
    * Calls set_left_motor_speed and set_right_motor_speed with equal values
    *
    * @params speed - The forward speed value from -1.0 to 1.0
    */
    void forwards(float speed);

    /**
    * Calls set_left_motor_speed and set_right_motor_speed with equal inverse values
    *
    * @params speed - The forward speed value from -1.0 (max forwards) to 1.0 (max reverse)
    */
    void backwards(float speed);

    /**
    * Calls set_left_motor_speed and set_right_motor_speed with sign-opposite values
    *
    * @params speed - The forward speed value from -1.0 (max counterclockwise) to 1.0 (max clockwise)
    */
    void turn(float speed);

    /**
     * Put the H-Bridge drivers in sleep (low-power) mode
     */
    void sleep(void);

    /**
     * Wake the H-Bridge drivers from sleep mode
     */
    void wake_up(void);

    /**
     * Set the left H-Bridge driver to high-Z (coast mode) for the motor
     */
    void coast_left(void);

    /**
     * Set the left H-Bridge driver to active brake
     */
    void brake_left(void);

    /**
     * Set the speed and direction of the left-motor H-Bridge driver
     *
     * @param speed - A value between -1.0 [max reverse], 0 [coast] and 1.0 [max forward]
     */
    void set_left_motor_speed(float speed);

    /**
     * Set the right H-Bridge driver to high-Z (coast mode) for the motor
     */
    void coast_right(void);

    /**
     * Set the right H-Bridge driver to active brake
     */
    void brake_right(void);

    /**
     * Set both H-Bridge drivers to high-Z (coast mode) for the motor
     */
    void coast(void);

    /**
    * Set both H-Bridge drivers to active brake
    */
    void brake(void);

    /**
     * Set the speed and direction of the right-motor H-Bridge driver
     *
     * @param speed - A value between -1.0 [max reverse], 0 [coast] and 1.0 [max forward]
     */
    void set_right_motor_speed(float speed);

    /**
     * Get the approximate instantaneous current draw of the left-motor H-Bridge driver
     *
     * @returns The current value in A
     */
    float get_current_left(void);

    /**
     * Get the approximate instantaneous current draw of the right-motor H-Bridge driver
     *
     * @returns The current value in A
     */
    float get_current_right(void);

    /**
     * Calculates an adjusted speed value to avoid stalls at low-speeds if USE_STALL_OFFSET is set to 1 [in robot.h]
     * @params speed_in - The speed value from 0.0 to 1.0
     * @returns The adjusted speed value from 0.0 to 1.0, increased by STALL_OFFSET then scaled to fit
     */
    float get_adjusted_speed(float speed_in);
    
    /**
     * Sets the PWM period in microseconds (minimum value set to 10 for 100kHZ PWM)
     */
    void set_pwm_period(int period_in);


};

#endif